<?php 

use Illuminate\Support\Str;

if (! function_exists('highlight')) {

    function highlight($text, $query) {

        if(!$query) return e($text);

        $escapedQuery = preg_quote($query, '/');

        $safe = e($text);

        return preg_replace_callback("/($escapedQuery)/iu", function($m) {

            return '<mark>'.$m[0].'</mark>';

        }, $safe);

    }

}

if (! function_exists('trans_field')) {
    /**
     * Return a localized string from a record or array.
     *
     * @param object|array $record
     * @param string $baseField   e.g. 'titre', 'resume'
     * @param string|null $fallbackLocale
     * @return string|null
     */
    function trans_field($record, string $baseField, ?string $fallbackLocale = null)
    {
        $locale = app()->getLocale() ?? config('app.locale', 'ar');
        $fallback = $fallbackLocale ?? config('app.fallback_locale', 'ar');

        // helper: read attribute / key and decode JSON if necessary
        $getRaw = function ($obj, $col) {
            $val = null;
            if (is_array($obj)) {
                $val = $obj[$col] ?? null;
            } elseif (is_object($obj)) {
                // Eloquent models: use getAttribute if available
                if (method_exists($obj, 'getAttribute')) {
                    $val = $obj->getAttribute($col);
                } else {
                    $val = $obj->$col ?? null;
                }
            }

            // if string that is JSON, decode
            if (is_string($val)) {
                $decoded = json_decode($val, true);
                if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
                    return $decoded;
                }
            }

            return $val;
        };

        // Normalize result to scalar string or null
        $normalize = function ($v) use ($locale, $fallback) {
            if ($v === null) return null;

            // If associative array like ['en' => '...', 'fr' => '...']
            if (is_array($v)) {
                // prefer locale key
                if (isset($v[$locale]) && trim((string)$v[$locale]) !== '') return (string)$v[$locale];
                if (isset($v[$fallback]) && trim((string)$v[$fallback]) !== '') return (string)$v[$fallback];
                // otherwise take first non-empty item
                foreach ($v as $item) {
                    if (is_scalar($item) && trim((string)$item) !== '') return (string)$item;
                }
                return null;
            }

            // Other scalars (int/float/string/boolean) -> cast to string
            if (is_scalar($v)) return (string) $v;

            // objects -> try __toString
            if (is_object($v) && method_exists($v, '__toString')) return (string) $v;

            return null;
        };

        // Helper to produce column name variations for a given language code
        $colVariantsFor = function(string $baseField, string $lang) {
            $variants = [];
            $langLower = strtolower($lang);
            $langUpper = strtoupper($lang);

            // try lowercase suffix first
            $variants[] = $baseField . '_' . $langLower;

            // try uppercase suffix (some schemas use uppercase)
            if ($langUpper !== $langLower) {
                $variants[] = $baseField . '_' . $langUpper;
            }

            // special-case: some legacy columns (like guide_G) hold Arabic content
            // the user noted that for Arabic we should also try *_G
            if ($langLower === 'ar') {
                $variants[] = $baseField . '_G';
            }

            // de-duplicate while preserving order
            $variants = array_values(array_unique($variants));

            return $variants;
        };

        // 1) try locale-specific columns with common variants (e.g. titre_ar, titre_AR, titre_G)
        foreach ($colVariantsFor($baseField, $locale) as $col) {
            $val = $normalize($getRaw($record, $col));
            if ($val) return $val;
        }

        // 2) try direct baseField (maybe JSON or array)
        $val = $normalize($getRaw($record, $baseField));
        if ($val) return $val;

        // 3) try fallback locale variants
        foreach ($colVariantsFor($baseField, $fallback) as $colFb) {
            $val = $normalize($getRaw($record, $colFb));
            if ($val) return $val;
        }

        // 4) try other supported locales columns (and their variants)
        $supported = config('app.supported_locales', ['ar','fr','en']);
        foreach ($supported as $l) {
            if ($l === $locale || $l === $fallback) continue;
            foreach ($colVariantsFor($baseField, $l) as $c) {
                $val = $normalize($getRaw($record, $c));
                if ($val) return $val;
            }
        }

        // 5) nothing found
        return null;
    }
}
