<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Candidate extends Model
{
    use HasFactory;

    protected $table = 'candidates'; // make sure you've renamed the table

    protected $fillable = [
        'nom',
        'prenom',
        'cin',
        'date_naissance',
        'tel',
        'email',
        'adresse',
        'gouvernorat',
        'code_postale',
        'etat',         // application state: en_cours, refuse, acceptation_primaire, acceptation_definitif
        // 'concours_id' // if you preferred one-to-many alternative
    ];

    protected $casts = [
        'date_naissance' => 'date',
    ];

    /**
     * The dynamic field values for this candidate.
     * field_values table should use candidate_id (BIGINT UNSIGNED).
     */
    public function fieldValues()
    {
        return $this->hasMany(FieldValue::class, 'candidate_id', 'id');
    }

    /**
     * Uploaded files for this candidate.
     * field_files table should use candidate_id (BIGINT UNSIGNED).
     */
    public function files()
    {
        return $this->hasMany(FieldFile::class, 'candidate_id', 'id');
    }

    /**
     * If you keep the many-to-many pivot between concours and candidates:
     */
    public function concours()
    {
        return $this->belongsToMany(Concours::class, 'concours_candidate', 'candidate_id', 'concours_id')
                    ->withTimestamps();
    }

    /**
     * Short full name
     */
    public function getFullNameAttribute()
    {
        return trim("{$this->nom} {$this->prenom}");
    }

    /**
     * Human label for etat (Arabic)
     */
    public function getEtatLabelAttribute()
    {
        return match($this->etat) {
            'refuse' => 'مرفوض',
            'en_cours' => 'قيد الدراسة',
            'acceptation_primaire' => 'قبول أولي',
            'acceptation_definitif' => 'قبول نهائي',
            default => $this->etat,
        };
    }

    /**
     * Scope helpers
     */
    public function scopeAccepted($query)
    {
        return $query->whereIn('etat', ['acceptation_primaire', 'acceptation_definitif']);
    }

    public function scopePending($query)
    {
        return $query->where('etat', 'en_cours');
    }
    
}
