<?php

namespace App\Http\Controllers;

use App\Models\Utilisateur;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Hash;
use App\Mail\CompteActiveMail;
use App\Mail\CompteSetMdpMail;
use App\Models\Role;

class UtilisateurController extends Controller
{
    public function index(Request $request)
{
    // On prend uniquement les utilisateurs qui ont AU MOINS un rôle
    $query = Utilisateur::with('roles')
        ->whereHas('roles');

    if ($request->filled('cin')) {
        $query->where('cin', 'LIKE', '%' . $request->cin . '%');
    }

    if ($request->filled('role_id')) {
        $query->whereHas('roles', function($q) use ($request) {
            $q->where('id_roles', $request->role_id);
        });
    }

    $utilisateurs = $query->orderBy('id', 'desc')->paginate(10)->withQueryString();

    $roles = Role::all();

    return view('BackOffice.role-permission.user.index', compact('utilisateurs', 'roles'));
}


    public function create()
    {
        $roles = Role::all(); 
        return view('BackOffice.role-permission.user.create', compact('roles'));

    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'nom_prenom' => 'required|string|max:100',
            'cin' => 'nullable|string|size:8|unique:utilisateur,cin',
            'email' => 'required|email|max:100|unique:utilisateur,email',
            'mdp' => 'nullable|string|min:6',
            'tel' => 'required|string|max:12',
            'adresse' => 'required|string|max:100',
            'image' => 'nullable|image|max:2048',
            'role_id' => 'required|exists:roles,id_roles',
        ], [
            'nom_prenom.required' => ' الرجاء إدخال الاسم واللقب',
            'cin.size' => ' رقم بطاقة التعريف يجب أن يحتوي على 8 أرقام',
            'cin.unique' => ' رقم بطاقة التعريف مستعمل من قبل',
            'email.required' => ' الرجاء إدخال البريد الإلكتروني',
            'email.email' => ' الرجاء إدخال بريد إلكتروني صحيح',
            'email.unique' => ' هذا البريد الإلكتروني مستخدم بالفعل',
            'mdp.required' => ' الرجاء إدخال كلمة المرور',
            'mdp.min' => 'كلمة المرور يجب أن تحتوي على الأقل 6 حروف',
            'tel.required' => ' الرجاء إدخال رقم الهاتف',
            'adresse.required' => ' الرجاء إدخال العنوان',
            'role_id.required' => ' الرجاء اختيار نوع المستخدم',
            'role_id.exists' => 'نوع المستخدم المختار غير صالح',
        ]);

        //$data['mdp'] = Hash::make($data['mdp']);
        if (!empty($data['mdp'])) {
            $data['mdp'] = Hash::make($data['mdp']);
        } else {
            unset($data['mdp']); 
        }

        if ($request->hasFile('image')) {
            $file = $request->file('image');
            $filename = $file->getClientOriginalName();
            $file->storeAs('public/utilisateurs', $filename);
            $data['image'] = $filename;
        }
        $utilisateur = Utilisateur::create($data);
        $utilisateur->roles()->attach($request->role_id);

        return redirect()->route('utilisateurs.index')->with('ok', 'تم إنشاء المستخدم بنجاح.');
    }

    public function show(Utilisateur $utilisateur)
    {
        $roles = Role::all();
        return view('BackOffice.role-permission.user.show', compact('utilisateur', 'roles'));
    }

    public function edit(Utilisateur $utilisateur)
    {
        $roles = Role::all();
        return view('BackOffice.role-permission.user.edit', compact('utilisateur', 'roles'));
    }

    public function update(Request $request, Utilisateur $utilisateur)
    {
        $data = $request->validate([
            'nom_prenom' => 'required|string|max:100',
            'cin' => 'nullable|string|size:8|unique:utilisateur,cin,' . $utilisateur->id,
            'email' => 'required|email|max:100|unique:utilisateur,email,' . $utilisateur->id,
            'mdp' => 'nullable|string|min:6', 
            'tel' => 'required|string|max:12',
            'adresse' => 'required|string|max:100',
            'image' => 'nullable|image|max:2048',
            'role_id' => 'required|exists:roles,id_roles',
        ], [
            'nom_prenom.required' => ' الرجاء إدخال الاسم واللقب',
            'cin.required' => ' الرجاء إدخال رقم بطاقة التعريف الوطنية',
            'cin.size' => ' رقم بطاقة التعريف يجب أن يحتوي على 8 أرقام',
            'cin.unique' => ' رقم بطاقة التعريف مستعمل من قبل',
            'email.required' => ' الرجاء إدخال البريد الإلكتروني',
            'email.email' => ' الرجاء إدخال بريد إلكتروني صحيح',
            'email.unique' => ' هذا البريد الإلكتروني مستخدم بالفعل',
            'mdp.min' => 'كلمة المرور يجب أن تحتوي على الأقل 6 حروف',
            'tel.required' => ' الرجاء إدخال رقم الهاتف',
            'adresse.required' => ' الرجاء إدخال العنوان',
            'role_id.required' => ' الرجاء اختيار نوع المستخدم',
            'role_id.exists' => 'نوع المستخدم المختار غير صالح',
        ]);

        if (!empty($data['mdp'])) {
            $data['mdp'] = Hash::make($data['mdp']);
        } else {
            unset($data['mdp']);
        }

        if ($request->hasFile('image')) {
            $file = $request->file('image');
            $filename = time() . '_' . $file->getClientOriginalName();
            $file->storeAs('public/utilisateurs', $filename);
            $data['image'] = $filename;
        }

        $utilisateur->update($data);

        $utilisateur->roles()->sync([$request->role_id]);

        return redirect()->route('utilisateurs.index')->with('ok', 'تم تعديل المستخدم بنجاح.');
    }

    public function destroy(Utilisateur $utilisateur)
    {
        $utilisateur->delete();
        return redirect()->route('utilisateurs.index')->with('ok', 'تم حذف المستخدم بنجاح.');
    }

    /*public function toggleActivation(Request $request, Utilisateur $utilisateur)
    {
        $etaitInactif = !$utilisateur->est_active;
        $utilisateur->est_active = !$utilisateur->est_active;
        $utilisateur->save();

        if ($etaitInactif) {
            Mail::to($utilisateur->email)->send(new CompteActiveMail($utilisateur));
        }

        return back()->with('ok', $utilisateur->est_active ? 'تم تفعيل الحساب.' : 'تم تعطيل الحساب.');
    }*/

    public function toggleActivation(Request $request, Utilisateur $utilisateur)
    {
        $etaitInactif = !$utilisateur->est_active;
        $utilisateur->est_active = !$utilisateur->est_active;
        $utilisateur->save();

        if ($etaitInactif) {
            $roleIds = $utilisateur->roles->pluck('id_roles')->toArray();
            if (in_array(2, $roleIds) || in_array(3, $roleIds)) {
                Mail::to($utilisateur->email)->send(new CompteActiveMail($utilisateur));
            } else {
                Mail::to($utilisateur->email)->send(new CompteSetMdpMail($utilisateur));
            }
        }

        return back()->with('ok', $utilisateur->est_active ? 'تم تفعيل الحساب.' : 'تم تعطيل الحساب.');
    }


    public function profil(Utilisateur $utilisateur)
    {
        return view('BackOffice.role-permission.user.profile', compact('utilisateur'));
    }
        
    public function editProfil()
    {
        $utilisateur = auth()->user();
        return view('BackOffice.role-permission.user.editProfile', compact('utilisateur'));
    }

    public function updateProfil(Request $request, Utilisateur $utilisateur)
    {
        $utilisateur = auth()->user();
        $data = $request->validate([
            'nom_prenom' => 'required|string|max:100',
            'cin' => 'nullable|string|size:8|unique:utilisateur,cin,'.$utilisateur->id,
            'email' => 'required|email|max:100|unique:utilisateur,email,'.$utilisateur->id,
            'tel' => 'nullable|string|max:12',
            'adresse' => 'nullable|string|max:100',
            'image' => 'nullable|image|max:2048', 
        ]);

        if ($request->hasFile('image')) {
            $file = $request->file('image');
            $filename = time().'_'.$file->getClientOriginalName();
            $file->storeAs('public/utilisateurs', $filename);
            $data['image'] = $filename;
        }
        $utilisateur->update($data);
        return redirect()->route('utilisateurs.editprofil')->with('ok', 'تم تحديث الملف الشخصي بنجاح.');    
    }
}