<?php

namespace App\Http\Controllers\Investissement;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Models\TypeFormulaire;
use App\Models\Identite_personnel;
use App\Models\NatureProjet; 
use App\Models\SecteurFinancement;
use App\Models\Utilisateur;
use App\Models\Investissement;
use App\Models\ProjetDivertissementJeuneEnfant;
use App\Models\DirecteurEtablissement;
use App\Mail\JeuneEnfantStatutMail;
use Illuminate\Support\Facades\Mail;
use Carbon\Carbon;
use App\Notifications\JeuneEnfantStatutNotification;

class JeuneEnfantController extends Controller
{
    public function index(Request $request)
    {
        $userId = Auth::id(); 

        $projets = ProjetDivertissementJeuneEnfant::whereHas('investissements', function ($query) use ($userId) {
                $query->where('user_id', $userId);
            })
            ->with(['directeur', 'investissements' => function($q) use ($userId) {
                $q->where('user_id', $userId);
            }])
            ->orderByDesc('created_at')
            ->paginate(10);

        return view('investissement.gestion_jeune_enfant.index', compact('projets'));
    }



    public function create()
    {
        $formulaire = TypeFormulaire::find(3);
        $typesIdentite = Identite_personnel::all();
        $naturesProjet = NatureProjet::all();
        $secteurs = SecteurFinancement::all();

        $user = Auth::user();
        return view('investissement.gestion_jeune_enfant.create', compact(
            'formulaire','typesIdentite','naturesProjet','secteurs','user'
        ));
    }


    
    public function store(Request $request)
    {
        DB::beginTransaction();
        try {
            $request->validate([
                               
                // Directeur
                'nom_prenom_d'     => 'required|string|max:255',
                'num_cin'          => 'required|string|max:20',
                'date_delivrance'  => 'required|date',
                'lieu_delivrance'  => 'required|string|max:255',
                'niveau_etude'     => 'nullable|string|max:255',
                'dernier_diplome_obtenu' => 'nullable|string|max:255',

                // Investissement
                'investissements.*.secteur_invest_id' => 'required|integer|exists:tp_secteur_financement,id_secteur',
                'investissements.*.montant_inv'           => 'required|numeric',
                'investissements.*.source'            => 'nullable|string|max:255',
                'investissements.*.remarques'         => 'nullable|string',

                // Projet
                'nature_projet_id'          => 'required|integer|exists:tp_nature_projet,id_nature_p',
                'nbr_diplome_universitaire' => 'nullable|integer',
                'nbr_autre'                 => 'nullable|integer',
            ]);
            
            $directeur = DirecteurEtablissement::create([
                'nom_prenom_d'           => $request->nom_prenom_d,
                'num_cin'                => $request->num_cin,
                'date_delivrance'        => $request->date_delivrance,
                'lieu_delivrance'        => $request->lieu_delivrance,
                'niveau_etude'           => $request->niveau_etude,
                'dernier_diplome_obtenu' => $request->dernier_diplome_obtenu,
            ]);
            $projet = ProjetDivertissementJeuneEnfant::create([
                'nature_projet_id'          => $request->nature_projet_id,
                'id_directeur'              => $directeur->id_directeur,
                'nbr_diplome_universitaire' => $request->nbr_diplome_universitaire,
                'nbr_autre'                 => $request->nbr_autre,
                'espace_spectacle'          => $request->espace_spectacle ?? 0,
                'caracteristique_spec'      => $request->caracteristique_spec,
                'espace_sportif'            => $request->espace_sportif ?? 0,
                'caracteristique_sportif'   => $request->caracteristique_sportif,
                'espace_activite_service'   => $request->espace_activite_service ?? 0,
                'caracteristique_activite_serv' => $request->caracteristique_activite_serv,
                'espace_activite_culturelle'=> $request->espace_activite_culturelle ?? 0,
                'caracteristique_activite_cult' => $request->caracteristique_activite_cult,
                'remarques'                 => $request->remarques,
                'id_statut'                 => 2,
                'id_formulaire'             => 3,
                'commentaire_motif'         => $request->commentaire_motif ?? 0,
            ]);
            if ($request->has('investissements')) {
                foreach ($request->investissements as $inv) {
                    Investissement::create([
                        'user_id'           => Auth::id(), 
                        'secteur_invest_id' => $inv['secteur_invest_id'],
                        'montant_inv'       => $inv['montant_inv'],
                        'source'            => $inv['source'] ?? null,
                        'remarques'         => $inv['remarques'] ?? null,
                        'id_projet'         => $projet->id_projet,
                    ]);
                }
            }
            DB::commit();
            return redirect()->route('jeune_enfant.index')
                            ->with('success', 'Projet enregistré avec succès ✅');
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->withErrors(['error' => 'Erreur lors de l\'enregistrement : '.$e->getMessage()])
                        ->withInput();
        }
    }

    public function toggleStatut(Request $request, $id)
    {
        $projet = ProjetDivertissementJeuneEnfant::with('investissements.utilisateur')->findOrFail($id);
        $projet->id_statut = $request->id_statut;

        $user = $projet->investissements->first()?->utilisateur;

        if ($user && !empty($user->email)) { 
            if ($request->id_statut == 3 && $request->commentaire_motif) {
                $projet->commentaire_motif = $request->commentaire_motif;

                Mail::to($user->email)->send(
                    new JeuneEnfantStatutMail('refuse', $user, null, $request->commentaire_motif)
                );

                $user->notify(new JeuneEnfantStatutNotification('refuse', $request->commentaire_motif));

            } elseif ($request->id_statut == 1) {
                Mail::to($user->email)->send(
                    new JeuneEnfantStatutMail('accepte', $user)
                );

                $user->notify(new JeuneEnfantStatutNotification('accepte'));
            }
        } else {
            Log::warning("Aucun utilisateur avec email trouvé pour le projet ID: {$id}");
        }

        $projet->save();

        return redirect()->back()->with('success', 'تم تغيير حالة الطلب بنجاح');
    }

    public function show($id)
    {
        $projet = ProjetDivertissementJeuneEnfant::with([
            'directeur',
            'investissements',
            'investissements.utilisateur'
        ])->findOrFail($id);

        $typesIdentite = Identite_personnel::all();
        $naturesProjet = NatureProjet::all();
        $secteurs = SecteurFinancement::all();
        $formulaire = TypeFormulaire::find($projet->id_formulaire);

        return view('investissement.gestion_jeune_enfant.show', compact(
            'projet', 'typesIdentite', 'naturesProjet', 'secteurs', 'formulaire'
        ));
    }

    public function edit($id)
    {
        $projet = ProjetDivertissementJeuneEnfant::with(['directeur', 'investissements', 'investissements.utilisateur'])->findOrFail($id);
        $typesIdentite = Identite_personnel::all();
        $naturesProjet = NatureProjet::all();
        $secteurs = SecteurFinancement::all();
        $formulaire = TypeFormulaire::find($projet->id_formulaire);

        return view('investissement.gestion_jeune_enfant.edit', compact(
            'projet', 'typesIdentite', 'naturesProjet', 'secteurs', 'formulaire'
        ));
    }

    public function update(Request $request, $id)
    {
        DB::beginTransaction();
        try {
            $projet = ProjetDivertissementJeuneEnfant::findOrFail($id);
            $request->validate([
                // Utilisateur
                'nom_prenom'       => 'required|string|max:255',
                'id_type_identite' => 'required|integer|exists:tp_type_identite_pers,id_type',
                'cin'              => 'nullable|string|max:20',
                'date_delivrance_cin' => 'nullable|date',
                'lieu_delivrance_cin' => 'nullable|string|max:255',
                'passport'         => 'nullable|string|max:50',
                'num_fiscal'       => 'nullable|string|max:50',
                'tel'              => 'nullable|string|max:20',
                'adresse'          => 'nullable|string|max:255',
                'lieu_naissance'   => 'nullable|string|max:100',
                'date_naissance'   => 'nullable|date',
                'niveau_etude'     => 'nullable|string|max:100',
                'dernier_diplome'  => 'nullable|string|max:100',
                'experience'       => 'nullable|string|max:100',

                // Directeur
                'nom_prenom_d'     => 'required|string|max:255',
                'num_cin'          => 'required|string|max:20',
                'date_delivrance'  => 'required|date',
                'lieu_delivrance'  => 'required|string|max:255',
                'niveau_etude'     => 'nullable|string|max:255',
                'dernier_diplome_obtenu' => 'nullable|string|max:255',

                // Investissement
                'investissements.*.secteur_invest_id' => 'required|integer|exists:tp_secteur_financement,id_secteur',
                'investissements.*.montant_inv'           => 'required|numeric',
                'investissements.*.source'            => 'nullable|string|max:255',
                'investissements.*.remarques'         => 'nullable|string',

                // Projet
                'nature_projet_id'          => 'required|integer|exists:tp_nature_projet,id_nature_p',
                'nbr_diplome_universitaire' => 'nullable|integer',
                'nbr_autre'                 => 'nullable|integer',
                'id_statut' => 'nullable|in:1,3',
            ]);
            $utilisateur = $projet->investissements->first()?->utilisateur;
            if ($utilisateur) {
                $utilisateur->update([
                    'nom_prenom'          => $request->nom_prenom,
                    'id_type_identite'    => $request->id_type_identite,
                    'cin'                 => $request->cin,
                    'date_delivrance_cin' => $request->date_delivrance_cin,
                    'lieu_delivrance_cin' => $request->lieu_delivrance_cin,
                    'passport'            => $request->passport,
                    'num_fiscal'          => $request->num_fiscal,
                    'adresse'             => $request->adresse,
                    'tel'                 => $request->tel,
                    'lieu_naissance'      => $request->lieu_naissance,
                    'date_naissance'      => $request->date_naissance,
                    'niveau_etude'        => $request->niveau_etude,
                    'dernier_diplome'     => $request->dernier_diplome,
                    'experience'          => $request->experience,
                ]);
            }
            $directeur = $projet->directeur;
            $directeur->update([
                'nom_prenom_d'           => $request->nom_prenom_d,
                'num_cin'                => $request->num_cin,
                'date_delivrance'        => $request->date_delivrance,
                'lieu_delivrance'        => $request->lieu_delivrance,
                'niveau_etude'           => $request->niveau_etude,
                'dernier_diplome_obtenu' => $request->dernier_diplome_obtenu,
            ]);
            $projet->update([
                'nature_projet_id'          => $request->nature_projet_id,
                'id_statut'                 => $request->id_statut, 
                'nbr_diplome_universitaire' => $request->nbr_diplome_universitaire,
                'nbr_autre'                 => $request->nbr_autre,
                'espace_spectacle'          => $request->espace_spectacle ?? 0,
                'caracteristique_spec'      => $request->caracteristique_spec,
                'espace_sportif'            => $request->espace_sportif ?? 0,
                'caracteristique_sportif'   => $request->caracteristique_sportif,
                'espace_activite_service'   => $request->espace_activite_service ?? 0,
                'caracteristique_activite_serv' => $request->caracteristique_activite_serv,
                'espace_activite_culturelle'=> $request->espace_activite_culturelle ?? 0,
                'caracteristique_activite_cult' => $request->caracteristique_activite_cult,
                'remarques'                 => $request->remarques,
                'id_formulaire'             => $projet->id_formulaire ?? 3,
                'commentaire_motif'         => $request->commentaire_motif ?? 0,
            ]);
            if ($request->has('investissements')) {
                $projet->investissements()->delete();
                foreach ($request->investissements as $inv) {
                    Investissement::create([
                        'user_id'          => $utilisateur->id,
                        'secteur_invest_id'=> $inv['secteur_invest_id'],
                        'montant_inv'          => $inv['montant_inv'],
                        'source'           => $inv['source'] ?? null,
                        'remarques'        => $inv['remarques'] ?? null,
                        'id_projet'        => $projet->id_projet,
                    ]);
                }
            }
            DB::commit();
            return redirect()->route('jeune_enfant.index')
                            ->with('success', 'Projet mis à jour avec succès ✅');
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->withErrors(['error' => 'Erreur lors de la mise à jour : '.$e->getMessage()])
                        ->withInput();
        }
    }


    public function destroy($id)
    {
        DB::beginTransaction();
        try {
            $projet = ProjetDivertissementJeuneEnfant::with(['investissements', 'directeur', 'investissements.utilisateur'])->findOrFail($id);
            foreach ($projet->investissements as $investissement) {
                $investissement->delete();
            }
            if ($projet->directeur) {
                $projet->directeur->delete();
            }
            $projet->delete();
            DB::commit();
            return redirect()->route('jeune_enfant.index')
                            ->with('success', 'Projet supprimé avec succès ✅');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->route('jeune_enfant.index')
                            ->withErrors(['error' => 'Erreur lors de la suppression : ' . $e->getMessage()]);
        }
    }


}
