<?php

namespace App\Http\Controllers\Investissement;

use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Illuminate\Http\Request;
use App\Models\TypeFormulaire;
use App\Models\TypeEspaceS;
use App\Models\Utilisateur;
use App\Models\InvestissementSport;
use App\Models\DirecteurEtablissement;
use App\Models\Investissement;
use App\Mail\FormationStatutMail;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;
use App\Notifications\FormationStatutNotification;

use App\Models\NatureProjet; 
use App\Models\SecteurFinancement;
use App\Models\TypeEspaceInstitution;
use App\Models\ServiceInstitution;
use App\Models\SourceFinancement;

class FormationStageController extends Controller
{
    public function index(Request $request)
    {
        $userId = Auth::id(); 

        $formations = InvestissementSport::whereHas('investissements', function ($query) use ($userId) {
                $query->where('user_id', $userId);
            })
            ->with([
                'directeur',
                'natureProjet',
                'investissements' => function($q) use ($userId) {
                    $q->where('user_id', $userId);
                },
                'investissements.utilisateur',
                'statut'
            ])
            ->orderByDesc('created_at')
            ->paginate(10);

        return view('investissement.gestion_formation_stage.index', compact('formations'));
    }

    public function create()
    {
        $formulaire = TypeFormulaire::find(5);
        $types_espace_es = TypeEspaceS::all();
        $types_espace = TypeEspaceInstitution::all();
        $nature_projet = NatureProjet::all();
        $secteurs = SecteurFinancement::all();
        $service_inst = ServiceInstitution::all();
        $source_fin = SourceFinancement::all();
       

        $user = Auth::user();

        return view('investissement.gestion_formation_stage.create', compact(
            'formulaire','types_espace_es','types_espace','service_inst','secteurs',
            'nature_projet','source_fin','user'
        ));
    }


    public function store(Request $request)
    {
        DB::beginTransaction();
        try {
            $request->validate([
                // Directeur
                'nom_prenom_d'     => 'required|string|max:255',
                'num_cin'          => 'required|string|max:20',
                'date_delivrance'  => 'required|date',
                'lieu_delivrance'  => 'required|string|max:255',
                'niveau_etude'     => 'nullable|string|max:255',
                'dernier_diplome_obtenu' => 'nullable|string|max:255',

                // Investissements (financement)
                'investissements.*.secteur_invest_id'     => 'required|integer|exists:tp_secteur_financement,id_secteur',
                'investissements.*.montant_inv'           => 'required|numeric',
                'investissements.*.id_source_financement' => 'required|integer|exists:tp_source_financement,id_source',
                'investissements.*.montant_fin'           => 'required|numeric',

                // Investissement Sport
                'id_formulaire'   => 'required|integer|exists:tp_type_formulaire,id_type',
                'id_statut'       => 'nullable|integer|in:1,2,3',
                'type_espace_s_id'=> 'required|integer|exists:tp_type_espace_s,id_type_es',
                'nbr_chambre'     => 'nullable|integer',
                'nbr_stade'       => 'nullable|integer',
                'nbr_stages_sportifs' => 'nullable|integer',
                'nbr_salles'      => 'nullable|integer',
                'nbr_vestiaire'   => 'nullable|integer',
                'nbr_salle_renforcement_musculaire' => 'nullable|integer',
                'capacite_accueil_foyer' => 'nullable|integer',
                'capacite_accueil_stade' => 'nullable|integer',
                'heberg_rest'     => 'nullable|boolean',
                'stage_activ'     => 'nullable|boolean',
                'espace_exercice' => 'nullable|boolean',
                'activite_culturel' => 'nullable|boolean',
                'manifestation_sportif' => 'nullable|boolean',
                'autres'          => 'nullable|boolean',
                'autres_services' => 'nullable|string|max:200',
                'nom_projet'      => 'nullable|string|max:100',
                'nature_projet_id'=> 'nullable|integer|exists:tp_nature_projet,id_nature_p',
                'nbr_emplois'     => 'nullable|integer',
                'remarques'       => 'nullable|string',
                'commentaire_motif' => 'nullable|string',
            ]);
            $directeur = DirecteurEtablissement::create([
                'nom_prenom_d'          => $request->nom_prenom_d,
                'num_cin'               => $request->num_cin,
                'date_delivrance'       => $request->date_delivrance,
                'lieu_delivrance'       => $request->lieu_delivrance,
                'niveau_etude'          => $request->niveau_etude,
                'dernier_diplome_obtenu'=> $request->dernier_diplome_obtenu,
            ]);

            $formation = InvestissementSport::create([
                'id_formulaire'  => $request->id_formulaire,
                'id_statut'      => $request->id_statut ?? 2,
                'type_espace_s_id' => $request->type_espace_s_id,
                'nbr_chambre'    => $request->nbr_chambre,
                'nbr_stade'      => $request->nbr_stade,
                'nbr_stages_sportifs' => $request->nbr_stages_sportifs,
                'nbr_salles'     => $request->nbr_salles,
                'nbr_vestiaire'  => $request->nbr_vestiaire,
                'nbr_salle_renforcement_musculaire' => $request->nbr_salle_renforcement_musculaire,
                'capacite_accueil_foyer' => $request->capacite_accueil_foyer,
                'capacite_accueil_stade' => $request->capacite_accueil_stade,
                'heberg_rest'    => $request->heberg_rest,
                'stage_activ'    => $request->stage_activ,
                'espace_exercice'=> $request->espace_exercice,
                'activite_culturel' => $request->activite_culturel,
                'manifestation_sportif' => $request->manifestation_sportif,
                'autres'         => $request->autres,
                'autres_services'=> $request->autres_services,
                'nom_projet'     => $request->nom_projet,
                'nature_projet_id' => $request->nature_projet_id,
                'nbr_emplois'    => $request->nbr_emplois,
                'directeur_etabl_id' => $directeur->id_directeur,
                'remarques'      => $request->remarques,
                'commentaire_motif' => $request->commentaire_motif,
            ]);

            if ($request->has('investissements')) {
                foreach ($request->investissements as $inv) {
                    Investissement::create([
                        'user_id' => Auth::id(),
                        'id_inv_sport'       => $formation->id_inv_sport, 
                        'secteur_invest_id'  => $inv['secteur_invest_id'],
                        'montant_inv'        => $inv['montant_inv'],
                        'id_source_financement' => $inv['id_source_financement'] ?? null,
                        'montant_fin'        => $inv['montant_fin'] ?? null,
                    ]);
                }
            }

            DB::commit();
        return redirect()->route('formation.index')
            ->with('success', ' تمّ تسجيل الإستثمار الرياضي بنجاح');

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->withErrors(['error' => " خطأ أثناء التسجيل : " . $e->getMessage()])
                        ->withInput();
        }

    }

    public function edit($id)
    {
        $formation = InvestissementSport::with(['investissements', 'directeur', 'investissements.utilisateur'])->findOrFail($id);

        $formulaire = TypeFormulaire::find(5);
        $types_espace_es = TypeEspaceS::all();
        $types_espace = TypeEspaceInstitution::all();
        $nature_projet = NatureProjet::all();
        $secteurs = SecteurFinancement::all();
        $service_inst = ServiceInstitution::all();
        $source_fin = SourceFinancement::all();

        return view('investissement.gestion_formation_stage.edit', compact(
            'formation','formulaire','types_espace_es','types_espace',
            'service_inst','secteurs','nature_projet','source_fin'
        ));
    }

    public function update(Request $request, $id)
    {
        DB::beginTransaction();
        try {
            $request->validate([
                // Directeur
                'nom_prenom_d'     => 'required|string|max:255',
                'num_cin'          => 'required|string|max:20',
                'date_delivrance'  => 'required|date',
                'lieu_delivrance'  => 'required|string|max:255',
                'niveau_etude'     => 'nullable|string|max:255',
                'dernier_diplome_obtenu' => 'nullable|string|max:255',

                // Investissements (financement)
                'investissements.*.secteur_invest_id'     => 'required|integer|exists:tp_secteur_financement,id_secteur',
                'investissements.*.montant_inv'           => 'required|numeric',
                'investissements.*.id_source_financement' => 'required|integer|exists:tp_source_financement,id_source',
                'investissements.*.montant_fin'           => 'required|numeric',

                // Investissement Sport
                'id_formulaire'   => 'required|integer|exists:tp_type_formulaire,id_type',
                'id_statut'       => 'nullable|integer|in:1,2,3',
                'type_espace_s_id'=> 'required|integer|exists:tp_type_espace_s,id_type_es',
                'nbr_chambre'     => 'nullable|integer',
                'nbr_stade'       => 'nullable|integer',
                'nbr_stages_sportifs' => 'nullable|integer',
                'nbr_salles'      => 'nullable|integer',
                'nbr_vestiaire'   => 'nullable|integer',
                'nbr_salle_renforcement_musculaire' => 'nullable|integer',
                'capacite_accueil_foyer' => 'nullable|integer',
                'capacite_accueil_stade' => 'nullable|integer',
                'heberg_rest'     => 'nullable|boolean',
                'stage_activ'     => 'nullable|boolean',
                'espace_exercice' => 'nullable|boolean',
                'activite_culturel' => 'nullable|boolean',
                'manifestation_sportif' => 'nullable|boolean',
                'autres'          => 'nullable|boolean',
                'autres_services' => 'nullable|string|max:200',
                'nom_projet'      => 'nullable|string|max:100',
                'nature_projet_id'=> 'nullable|integer|exists:tp_nature_projet,id_nature_p',
                'nbr_emplois'     => 'nullable|integer',
                'remarques'       => 'nullable|string',
                'commentaire_motif'       => 'nullable|string',
                
            ]);

            $formation = InvestissementSport::with(['directeur','investissements','investissements.utilisateur'])
                ->findOrFail($id);

            $directeur = $formation->directeur;
            if ($directeur) {
                $directeur->update([
                    'nom_prenom_d' => $request->nom_prenom_d,
                    'num_cin' => $request->num_cin,
                    'date_delivrance' => $request->date_delivrance,
                    'lieu_delivrance' => $request->lieu_delivrance,
                    'niveau_etude' => $request->niveau_etude,
                    'dernier_diplome_obtenu' => $request->dernier_diplome_obtenu,
                ]);
            }

            $formation->update([
                'id_formulaire'  => $request->id_formulaire,
                'id_statut'      => $request->id_statut ?? 2,
                'type_espace_s_id' => $request->type_espace_s_id,
                'nbr_chambre'    => $request->nbr_chambre,
                'nbr_stade'      => $request->nbr_stade,
                'nbr_stages_sportifs' => $request->nbr_stages_sportifs,
                'nbr_salles'     => $request->nbr_salles,
                'nbr_vestiaire'  => $request->nbr_vestiaire,
                'nbr_salle_renforcement_musculaire' => $request->nbr_salle_renforcement_musculaire,
                'capacite_accueil_foyer' => $request->capacite_accueil_foyer,
                'capacite_accueil_stade' => $request->capacite_accueil_stade,
                'heberg_rest'    => $request->heberg_rest,
                'stage_activ'    => $request->stage_activ,
                'espace_exercice'=> $request->espace_exercice,
                'activite_culturel' => $request->activite_culturel,
                'manifestation_sportif' => $request->manifestation_sportif,
                'autres'         => $request->autres,
                'autres_services'=> $request->autres_services,
                'nom_projet'     => $request->nom_projet,
                'nature_projet_id' => $request->nature_projet_id,
                'nbr_emplois'    => $request->nbr_emplois,
                'remarques'      => $request->remarques,
                'commentaire_motif' => $request->commentaire_motif,
            ]);

            $utilisateur = $formation->investissements->first()?->utilisateur;
            if ($request->has('investissements')) {
                $formation->investissements()->delete();
                foreach ($request->investissements as $inv) {
                    Investissement::create([
                        'user_id' => $utilisateur->id ?? Auth::id(),
                        'secteur_invest_id' => $inv['secteur_invest_id'],
                        'montant_inv' => $inv['montant_inv'],
                        'source' => $inv['id_source_financement'] ?? null,
                        'remarques' => $inv['remarques'] ?? null,
                        'id_inv_sport' => $formation->id_inv_sport,
                    ]);
                }
            }
            DB::commit();
            return redirect()->route('formation.index')
                ->with('success', ' تمّ تعديل الإستثمار الرياضي بنجاح');

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->withErrors(['error' => " خطأ أثناء التحديث : ".$e->getMessage()])
                        ->withInput();
        }
    }

    public function show($id)
    {
        $formation = InvestissementSport::with(['investissements', 'directeur', 'investissements.utilisateur'])->findOrFail($id);

        $formulaire = TypeFormulaire::find(5);
        $types_espace_es = TypeEspaceS::all();
        $types_espace = TypeEspaceInstitution::all();
        $nature_projet = NatureProjet::all();
        $secteurs = SecteurFinancement::all();
        $service_inst = ServiceInstitution::all();
        $source_fin = SourceFinancement::all();

        return view('investissement.gestion_formation_stage.show', compact(
            'formation','formulaire','types_espace_es','types_espace',
            'service_inst','secteurs','nature_projet','source_fin'
        ));
    }

    public function destroy($id)
    {
        DB::beginTransaction();
        try {
            $formation = InvestissementSport::findOrFail($id);
            $formation->investissements()->delete();
            $formation->delete();

            DB::commit();
            return redirect()->route('formation.index')
                ->with('success', 'تمّ حذف الإستثمار الرياضي بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->withErrors(['error' => " خطأ أثناء الحذف : " . $e->getMessage()]);
        }
    }

    public function toggleStatut(Request $request, $id)
    {
        $formation = InvestissementSport::with('investissements.utilisateur')->findOrFail($id);
        $formation->id_statut = $request->id_statut;

        $user = $formation->investissements->first()?->utilisateur;

        if ($user && !empty($user->email)) { 
            if ($request->id_statut == 3 && $request->commentaire_motif) {
                $formation->commentaire_motif = $request->commentaire_motif;

                Mail::to($user->email)->send(
                    new FormationStatutMail('refuse', $user, null, $request->commentaire_motif)
                );

                $user->notify(new FormationStatutNotification('refuse', $request->commentaire_motif));

            } elseif ($request->id_statut == 1) {
                Mail::to($user->email)->send(
                    new FormationStatutMail('accepte', $user)
                );

                $user->notify(new FormationStatutNotification('accepte'));
            }
        } else {
            Log::warning("Aucun utilisateur avec email trouvé pour le projet ID: {$id}");
        }

        $formation->save();

        return redirect()->back()->with('success', 'تم تغيير حالة الطلب بنجاح');
    }

}
