<?php

namespace App\Http\Controllers\Investissement;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\DebutActivite;
use App\Models\Immobilier;
use App\Models\Utilisateur;
use Illuminate\Support\Facades\Storage;
use App\Mail\DebutActiviteStatutMail;
use Illuminate\Support\Facades\Mail;
use Carbon\Carbon;
use App\Notifications\DebutActiviteStatutNotification;

class DebutActiviteController extends Controller
{
    public function index2(Request $request)
    {
        $debutActivites = DebutActivite::with('immobilier.utilisateur')
            ->orderByDesc('created_at')
            ->paginate(10);

        return view('investissement.gestion_salle_sport.debut_activite.index', compact('debutActivites'));
    }

    

        public function create2(Request $request)
{
    $immobilier_id = $request->query('immobilier_id', null);

    if ($immobilier_id) {
        $immobilier = Immobilier::with('utilisateur')->find($immobilier_id);
    } else {
        $immobilier = null;
    }

    return view('investissement.gestion_salle_sport.debut_activite.create', compact('immobilier_id', 'immobilier'));
}


    public function store2(Request $request)
    {
        $validated = $request->validate([
            'immobilier_id' => 'required|exists:immobilier,id_imm',
            'date_debut_activite' => 'required|date',
            'preuve_propriete_location' => 'required|file|mimes:pdf,jpg,jpeg,png|max:2048',
            'plan_architectural' => 'required|file|mimes:pdf,jpg,jpeg,png|max:2048',
            'declaration_investissement' => 'required|file|mimes:pdf,jpg,jpeg,png|max:2048',
            'identification' => 'required|file|mimes:pdf,jpg,jpeg,png|max:2048',
            'extrait_registre_national_entreprise' => 'required|file|mimes:pdf,jpg,jpeg,png|max:2048',
            'certificat_prevention' => 'required|file|mimes:pdf,jpg,jpeg,png|max:2048',
            'contrat_assurance' => 'required|file|mimes:pdf,jpg,jpeg,png|max:2048',
            'approbation_autorite_minicipale' => 'required|file|mimes:pdf,jpg,jpeg,png|max:2048',
        ]);

        $uploads = [];
        foreach ([
            'preuve_propriete_location',
            'plan_architectural',
            'declaration_investissement',
            'identification',
            'extrait_registre_national_entreprise',
            'certificat_prevention',
            'contrat_assurance',
            'approbation_autorite_minicipale',
        ] as $fileField) {
            if ($request->hasFile($fileField)) {
            $file = $request->file($fileField);
            $originalName = $file->getClientOriginalName();
            $filename = pathinfo($originalName, PATHINFO_FILENAME);
            $extension = $file->getClientOriginalExtension();

            // Générer un nom unique en ajoutant un timestamp
            $uniqueName = $filename. '.' . $extension;

            // Stocker le fichier dans public/uploads/debut_activite
            $uploads[$fileField] = $file->storeAs('uploads/debut_activite', $uniqueName, 'public');
        }
    }

        DebutActivite::create([
            'immobilier_id' => $request->immobilier_id,
            'date_debut_activite' => $request->date_debut_activite,
            'preuve_propriete_location' => $uploads['preuve_propriete_location'] ?? null,
            'plan_architectural' => $uploads['plan_architectural'] ?? null,
            'declaration_investissement' => $uploads['declaration_investissement'] ?? null,
            'identification' => $uploads['identification'] ?? null,
            'extrait_registre_national_entreprise' => $uploads['extrait_registre_national_entreprise'] ?? null,
            'certificat_prevention' => $uploads['certificat_prevention'] ?? null,
            'contrat_assurance' => $uploads['contrat_assurance'] ?? null,
            'approbation_autorite_minicipale' => $uploads['approbation_autorite_minicipale'] ?? null,
            'etat_formulaire' => 'remplir',
            'id_type_formulaire' => 2,
            'id_statut' => 2,
        ]);

        return redirect()->route('salle-sport.index2')->with('success', 'تم تسجيل بداية النشاط بنجاح ');
    }

public function verifierIdentite2(Request $request)
{
    $request->validate([
        'id_type_identite' => 'required|exists:tp_type_identite_pers,id_type',
        'identifiant' => 'required|string',
        'immobilier_id' => 'required|exists:immobilier,id_imm'
    ]);

    $idType = $request->id_type_identite;
    $identifiant = $request->identifiant;
    $immobilier_id = $request->immobilier_id;

    $immobilier = Immobilier::with('utilisateur','gouvernorat')->find($immobilier_id);
    if(!$immobilier) return response()->json(['success' => false]);

    $user = $immobilier->utilisateur;
    $match = false;

    if($idType == 1 && $user->cin == $identifiant) $match = true;
    if($idType == 2 && $user->passport == $identifiant) $match = true;
    if($idType == 3 && $user->num_fiscal == $identifiant) $match = true;

    if($match) {
        return response()->json([
            'success' => true,
            'user' => $user,
            'immobilier' => [
                'id_imm' => $immobilier->id_imm,
                'gouvernorat_nom_ar' => $immobilier->gouvernorat->nom_ar ?? null
            ]
        ]);
    }

    return response()->json(['success' => false]);
}

        public function edit2($id)
    {
        $debutActivite = DebutActivite::with('immobilier.utilisateur', 'immobilier.gouvernorat')->findOrFail($id);
        $immobilier_id = $debutActivite->immobilier_id;

        return view('investissement.gestion_salle_sport.debut_activite.edit', compact('debutActivite', 'immobilier_id'));
    }

    
public function update2(Request $request, $id)
{
    $debutActivite = DebutActivite::findOrFail($id);

    // Validation
    $validated = $request->validate([
        'immobilier_id' => 'required|exists:immobilier,id_imm',
        'date_debut_activite' => 'required|date',
        'id_statut' => 'nullable|in:1,3',
        'commentaire_motif' => 'nullable|string|required_if:id_statut,3|max:255',
        'preuve_propriete_location' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
        'plan_architectural' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
        'declaration_investissement' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
        'identification' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
        'extrait_registre_national_entreprise' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
        'certificat_prevention' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
        'contrat_assurance' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
        'approbation_autorite_minicipale' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
    ]);

    // Gestion des fichiers uploadés
    $uploads = [];
    foreach ([
        'preuve_propriete_location',
        'plan_architectural',
        'declaration_investissement',
        'identification',
        'extrait_registre_national_entreprise',
        'certificat_prevention',
        'contrat_assurance',
        'approbation_autorite_minicipale',
    ] as $fileField) {
        if ($request->hasFile($fileField)) {
            // Supprimer ancien fichier si existant
            if ($debutActivite->$fileField) {
                Storage::disk('public')->delete($debutActivite->$fileField);
            }

            $file = $request->file($fileField);
            $originalName = $file->getClientOriginalName();
            $filename = pathinfo($originalName, PATHINFO_FILENAME);
            $extension = $file->getClientOriginalExtension();

            // Générer un nom unique en ajoutant un timestamp
            $uniqueName = $filename . '.' . $extension;

            // Stocker le fichier
            $uploads[$fileField] = $file->storeAs('uploads/debut_activite', $uniqueName, 'public');
        }
    }

    // Préparer les données à mettre à jour
    $updateData = [
        'immobilier_id' => $request->immobilier_id,
        'date_debut_activite' => $request->date_debut_activite,
        'id_statut' => $request->id_statut ?? $debutActivite->id_statut, // garde l'ancien statut si null
        'commentaire_motif' => $request->commentaire_motif,
        'etat_formulaire' => in_array($request->id_statut, [1, 3]) ? 'reponse' : $debutActivite->etat_formulaire,
    ];

    // Merge fichiers uploadés
    $updateData = array_merge($updateData, $uploads);

    // Mise à jour de l'enregistrement
    $debutActivite->update($updateData);

    // Envoi du mail selon le statut
    $user = $debutActivite->immobilier->utilisateur;
    if ($request->id_statut == 3 && $request->commentaire_motif) {
        Mail::to($user->email)->send(
            new DebutActiviteStatutMail('refuse', $user, null, $request->commentaire_motif)
        );
    } elseif ($request->id_statut == 1) {
        Mail::to($user->email)->send(
            new DebutActiviteStatutMail('accepte', $user)
        );
    }

    return redirect()->route('salle-sport.index2')
                     ->with('success', 'تم تحديث بيانات بداية النشاط بنجاح');
}


        public function toggleStatut2(Request $request, $id)
{
    $debutActivite = DebutActivite::with('immobilier.utilisateur')->findOrFail($id);

    // Supprimer la restriction sur le statut actuel
    $debutActivite->id_statut = $request->id_statut;

    if (in_array($request->id_statut, [1, 3])) {
        $debutActivite->etat_formulaire = 'reponse';
    }

    $user = $debutActivite->immobilier->utilisateur;

    if ($request->id_statut == 3 && $request->commentaire_motif) {
        $debutActivite->commentaire_motif = $request->commentaire_motif;

        Mail::to($user->email)->send(
            new DebutActiviteStatutMail('refuse', $user, null, $request->commentaire_motif)
        );

        $user->notify(new DebutActiviteStatutNotification('refuse', $request->commentaire_motif));

    } elseif ($request->id_statut == 1) {
        $dateDebut = Carbon::now()->addDays(15);

        Mail::to($user->email)->send(
            new DebutActiviteStatutMail('accepte', $user, $dateDebut)
        );

        $user->notify(new DebutActiviteStatutNotification('accepte', $dateDebut));
    }

    $debutActivite->save();

    return redirect()->back()->with('success', 'تم تغيير حالة الطلب بنجاح');
}


public function show2($id)
    {
        $debutActivite = DebutActivite::with('immobilier.utilisateur', 'immobilier.gouvernorat')->findOrFail($id);
        $immobilier_id = $debutActivite->immobilier_id;

        return view('investissement.gestion_salle_sport.debut_activite.show', compact('debutActivite', 'immobilier_id'));
    }

    
}