<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Concours;
use App\Models\FieldDefinition;
use App\Models\FieldValue;
use App\Models\FieldFile;
use App\Models\Candidate;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Validator;    // <-- import here
use Illuminate\Database\QueryException; 

class ConcoursExtController extends Controller
{
    // list available concours
    public function index(Request $request)
    {
        $q = trim($request->input('q', ''));

        $query = Concours::active()
            ->select(['id', 'titre', 'etat', 'date_debut', 'date_fin', 'id_user'])
            ->withCount(['fields', 'candidates'])
            ->orderBy('date_debut', 'desc');

        if ($q !== '') {
            $query->where('titre', 'like', "%{$q}%");
        }

        $concours = $query->paginate(12)->withQueryString();

        return view('concours_ext.index', compact('concours', 'q'));
    }

    // Show the public form for a single concours
// at top of file there's already: use Illuminate\Support\Facades\DB;
public function show(Concours $concours)
{
    $fieldDefinitions = $concours->fields()
        ->where('field_definitions.is_visible', 1)
        ->orderBy('sort_order')
        ->get();

    // load gouvernorats (Arabic names)
    $gouvernorats = DB::table('tp_gouvernorat')
        ->select('id_gouver', 'nom_ar')
        ->orderBy('nom_ar')
        ->get(); // collection of rows with nom_ar

    return view('concours_ext.create', compact('concours', 'fieldDefinitions', 'gouvernorats'));
}


    /**
     * Store a public submission for the given concours.
     */



public function store(Request $request, Concours $concours)
{
    // load definitions used by this concours and that are visible
    $defs = $concours->fields()->where('field_definitions.is_visible', 1)->get()->keyBy('id');

    // gouvernorat options not needed here except to resolve id->name later
    // validation rules (we validate gouvernorat_id coming from your select)
    $rules = [
        'nom' => ['required','string','max:100'],
        'prenom' => ['required','string','max:100'],
        'cin' => ['nullable','regex:/^\d{8}$/'],
        'date_naissance' => ['nullable','date'],
        'tel' => ['nullable','regex:/^\d{8}$/'],
        'email' => ['nullable','email','max:255'],
        'adresse' => ['nullable','string','max:255'],
        // your form uses gouvernorat_id => ensure it exists in tp_gouvernorat
        'gouvernorat_id' => ['nullable','integer','exists:tp_gouvernorat,id_gouver'],
        'code_postale' => ['nullable','regex:/^\d{4}$/'],
    ];

    // add dynamic field rules (unchanged)
    foreach ($defs as $def) {
        $fieldKey = 'field_'.$def->id;
        $r = [];
        $r[] = $def->is_required ? 'required' : 'nullable';

        switch ($def->input_type) {
            case 'date': $r[] = 'date'; break;
            case 'number': $r[] = 'numeric'; break;
            case 'file': $r[] = 'file'; break;
            case 'email': $r[] = 'email'; break;
            default:
                if (in_array($def->input_type, ['checkbox','multiselect'])) {
                    if ($def->is_required) {
                        $r[] = 'array';
                        $r[] = 'min:1';
                    } else {
                        $r[] = 'array';
                    }
                } else {
                    $r[] = 'string';
                    $r[] = 'max:2000';
                }
        }

        $rules[$fieldKey] = $r;
    }

    // friendly messages (Arabic)
    $messages = [
        'cin.regex' => 'يجب أن يتكون رقم بطاقة التعريف من 8 أرقام.',
        'tel.regex' => 'يجب أن يتكون رقم الهاتف من 8 أرقام.',
        'code_postale.regex' => 'يجب أن يتكون الرمز البريدي من 4 أرقام.',
    ];

    $validator = Validator::make($request->all(), $rules, $messages);

    // Post-validation: ensure cin is not already used for this concours
    $validator->after(function ($v) use ($request, $concours) {
        $cin = $request->input('cin');
        if ($cin) {
            // if this concours already has a candidate with same cin => validation error
        $exists = DB::table('concours_candidate')
    ->join('candidates', 'concours_candidate.candidate_id', '=', 'candidates.id')
    ->where('concours_candidate.concours_id', $concours->id)
    ->where('candidates.cin', $cin)
    ->exists();

if ($exists) {
    $v->errors()->add('cin', 'رقم بطاقة التعريف مستخدم مسبقاً لهذا المناظرة.');
}

        }
    });

    if ($validator->fails()) {
        return back()->withErrors($validator)->withInput();
    }

    $validated = $validator->validated();

    // Resolve gouvernorat name (nom_ar) from gouvernorat_id (if provided)
    $gouvName = null;
    if (!empty($validated['gouvernorat_id'])) {
        $gouvName = DB::table('tp_gouvernorat')
            ->where('id_gouver', $validated['gouvernorat_id'])
            ->value('nom_ar');
    }

    DB::beginTransaction();

    try {
        // Try to reuse existing candidate by CIN (if exists anywhere)
        $candidate = null;
        if (!empty($validated['cin'])) {
            $candidate = Candidate::where('cin', $validated['cin'])->first();
        }

        if ($candidate) {
            // update some fields (optional)
            $candidate->update([
                'nom' => $validated['nom'],
                'prenom' => $validated['prenom'],
                'date_naissance' => $validated['date_naissance'] ?? $candidate->date_naissance,
                'tel' => $validated['tel'] ?? $candidate->tel,
                'email' => $validated['email'] ?? $candidate->email,
                'adresse' => $validated['adresse'] ?? $candidate->adresse,
                'gouvernorat' => $gouvName ?? $candidate->gouvernorat,
                'code_postale' => $validated['code_postale'] ?? $candidate->code_postale,
            ]);
        } else {
            $candidate = Candidate::create([
                'nom' => $validated['nom'],
                'prenom' => $validated['prenom'],
                'cin' => $validated['cin'] ?? null,
                'date_naissance' => $validated['date_naissance'] ?? null,
                'tel' => $validated['tel'] ?? null,
                'email' => $validated['email'] ?? null,
                'adresse' => $validated['adresse'] ?? null,
                'gouvernorat' => $gouvName ?? null,
                'code_postale' => $validated['code_postale'] ?? null,
                'etat' => 'en_cours',
            ]);
        }

        // attach candidate to concours if not already attached
        if (method_exists($concours, 'candidates')) {
            if (! $concours->candidates()->where('candidates.id', $candidate->id)->exists()) {
                $concours->candidates()->attach($candidate->id);
            }
        }

        // ------ save dynamic fields & files (your existing logic) ------
        foreach ($defs as $def) {
            $fieldKey = 'field_'.$def->id;

            if ($def->input_type === 'file') {
                if ($request->hasFile($fieldKey)) {
                    $file = $request->file($fieldKey);
                    if ($file && $file->isValid()) {
                        $stream = fopen($file->getRealPath(), 'rb');
                        $content = stream_get_contents($stream);
                        fclose($stream);

                        FieldFile::create([
                            'candidate_id'         => $candidate->id,
                            'field_definition_id'  => $def->id,
                            'file_path'            => '',
                            'original_name'        => $file->getClientOriginalName(),
                            'mime_type'            => $file->getClientMimeType(),
                            'size'                 => $file->getSize(),
                            'content'              => $content,
                        ]);
                    }
                }
            } else {
                if ($request->has($fieldKey)) {
                    $val = $request->input($fieldKey);
                    if (is_array($val)) {
                        $stored = json_encode(array_values($val), JSON_UNESCAPED_UNICODE);
                        FieldValue::create([
                            'candidate_id' => $candidate->id,
                            'field_definition_id' => $def->id,
                            'value' => $stored,
                        ]);
                    } else {
                        if ($request->filled($fieldKey) || $def->is_required) {
                            FieldValue::create([
                                'candidate_id' => $candidate->id,
                                'field_definition_id' => $def->id,
                                'value' => $val,
                            ]);
                        }
                    }
                }
            }
        }
        // ---------------------------------------------------------------

        DB::commit();

        return redirect()->route('concours.show', $concours)->with('success', 'تم إرسال مطلب الترشّح بنجاح. شكرًا لك.');
    } catch (QueryException $qe) {
        DB::rollBack();
        // If duplicate key at DB level (defensive), show field error
        $msg = $qe->getMessage();
        if (str_contains($msg, 'Duplicate') || str_contains($msg, 'duplicate')) {
            return back()->withInput()->withErrors(['cin' => 'رقم بطاقة التعريف مستخدم بالفعل.']);
        }
        \Log::error('Candidate store error: ' . $qe->getMessage());
        return back()->withInput()->withErrors(['general' => 'حدث خطأ أثناء حفظ المطلب. حاول مرة أخرى أو تواصل مع الدعم.']);
    } catch (\Throwable $e) {
        DB::rollBack();
        \Log::error('Candidate store error: ' . $e->getMessage());
        return back()->withInput()->withErrors(['general' => 'حدث خطأ أثناء حفظ المطلب. حاول مرة أخرى أو تواصل مع الدعم.']);
    }
}


public function results(Request $request, Concours $concours)
{
    $q = trim($request->input('q', ''));

    if ($q === '') {
        // empty paginator so the view can still call ->links() without errors
        $empty = new Collection([]);
        $candidates = new LengthAwarePaginator($empty, 0, 12, 1, [
            'path' => $request->url(),
            'query' => $request->query(),
        ]);

        // pass an indicator to the view that no search was executed
        return view('concours_ext.results', compact('concours', 'candidates', 'q'))
            ->with('searchRan', false);
    }

    // when there is a query, search candidate fields attached to this concours
    $query = $concours->candidates()
        ->with(['fieldValues', 'files'])
        ->where(function ($qb) use ($q) {
            $qb->where('nom', 'like', "%{$q}%")
               ->orWhere('prenom', 'like', "%{$q}%")
               ->orWhere('cin', 'like', "%{$q}%")
               ->orWhere('email', 'like', "%{$q}%");
        })
        ->orderBy('created_at', 'desc');

    $candidates = $query->paginate(12)->withQueryString();

    return view('concours_ext.results', compact('concours', 'candidates', 'q'))
        ->with('searchRan', true);
}

}
