@extends('backoffice.dashboard')

@section('content')
@php
  $etatKey = $concours_ext->etat ?? 'en_cours';
  $etatLabel = match($etatKey) {
    'en_cours' => 'قيد الدراسة',
    'refuse' => 'مرفوض',
    'acceptation_primaire' => 'قبول أولي',
    'acceptation_definitif' => 'قبول نهائي',
    default => $etatKey,
  };
@endphp

<style>
  /* Scoped styling — move to global CSS if you prefer */
  .concours-card { border-radius:12px; box-shadow:0 6px 18px rgba(28,34,40,0.06); overflow:hidden; border:0; background:#fff; }
  .concours-header { padding:18px 22px; background:#fff; border-bottom:1px solid rgba(0,0,0,0.04); display:flex; justify-content:space-between; align-items:center; gap:12px; flex-wrap:wrap; }
  .concours-title { font-size:1.2rem; font-weight:700; }
  .concours-meta { color:#6c757d; font-size:0.9rem; }

  .fields-grid { display:grid; grid-template-columns: repeat(auto-fit, minmax(260px, 1fr)); gap:12px; margin-bottom:12px; }
  .field-card { background:#ffffff; border:1px solid rgba(0,0,0,0.04); padding:14px; border-radius:8px; min-height:74px; display:flex; flex-direction:column; justify-content:center; }
  .field-label { font-weight:700; color:#2b343b; margin-bottom:6px; }
  .field-value { color:#454f57; word-break:break-word; white-space:pre-wrap; }

  .files-grid { display:grid; grid-template-columns: repeat(auto-fit, minmax(200px,1fr)); gap:12px; margin-top:10px; }
  .file-card { background:#f8f9fa; border-radius:8px; padding:12px; border:1px solid rgba(0,0,0,0.03); display:flex; flex-direction:column; justify-content:space-between; height:100%; }
  .file-meta { font-size:0.85rem; color:#6c757d; margin-top:8px; }

  .status-control { display:flex; gap:8px; align-items:center; }
  .save-state-btn[disabled] { opacity:0.6; pointer-events:none; }

  /* RTL adjustments */
  .field-card, .file-card { direction: rtl; text-align: right; }
  .status-control { direction: ltr; } /* keep control LTR for dropdown alignment */

  /* soft status badge */
  .status-badge {
    display:inline-flex;
    align-items:center;
    gap:10px;
    padding:8px 12px;
    border-radius:999px;
    font-weight:600;
    font-size:0.9rem;
    box-shadow: 0 1px 0 rgba(255,255,255,0.4) inset;
    border: 1px solid rgba(0,0,0,0.06);
    color: #21303a;
    min-height:34px;
  }
  .status-badge::before{
    content: '';
    width:10px; height:10px; border-radius:50%;
    display:inline-block; flex:0 0 10px;
    box-shadow: 0 1px 0 rgba(255,255,255,0.35) inset;
  }

  /* Pastel palettes */
  .status-en_cours { background: linear-gradient(180deg, rgba(247,249,250,1), rgba(240,247,250,1)); border-color: rgba(34,150,243,0.12); color:#0f4c81; }
  .status-en_cours::before { background: rgba(34,150,243,0.75); }
  .status-refuse { background: linear-gradient(180deg, rgba(255,244,245,1), rgba(255,239,241,1)); border-color: rgba(220,53,69,0.12); color:#6b0710; }
  .status-refuse::before { background: rgba(220,53,69,0.75); }
  .status-acceptation_primaire { background: linear-gradient(180deg, rgba(255,250,240,1), rgba(255,247,235,1)); border-color: rgba(255,193,7,0.12); color:#6b4a03; }
  .status-acceptation_primaire::before { background: rgba(255,193,7,0.75); }
  .status-acceptation_definitif { background: linear-gradient(180deg, rgba(245,255,250,1), rgba(238,255,244,1)); border-color: rgba(40,167,69,0.12); color:#0b4d2e; }
  .status-acceptation_definitif::before { background: rgba(40,167,69,0.75); }

  /* Footer layout & button spacing */
  .concours-footer {
    padding: 14px 18px;
    display:flex;
    align-items:center;
    justify-content:space-between;
    gap:12px;
    background: transparent;
    border-top: 1px solid rgba(0,0,0,0.04);
  }

  .concours-footer .left-actions,
  .concours-footer .right-actions {
    display:flex;
    align-items:center;
    gap:10px;
  }

  /* put return button to extreme left visually */
  .concours-footer .left-actions { justify-content:flex-start; }
  /* put save button to extreme right visually */
  .concours-footer .right-actions { justify-content:flex-end; }

  /* make buttons slightly taller and with nicer spacing */
  .concours-footer .btn {
    padding: .45rem .8rem;
    border-radius: 8px;
    font-weight: 600;
  }

  /* mobile: stack actions, keep Save on top (so it's on the right visually on wide screens, but first on narrow view) */
  @media (max-width:576px) {
    .concours-footer {
      flex-direction: column-reverse;
      align-items: stretch;
      gap:10px;
    }
    .concours-footer .left-actions,
    .concours-footer .right-actions {
      width:100%;
      justify-content: center;
    }
    .concours-footer .right-actions { justify-content:flex-end; } /* keep save to the right on mobile if there's space */
    .concours-footer .left-actions .btn,
    .concours-footer .right-actions .btn { width: 140px; }
  }

  /* small visual polish for file buttons area */
  .file-card .btn { font-size: .85rem; padding: .35rem .6rem; border-radius:6px; }
</style>

<section class="py-4">
  <div class="container">
    <div class="row justify-content-center">
      <div class="col-lg-10">

        <div class="card concours-card">

          {{-- Header --}}
          <div class="concours-header">
            <div>
              <div class="concours-title">عرض الطلب</div>
              <div class="concours-meta">تفاصيل الطلب المعرّف رقم #{{ $concours_ext->id }}</div>
            </div>

            <div class="status-control" style="display:flex;align-items:center;gap:12px;">
              <span id="currentStatusBadge" class="status-badge status-{{ e($etatKey) ?? 'unknown' }}">
                {{ $etatLabel }}
              </span>
            </div>
          </div>

          {{-- Body --}}
          <div class="card-body">

            {{-- Status widget: select (keeps previous behavior but visually separated) --}}
            <div class="mb-3 d-flex align-items-center" id="statusWidget" style="gap:10px;">
              <select id="etatSelect" class="form-control" aria-label="تغيير حالة الطلب" style="max-width:220px; direction:ltr;">
                <option value="en_cours" {{ ($concours_ext->etat ?? '') === 'en_cours' ? 'selected' : '' }}>قيد الدراسة</option>
                <option value="refuse" {{ ($concours_ext->etat ?? '') === 'refuse' ? 'selected' : '' }}>مرفوض</option>
                <option value="acceptation_primaire" {{ ($concours_ext->etat ?? '') === 'acceptation_primaire' ? 'selected' : '' }}>قبول أولي</option>
                <option value="acceptation_definitif" {{ ($concours_ext->etat ?? '') === 'acceptation_definitif' ? 'selected' : '' }}>قبول نهائي</option>
              </select>
            </div>

            {{-- Build unified list of fields (static + dynamic) --}}
            @php
              $staticFields = [
                ['label'=>'الاسم', 'value'=>($concours_ext->nom . ' ' . $concours_ext->prenom)],
                ['label'=>'رقم بطاقة التعريف', 'value'=>$concours_ext->cin],
                ['label'=>'الهاتف', 'value'=>$concours_ext->tel],
                ['label'=>'البريد', 'value'=>$concours_ext->email],
                ['label'=>'العنوان', 'value'=>$concours_ext->adresse],
                ['label'=>'الولاية', 'value'=>$concours_ext->gouvernorat],
                ['label'=>'الرمز البريدي', 'value'=>$concours_ext->code_postale],
                ['label'=>'أنشئ في', 'value'=> optional($concours_ext->created_at)->format('Y-m-d H:i')],
              ];
              $allFields = $staticFields;
              if (!empty($dynamic)) {
                foreach($dynamic as $d) {
                  $allFields[] = ['label' => $d['label'], 'value' => $d['value']];
                }
              }
            @endphp

            {{-- Render fields uniformly --}}
            <div class="fields-grid mb-3">
              @foreach($allFields as $f)
                <div class="field-card">
                  <div class="field-label">{{ $f['label'] }}</div>
                  <div class="field-value">{!! nl2br(e($f['value'] ?? '—')) !!}</div>
                </div>
              @endforeach
            </div>

            {{-- Files --}}
            @if(!empty($files))
              <div class="d-flex justify-content-between align-items-center mb-2">
                <h6 class="mb-0">الملفات المرفوعة</h6>
                <small class="text-muted">{{ count($files) }} ملف</small>
              </div>

              <div class="files-grid">
                @foreach($files as $f)
                  <div class="file-card">
                    <div>
                      <div style="font-weight:700;">{{ $f['original_name'] }}</div>
                      <div class="file-meta">
                        {{ $f['definition_label'] ?? $f['definition_key'] ?? '' }}
                        • {{ $f['mime'] ?? '' }}
                        • {{ number_format($f['size'] ?? 0) }} bytes
                      </div>
                    </div>

                    <div class="d-flex gap-2 mt-2" style="justify-content:flex-start;">
                 <a href="{{ $f['download_url'] }}" class="btn btn-outline-primary btn-sm">تحميل</a>
                  <a href="{{ $f['url'] }}" target="_blank" class="btn btn-outline-secondary btn-sm">فتح</a>


                    </div>
                  </div>
                @endforeach
              </div>
            @endif

          </div>

          {{-- Footer actions: return on left, save on right --}}
          <div class="card-footer concours-footer">
            <div class="left-actions">
              <a href="{{ route('concours_ext.index', ['concours' => $concours->id]) }}" class="btn btn-outline-secondary btn-sm">العودة</a>
            </div>

            <div class="right-actions">
              {{-- fallback form / AJAX target (save) --}}
              <form id="statusFallbackForm" method="POST"
                    action="{{ route('concours_ext.change_state', ['concours' => $concours->id, 'candidate' => $concours_ext->id]) }}"
                    class="d-inline">
                @csrf
                <input type="hidden" name="etat" id="etatInputFallback" value="{{ $concours_ext->etat ?? 'en_cours' }}">
                <button type="submit" class="btn btn-primary btn-sm save-state-btn" id="saveStateFallback">حفظ</button>
              </form>
            </div>
          </div>

        </div> {{-- end card --}}

      </div>
    </div>
  </div>
</section>

<script>
  const csrfToken = "{{ csrf_token() }}";
  const etatSelect = document.getElementById('etatSelect');
  const etatInputFallback = document.getElementById('etatInputFallback');
  const saveBtn = document.getElementById('saveStateFallback');
  const currentBadge = document.getElementById('currentStatusBadge');
  const statusFallbackForm = document.getElementById('statusFallbackForm');

  // When the user changes the native select, update hidden input and enable Save
  etatSelect.addEventListener('change', function() {
    const sel = this.value;
    etatInputFallback.value = sel;
    saveBtn.removeAttribute('disabled');
  });

  // AJAX submit with same robust logic as before
  statusFallbackForm.addEventListener('submit', async function(e){
    e.preventDefault();
    const etat = etatInputFallback.value;
    if (!etat) return;

    saveBtn.setAttribute('disabled', 'disabled');

    try {
      const res = await fetch(this.action, {
        method: 'POST',
        headers: {
          'X-CSRF-TOKEN': csrfToken,
          'Accept': 'application/json',
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({ etat })
      });

      const contentType = res.headers.get('content-type') || '';
      if (res.ok && contentType.includes('application/json')) {
        const data = await res.json();

        // Update badge classes and text using the new etat key (guaranteed by controller)
        const newEtat = data.etat ?? etat;
        currentBadge.className = 'status-badge status-' + newEtat;

        // update text label
        currentBadge.textContent = data.label ?? currentBadge.textContent;

        // little highlight animation
        currentBadge.style.transition = 'background-color .25s, transform .18s';
        currentBadge.style.transform = 'scale(1.03)';
        setTimeout(() => currentBadge.style.transform = '', 180);

        saveBtn.setAttribute('disabled', 'disabled');
        return;
      }

      // fallback: non-json (redirect), submit normally
      this.submit();
    } catch (err) {
      console.error('Error updating state:', err);
      // fallback to full submit on network / parse error
      this.submit();
    } finally {
      saveBtn.removeAttribute('disabled');
    }
  });

  function confirmDelete(e) {
    if (!confirm('هل أنت متأكد من حذف هذا الطلب؟ هذا الإجراء لا يمكن التراجع عنه.')) {
      e && e.preventDefault();
      return false;
    }
    return true;
  }

  // disable Save initially until a change is made
  document.addEventListener('DOMContentLoaded', () => {
    if (saveBtn) saveBtn.setAttribute('disabled', 'disabled');
  });
</script>
@endsection
