<?php

namespace App\Http\Controllers\ManifestationSport;

use Illuminate\Support\Facades\Auth;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use App\Models\Utilisateur;

use App\Models\DemandeManifestation;
use App\Models\TpAttachement;
use App\Models\TpTypeMembre;
use App\Models\TpDomaineEducation;
use App\Models\TpTypeEtrangere;
use App\Models\TeamsAssociation;
use App\Models\OrganismeNonSportif;
use App\Models\EvenementSportif;
use App\Models\MembreOrgInvite;
use App\Models\AssocParticipante;
use App\Models\SportifParticipant;
use App\Models\Etrangere;
use App\Models\TypeFormulaire;
use App\Models\Gouvernorat;
use App\Models\Delegation;
use App\Models\TpStatut;
use App\Models\AthleteIndividuel;
use App\Models\Attachement;
use App\Models\Identite_personnel;
use Illuminate\Support\Facades\Mail;
use App\Mail\StatutDemandeMail;
use App\Notifications\StatutDemandeNotification;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use Mpdf\Mpdf;
use App\Models\ImmobilierStatutHistory;
use Illuminate\Support\Facades\Log;

class ManifestationSportController extends Controller
{
    public function index(Request $request)
    {
        $typesIdentite = Identite_personnel::all();

        $query = DemandeManifestation::with([
            'utilisateur',
            'organismeNonSportif',
            'evenementSportifs.gouvernorat',
            'evenementSportifs.delegation',
            'statut'
        ])
        ->where('type_formulaire_id', 12) 
        ->orderByDesc('id_demande');

        if ($request->filled('date')) {
            $query->whereDate('created_at', $request->date);
        }

        if ($request->filled('type_identite') && $request->filled('cin')) {
            $query->whereHas('utilisateur', function($q) use ($request) {
                $q->where('type_identite_id', $request->type_identite)
                ->where('num_cin', $request->cin);
            });
        }

        $demandes = $query->paginate(10);

        $rows = $demandes->map(function($d) {
            $idPad = str_pad($d->id_demande, 3, '0', STR_PAD_LEFT);

            $cin = $d->utilisateur->num_cin ?? $d->utilisateur->cin ?? null;
            $cinDigits = $cin ? preg_replace('/\D+/', '', $cin) : '';
            $cinPad = $cinDigits ? str_pad($cinDigits, 9, '0', STR_PAD_LEFT) : '000000000';

            $year = $d->created_at ? $d->created_at->format('Y') : now()->format('Y');
            $num_dossier = "{$idPad}-{$cinPad}-{$year}";

            $even = $d->evenementSportifs->first() ?? null;
            $nom_evenement = $even->nom_even ?? null;
            $date_even = $even->date ?? null;
            $heure_even = $even->heure ?? null;

            $gouvernorat = $even?->gouvernorat?->name ?? $even?->gouvernorat?->gouvernorat ?? null;
            $delegation  = $even?->delegation?->name ?? $even?->delegation?->delegation ?? null;

            $nom_organisme = $d->organismeNonSportif ? ($d->organismeNonSportif->nom ?? $d->organismeNonSportif->raison_sociale ?? null) : null;

            return (object) [
                'id_demande' => $d->id_demande,
                'num_dossier' => $num_dossier,
                'nom_organisme' => $nom_organisme,
                'nom_evenement' => $nom_evenement,
                'date_even' => $date_even ? \Carbon\Carbon::parse($date_even)->format('Y-m-d') : null,
                'heure_even' => $heure_even ? \Carbon\Carbon::parse($heure_even)->format('H:i') : null,
                'gouvernorat' => $gouvernorat,
                'delegation' => $delegation,
                'statut' => $d->statut,
                'raw' => $d,
            ];
        });

        return view('backoffice.manifestation_sport.demande.index', compact('demandes', 'rows', 'typesIdentite'));
    }


    private function filteredQuery(Request $request)
{
    $query = DemandeManifestation::with([
        'utilisateur',
        'organismeNonSportif',
        'evenementSportifs.gouvernorat',
        'evenementSportifs.delegation',
        'statut'
    ])->where('type_formulaire_id', 12)
      ->orderByDesc('id_demande');

    if ($request->filled('date')) {
        $query->whereDate('created_at', $request->date);
    }

    if ($request->filled('type_identite') && $request->filled('cin')) {
        $query->whereHas('utilisateur', function($q) use ($request) {
            $q->where('type_identite_id', $request->type_identite)
              ->where('num_cin', $request->cin);
        });
    }

    return $query;
}

public function exportExcel(Request $request)
{
    $demandes = $this->filteredQuery($request)->get();

    $rows = $demandes->map(function ($d) {

        // ===============================
        // 🔹 رقم الملف
        // ===============================
        $idPad = str_pad($d->id_demande, 3, '0', STR_PAD_LEFT);

        $cin = $d->utilisateur->num_cin ?? $d->utilisateur->cin ?? null;
        $cinDigits = $cin ? preg_replace('/\D+/', '', $cin) : '';
        $cinPad = $cinDigits
            ? str_pad($cinDigits, 9, '0', STR_PAD_LEFT)
            : '000000000';

        $year = $d->created_at
            ? $d->created_at->format('Y')
            : now()->format('Y');

        $num_dossier = "{$idPad}-{$cinPad}-{$year}";

        // ===============================
        // 🔹 الحدث الرياضي
        // ===============================
        $even = $d->evenementSportifs->first();

        // ===============================
        // 🔹 الولاية / البلدية (✔ تصحيح)
        // ===============================
        $gouvernorat = $even?->gouvernorat?->nom_ar
            ?? $even?->gouvernorat?->name
            ?? $even?->gouvernorat?->gouvernorat
            ?? '-';

        $delegation = $even?->delegation?->nom_ar
            ?? $even?->delegation?->name
            ?? $even?->delegation?->delegation
            ?? '-';

        // ===============================
        // 🔹 اسم الهيكل
        // ===============================
        $nom_organisme = $d->organismeNonSportif->nom
            ?? $d->organismeNonSportif->raison_sociale
            ?? '-';

        return [
            $num_dossier,
            $nom_organisme,
            $even?->nom_even ?? '-',
            $even?->date
                ? \Carbon\Carbon::parse($even->date)->format('Y-m-d')
                : '-',
            $gouvernorat,
            $delegation,
            $d->statut?->statut_ar ?? 'في الانتظار',
        ];
    });

    // ===============================
    // 🔹 Excel
    // ===============================
    $spreadsheet = new Spreadsheet();
    $sheet = $spreadsheet->getActiveSheet();
    $sheet->setRightToLeft(true);

    $sheet->fromArray([
        'رقم الملف',
        'اسم الهيكل',
        'اسم التظاهرة',
        'التاريخ',
        'الولاية',
        'البلدية',
        'الحالة',
    ], null, 'A1');

    $sheet->getStyle('A1:G1')->getFont()->setBold(true);

    $rowNum = 2;
    foreach ($rows as $row) {
        $sheet->fromArray($row, null, 'A' . $rowNum++);
    }

    foreach (range('A', 'G') as $col) {
        $sheet->getColumnDimension($col)->setAutoSize(true);
    }

    $writer = new Xlsx($spreadsheet);

    return response()->streamDownload(
        fn () => $writer->save('php://output'),
        'manifestation.xlsx'
    );
}
public function exportPdf(Request $request)
{
    $demandes = $this->filteredQuery($request)->get();

    $rows = $demandes->map(function ($d) {

        $idPad = str_pad($d->id_demande, 3, '0', STR_PAD_LEFT);

        $cin = $d->utilisateur->num_cin ?? $d->utilisateur->cin ?? null;
        $cinDigits = $cin ? preg_replace('/\D+/', '', $cin) : '';
        $cinPad = $cinDigits
            ? str_pad($cinDigits, 9, '0', STR_PAD_LEFT)
            : '000000000';

        $year = $d->created_at
            ? $d->created_at->format('Y')
            : now()->format('Y');

        $num_dossier = "{$idPad}-{$cinPad}-{$year}";

        $even = $d->evenementSportifs->first();

        // ✔ تصحيح الولاية / البلدية
        $gouvernorat = $even?->gouvernorat?->nom_ar
            ?? $even?->gouvernorat?->name
            ?? $even?->gouvernorat?->gouvernorat
            ?? '-';

        $delegation = $even?->delegation?->nom_ar
            ?? $even?->delegation?->name
            ?? $even?->delegation?->delegation
            ?? '-';

        $nom_organisme = $d->organismeNonSportif->nom
            ?? $d->organismeNonSportif->raison_sociale
            ?? '-';

        return (object)[
            'num_dossier'   => $num_dossier,
            'nom_organisme' => $nom_organisme,
            'nom_evenement' => $even?->nom_even ?? '-',
            'date_even'     => $even?->date
                ? \Carbon\Carbon::parse($even->date)->format('Y-m-d')
                : '-',
            'heure_even'    => $even?->heure
                ? \Carbon\Carbon::parse($even->heure)->format('H:i')
                : '-',
            'gouvernorat'   => $gouvernorat,
            'delegation'    => $delegation,
            'statut'        => $d->statut,
        ];
    });

    $html = view(
        'backoffice.manifestation_sport.demande.exports.manifestation_pdf',
        compact('rows')
    )->render();

    $mpdf = new \Mpdf\Mpdf([
        'mode' => 'utf-8',
        'format' => 'A4',
        'default_font' => 'dejavusans',
        'directionality' => 'rtl',
    ]);

    $mpdf->WriteHTML($html);

    return response(
        $mpdf->Output('manifestation.pdf', 'S')
    )->header('Content-Type', 'application/pdf');
}


    public function create()
    {
        $formulaire   = TypeFormulaire::find(12);
        $attachments  = TpAttachement::orderBy('type_ar', 'asc')->get();
        $formulaire2  = TypeFormulaire::find(14);
        $formulaire3  = TypeFormulaire::find(15);
        $gouvernorats = Gouvernorat::all();
        $delegations  = Delegation::all();
        $formulaire4  = TypeFormulaire::find(16);
        $type_membre  = TpTypeMembre::all();
        $statut       = TpStatut::all();
        $formulaire5  = TypeFormulaire::find(17);
        $formulaire6  = TypeFormulaire::find(23);
        $formulaire7  = TypeFormulaire::find(18);
        

        return view('backoffice.manifestation_sport.demande.create', compact('formulaire', 'attachments','formulaire2','formulaire3',
            'gouvernorats','delegations','type_membre','statut','formulaire4','formulaire5','formulaire6','formulaire7'));
    }

    public function store(Request $request)
    {
        
        $request->validate([
            //demande
            'sujet' => 'nullable|string|max:255',
            'sport_type' => 'nullable|string|max:255',
            'event_name' => 'nullable|string|max:255',
            'objectif' => 'nullable|string',
            'commentaire_motif' => 'nullable|string',
            'type_formulaire_id' => 'required|integer',
            'user_id' => 'required|integer',
            'id_type' => 'nullable|integer',
            'organisme_non_sportif_id' => 'nullable|integer',
            'role' => 'nullable|string',
            'periode_min' => 'nullable|date',
            'periode_max' => 'nullable|date|after_or_equal:periode_min',
            'lieu' => 'nullable|string|max:255',

            // Organisme non sportif
            'org.nom' => 'required|string|max:255',
            'org.num_visa' => 'nullable|integer',
            'org.date_enregistrement' => 'nullable|date',
            'org.num_enregistrement' => 'nullable|string|max:50',
            'org.identifiant_fiscal' => 'nullable|int',
            'org.num_compte_bancaire' => 'nullable|string|max:50',
            'org.adresse' => 'nullable|string|max:255',
            'org.tel' => 'nullable|string|max:20',
            'org.fax' => 'nullable|string|max:20',
            'org.email' => 'nullable|email|max:255',
            'org.type_formulaire_id' => 'nullable|integer',

            // EvenementSportif
            'nom_even' => 'required|string|max:100',
            'objectifs' => 'nullable|string',
            'contenus' => 'nullable|string',
            'date' => 'required|date',
            'heure' => 'nullable|date_format:H:i',
            'instalation_sportives' => 'nullable|string|max:200',
            'reference' => 'nullable|string|max:100',
            'estimation_evenement' => 'nullable|numeric',
            'cout_total_evenement' => 'nullable|numeric',
            'difference_estimee' => 'nullable|numeric',
            'recompense' => 'nullable|string',
            'moyen_transport' => 'nullable|string|max:200',
            'user_id' => 'required|integer',
            'id_type' => 'nullable|integer',
            'id_formulaire' => 'nullable|integer',
            'gouvernorat_id' => 'required|integer',
            'delegation_id' => 'required|integer',
            'demande_id' => 'nullable|integer',

            // MembreOrgInvite
            'membres.*.nom_prenom' => 'required|string|max:255',
            'membres.*.role' => 'nullable|string',
            'membres.*.nationalite' => 'nullable|string',
            'membres.*.type_membre_id' => 'nullable|integer',

            //MembreOrgInvite2
            'invites.*.nom_prenom' => 'required|string|max:255',
            'invites.*.role' => 'nullable|string',
            'invites.*.nationalite' => 'nullable|string',
            'invites.*.type_membre_id' => 'nullable|integer',

            // AssocParticipante
            'assoc_participantes.*.nom' => 'required|string|max:255',
            'assoc_participantes.*.num_athlete_h' => 'nullable|integer',
            'assoc_participantes.*.num_athlete_f' => 'nullable|integer',
            'assoc_participantes.*.accompagnants' => 'nullable|integer',
            'assoc_participantes.*.total' => 'nullable|integer',

            // AthleteIndividuel
            'athletes.*.nom_prenom_sportif' => 'nullable|string|max:255',
            'athletes.*.num_athlete_h' => 'nullable|integer',
            'athletes.*.num_athlete_f' => 'nullable|integer',
            'athletes.*.accompagnants' => 'nullable|integer',
            'athletes.*.total' => 'nullable|integer',

            // SportifParticipant
            'sportifs.*.nom_prenom' => 'required|string|max:255',
            'sportifs.*.tranch_age' => 'nullable|string|max:50',
            'sportifs.*.date_naissance' => 'nullable|date',
            'sportifs.*.num_cin' => 'nullable|string|max:50',
            'sportifs.*.invitee' => 'nullable|string|max:255',
            'sportifs.*.jeune_id' => 'nullable|integer',

            // Attachements
            'assurance_file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
            'technique_file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
            'accord_file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
            'demande_file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
            
        ]);

        //OrganismeNonSportif
        $organisme = null;
        if ($request->has('org')) {
            $organismeData = $request->input('org');
            $organismeData['type_formulaire_id'] = 14; 
            $organisme = OrganismeNonSportif::create($organismeData);
        }

        // DemandeManifestation
        $demande = DemandeManifestation::create([
            'sujet' => $request->sujet,
            'sport_type' => $request->sport_type,
            'event_name' => $request->event_name,
            'objectif' => $request->objectif,
            'commentaire_motif' => $request->commentaire_motif,
            'type_formulaire_id' => 12,
            'user_id' => Auth::id(),
            'id_type' => 2,
            'organisme_non_sportif_id' => $organisme?->id_org,
            'role' => $request->role,
            'periode_min' => $request->periode_min,
            'periode_max' => $request->periode_max,
            'lieu' => $request->lieu,
        ]);

        

        if ($request->hasFile('assurance_file')) {
            $filePath = $request->file('assurance_file')->store('manifestations/', 'public');

            Attachement::create([
                'id_type_attach' => 2, 
                'file' => $filePath,
                'id_demande' => $demande->id_demande,
            ]);
        }

        if ($request->hasFile('accord_file')) {
            $filePath = $request->file('accord_file')->store('manifestations/', 'public');

            Attachement::create([
                'id_type_attach' => 7, 
                'file' => $filePath,
                'id_demande' => $demande->id_demande,
            ]);
        }

        if ($request->hasFile('technique_file')) {
            $filePath = $request->file('technique_file')->store('manifestations/', 'public');

            Attachement::create([
                'id_type_attach' => 8, 
                'file' => $filePath,
                'id_demande' => $demande->id_demande,
            ]);
        }

        if ($request->hasFile('demande_file')) {
            $filePath = $request->file('demande_file')->store('manifestations/', 'public');

            Attachement::create([
                'id_type_attach' => 9, 
                'file' => $filePath,
                'id_demande' => $demande->id_demande,
            ]);
        }

        // EvenementSportif
        if ($request->has('nom_even')) {
            $evenement = EvenementSportif::create([
                'nom_even' => $request->nom_even,
                'objectifs' => $request->objectifs,
                'contenus' => $request->contenus,
                'date' => $request->date,
                'heure' => $request->heure,
                'instalation_sportives' => $request->instalation_sportives,
                'reference' => $request->reference,
                'estimation_evenement' => $request->estimation_evenement,
                'cout_total_evenement' => $request->cout_total_evenement,
                'difference_estimee' => $request->difference_estimee,
                'recompense' => $request->recompense,
                'moyen_transport' => $request->moyen_transport,
                'user_id' => Auth::id(),
                'id_type' => 2,
                'id_formulaire' => 15,
                'gouvernorat_id' => $request->gouvernorat_id,
                'delegation_id' => $request->delegation_id,
                'demande_id' => $demande->id_demande,
            ]);

            // MembreOrgInvite
            if ($request->has('membres')) {
                foreach ($request->membres as $membre) {
                    MembreOrgInvite::create([
                        'nom_prenom' => $membre['nom_prenom'],
                        'role' => $membre['role'] ?? null,
                        'nationalite' => $membre['nationalite'] ?? null,
                        'event_id' => $evenement->id_even,
                        'type_membre_id' => 1,
                    ]);
                }
            }

            // MembreOrgInvite2
            if ($request->has('invites')) {
                foreach ($request->invites as $invite) {
                    MembreOrgInvite::create([
                        'nom_prenom' => $invite['nom_prenom'],
                        'role' => $invite['role'] ?? null,
                        'nationalite' => $invite['nationalite'] ?? null,
                        'event_id' => $evenement->id_even,
                        'type_membre_id' => 2,
                    ]);
                }
            }

            //AssocParticipante
            if ($request->has('assoc_participantes')) {
                foreach ($request->assoc_participantes as $assoc) {
                    // On calcule le total directement ici
                    $num_h = isset($assoc['num_athlete_h']) ? (int) $assoc['num_athlete_h'] : 0;
                    $num_f = isset($assoc['num_athlete_f']) ? (int) $assoc['num_athlete_f'] : 0;
                    $accomp = isset($assoc['accompagnants']) ? (int) $assoc['accompagnants'] : 0;
                    $total = $num_h + $num_f + $accomp;

                    AssocParticipante::create([
                        'nom' => $assoc['nom'] ?? null,
                        'num_athlete_h' => $num_h,
                        'num_athlete_f' => $num_f,
                        'accompagnants' => $accomp,
                        'total' => $total,
                        'event_id' => $evenement->id_even,
                    ]);
                }
            }

            //AthleteIndividuel
            if ($request->has('athletes')) {
                foreach ($request->athletes as $athlete) {
                    AthleteIndividuel::create([
                        'nom_prenom_sportif' => $athlete['nom_prenom_sportif'],
                        'num_athlete_h' => $athlete['num_athlete_h'] ?? 0,
                        'num_athlete_f' => $athlete['num_athlete_f'] ?? 0,
                        'accompagnants' => $athlete['accompagnants'] ?? 0,
                        'event_id' => $evenement->id_even,
                    ]);
                }
            }

            //SportifParticipant
            if ($request->has('sportifs')) {
                foreach ($request->sportifs as $sportif) {
                    SportifParticipant::create([
                        'nom_prenom' => $sportif['nom_prenom'],
                        'tranch_age' => $sportif['tranch_age'] ?? null,
                        'date_naissance' => $sportif['date_naissance'] ?? null,
                        'num_cin' => $sportif['num_cin'] ?? null,
                        'invitee' => $sportif['invitee'] ?? null,
                        'even_id' => $evenement->id_even,
                        'jeune_id' => $sportif['jeune_id'] ?? null,
                    ]);
                }
            }
        }

        return redirect()->route('admin.manifestation.index')
                    ->with('success', 'تم تسجيل الطلب بنجاح');
    }

    public function show($id)
    {
        $demande = DemandeManifestation::with([
            'organismeNonSportif',
            'evenementSportifs',
            'evenementSportifs.membres',
            'evenementSportifs.invites',
            'evenementSportifs.assocParticipantes',
            'evenementSportifs.athletesIndividuels',
            'evenementSportifs.sportifs',
            'attachements'
        ])->findOrFail($id);

        $org = OrganismeNonSportif::find($id);
        $formulaire   = TypeFormulaire::find(12);
        $attachments  = TpAttachement::orderBy('type_ar', 'asc')->get();
        $formulaire2  = TypeFormulaire::find(14);
        $formulaire3  = TypeFormulaire::find(15);
        $gouvernorats = Gouvernorat::all();
        $delegations  = Delegation::all();
        $formulaire4  = TypeFormulaire::find(16);
        $type_membre  = TpTypeMembre::all();
        $statut       = TpStatut::all();
        $formulaire5  = TypeFormulaire::find(17);
        $formulaire6  = TypeFormulaire::find(23);
        $formulaire7  = TypeFormulaire::find(18);

        return view('backoffice.manifestation_sport.demande.show', compact(
            'demande', 'formulaire', 'attachments','formulaire2','formulaire3',
            'gouvernorats','delegations','type_membre','statut','formulaire4',
            'formulaire5','formulaire6','formulaire7','org'
        ));
    }

    public function edit($id)
    {
        $demande = DemandeManifestation::with([
            'organismeNonSportif',
            'evenementSportifs',
            'evenementSportifs.membres',
            'evenementSportifs.invites',
            'evenementSportifs.assocParticipantes',
            'evenementSportifs.athletesIndividuels',
            'evenementSportifs.sportifs',
            'attachements',
            'statutHistories.fromStatut',
            'statutHistories.toStatut',
            'statutHistories.changer'
        ])->findOrFail($id);

        $org = OrganismeNonSportif::find($id);
        $formulaire   = TypeFormulaire::find(12);
        $attachments  = TpAttachement::orderBy('type_ar', 'asc')->get();
        $formulaire2  = TypeFormulaire::find(14);
        $formulaire3  = TypeFormulaire::find(15);
        $gouvernorats = Gouvernorat::all();
        $delegations  = Delegation::all();
        $formulaire4  = TypeFormulaire::find(16);
        $type_membre  = TpTypeMembre::all();
        $statut       = TpStatut::all();
        $formulaire5  = TypeFormulaire::find(17);
        $formulaire6  = TypeFormulaire::find(23);
        $formulaire7  = TypeFormulaire::find(18);
        $statuts = TpStatut::orderBy('id_statut')->get();
        return view('backoffice.manifestation_sport.demande.edit', compact(
            'demande', 'formulaire', 'attachments','formulaire2','formulaire3',
            'gouvernorats','delegations','type_membre','statut','formulaire4',
            'formulaire5','formulaire6','formulaire7','org','statuts'
        ));
    }

    /**
 * Envoie mail + notification selon le statut
 */
private function envoyerNotificationEtMail(DemandeManifestation $demande, $nouveauStatut)
{
    $date = now()->format('Y-m-d');
    $statut_ar = $nouveauStatut == 1 ? 'قبول' : 'رفض';

    $idPad = str_pad($demande->id_demande, 3, '0', STR_PAD_LEFT);
    $cin = $demande->utilisateur->num_cin ?? $demande->utilisateur->cin ?? null;
    $cinDigits = $cin ? preg_replace('/\D+/', '', $cin) : '';
    $cinPad = $cinDigits ? str_pad($cinDigits, 9, '0', STR_PAD_LEFT) : '000000000';
    $year = $demande->created_at ? $demande->created_at->format('Y') : now()->format('Y');
    $num_dossier = "{$idPad}-{$cinPad}-{$year}";
    $nomUtilisateur = $demande->utilisateur->nom_prenom ?? 'المستخدم';

    if ($nouveauStatut == 1) {
        $messageStatus = "مرحباً <strong>{$nomUtilisateur}</strong>،<br>" .
            "رقم الملف: <strong>{$num_dossier}</strong><br>" .
            "تم قبول طلبكم \" طلب في اقامة تظاهرة رياضية وطنية بالبالد التونسية \" ، بتاريخ: <strong>{$date}</strong>.<br>نهنئكم على هذا القبول.";
    } else {
        $messageStatus = "مرحباً <strong>{$nomUtilisateur}</strong>،<br>" .
            "رقم الملف: <strong>{$num_dossier}</strong><br>" .
            "تم رفض طلبكم \" طلب في اقامة تظاهرة رياضية وطنية بالبالد التونسية \" ، بتاريخ: <strong>{$date}</strong>.<br>" .
            "التعليق: {$demande->commentaire_motif}{$demande->commentaire_motif2}{$demande->commentaire_motif3}.";
    }

    if ($demande->utilisateur?->email) {
        Mail::to($demande->utilisateur->email)->send(
            new \App\Mail\StatutDemandeMail($demande->utilisateur, $num_dossier, $messageStatus)
        );
    }

    if ($demande->utilisateur) {
        $demande->utilisateur->notify(
            new \App\Notifications\StatutDemandeNotification($num_dossier, $messageStatus)
        );
    }
}

    public function update(Request $request, $id)
    {
        $request->validate([
            'sujet' => 'nullable|string|max:255',
            'sport_type' => 'nullable|string|max:255',
            'event_name' => 'nullable|string|max:255',
            'objectif' => 'nullable|string',
            'commentaire_motif' => 'nullable|string',
            'type_formulaire_id' => 'required|integer',
            'statut_id' => 'nullable|integer',
            'statut2_id' => 'nullable|integer',
            'commentaire_motif2' => 'nullable|string|max:255',
            'commentaire_motif3' => 'nullable|string|max:255',
            'file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:4096',
            'id_type' => 'nullable|integer',
            'organisme_non_sportif_id' => 'nullable|integer',
            'role' => 'nullable|string',
            'periode_min' => 'nullable|date',
            'periode_max' => 'nullable|date|after_or_equal:periode_min',
            'lieu' => 'nullable|string|max:255',

            // Organisme non sportif : valider org comme array
            'org' => 'nullable|array',
            'org.nom' => 'nullable|string|max:255',
            'org.num_visa' => 'nullable|integer',
            'org.date_enregistrement' => 'nullable|date',
            'org.num_enregistrement' => 'nullable|string|max:50',
            'org.identifiant_fiscal' => 'nullable|numeric',
            'org.num_compte_bancaire' => 'nullable|string|max:50',
            'org.adresse' => 'nullable|string|max:255',
            'org.tel' => 'nullable|string|max:20',
            'org.fax' => 'nullable|string|max:20',
            'org.email' => 'nullable|email|max:255',
            'org.type_formulaire_id' => 'nullable|integer',

            // EvenementSportif
            'nom_even' => 'required_with:date|string|max:100',
            'objectifs' => 'nullable|string',
            'contenus' => 'nullable|string',
            'date' => 'required_with:nom_even|date',
            'heure' => 'nullable|date_format:H:i',
            'instalation_sportives' => 'nullable|string|max:200',
            'reference' => 'nullable|string|max:100',
            'estimation_evenement' => 'nullable|numeric',
            'cout_total_evenement' => 'nullable|numeric',
            'difference_estimee' => 'nullable|numeric',
            'recompense' => 'nullable|string',
            'moyen_transport' => 'nullable|string|max:200',
            'gouvernorat_id' => 'required_with:nom_even|integer',
            'delegation_id' => 'required_with:nom_even|integer',

            // collections
            'membres.*.nom_prenom' => 'required_with:membres|string|max:255',
            'invites.*.nom_prenom' => 'required_with:invites|string|max:255',
            'assoc_participantes.*.nom' => 'required_with:assoc_participantes|string|max:255',
            'athletes.*.nom_prenom_sportif' => 'nullable|string|max:255',
            'sportifs.*.nom_prenom' => 'required_with:sportifs|string|max:255',

            // attachements
            'assurance_file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
            'technique_file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
            'accord_file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
            'demande_file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
        ]);

        DB::beginTransaction();

        try {
            $demande = DemandeManifestation::findOrFail($id);

            $organisme = null;
            $orgInput = $request->input('org', null);

            if (is_array($orgInput) && count(array_filter($orgInput)) > 0) {
                $orgInput['type_formulaire_id'] = $orgInput['type_formulaire_id'] ?? 14;

                if ($demande->organisme_non_sportif_id) {
                    $organisme = OrganismeNonSportif::find($demande->organisme_non_sportif_id);
                    if ($organisme) {
                        $organisme->fill($orgInput);
                        $organisme->save();
                    } else {
                        $organisme = OrganismeNonSportif::create($orgInput);
                    }
                } else {
                    $organisme = OrganismeNonSportif::create($orgInput);
                }

                $pkName = $organisme->getKeyName(); 
                $demande->organisme_non_sportif_id = $organisme->{$pkName};
            } else {
                
            }

            /*if ($request->hasFile('file')) {
                if (!empty($demande->file) && Storage::disk('public')->exists($demande->file)) {
                    Storage::disk('public')->delete($demande->file);
                }
                $filePath = $request->file('file')->store('manifestations/files', 'public');
            } else {
                $filePath = $demande->file;
            }*/
			
			$filePath = $demande->file; // garder l'ancien par défaut

			if ($request->hasFile('file')) {

				// supprimer ancien fichier s'il existe
				if ($demande->file && Storage::disk('public')->exists($demande->file)) {
					Storage::disk('public')->delete($demande->file);
				}

				// stocker le nouveau
				$filePath = $request->file('file')
							->store('manifestations/files', 'public');
			}


            $ancienStatut = $demande->getOriginal('id_type');
            $ancienStatut1 = $demande->getOriginal('statut_id');
            $ancienStatut2 = $demande->getOriginal('statut2_id');

            $demande->update([
                'sujet' => $request->sujet,
                'sport_type' => $request->sport_type,
                'event_name' => $request->event_name,
                'objectif' => $request->objectif,
                'commentaire_motif' => $request->commentaire_motif,
                'commentaire_motif2' => $request->commentaire_motif2,
                'commentaire_motif3' => $request->commentaire_motif3,
                'type_formulaire_id' => $request->type_formulaire_id ?? $demande->type_formulaire_id,
                //'user_id' => Auth::id(),
                'user_id' => $demande->user_id,
                'id_type' => $request->id_type ?? $demande->id_type,
                //'id_type' => $idType,

                'statut_id' => $request->statut_id ?? $demande->statut_id,
                'statut2_id' => $request->statut2_id ?? $demande->statut2_id,
                'role' => $request->role,
                'periode_min' => $request->periode_min,
                'periode_max' => $request->periode_max,
                'lieu' => $request->lieu,
                'organisme_non_sportif_id' => $demande->organisme_non_sportif_id,
                'file' => $filePath,
            ]);

            
if (
    $demande->id_type != $ancienStatut ||
    $demande->statut_id != $ancienStatut1 ||
    $demande->statut2_id != $ancienStatut2
) {
    // Trouver le bon statut à notifier
 $nouveauStatut = $request->id_type ?? $request->statut_id ?? $request->statut2_id;
$old_id_type    = $demande->getOriginal('id_type');     // id_type (ادارة فرعية)
$old_statut_id  = $demande->getOriginal('statut_id');   // الجامعة
$old_statut2_id = $demande->getOriginal('statut2_id');  // اللجنة

$demande->refresh();

// after you have done $demande->update([...]) and $demande->refresh();
$normalize = function($v) {
    if (is_null($v) || $v === '') return null;
    $i = (int) $v;
    return ($i > 0) ? $i : null;
};

$old_id_type    = $normalize($ancienStatut);   // original values you captured earlier
$old_statut_id  = $normalize($ancienStatut1);
$old_statut2_id = $normalize($ancienStatut2);

$new_id_type    = $normalize($demande->id_type);
$new_statut_id  = $normalize($demande->statut_id);
$new_statut2_id = $normalize($demande->statut2_id);

// helper to check tp_statut existence
$statExists = function($id) {
    if (is_null($id)) return false;
    return DB::table('tp_statut')->where('id_statut', $id)->exists();
};

$commentForHistory = null;
if ($request->filled('commentaire_motif')) {
    $commentForHistory = trim($request->input('commentaire_motif'));
} elseif ($request->filled('commentaire_motif2')) {
    $commentForHistory = trim($request->input('commentaire_motif2'));
} elseif ($request->filled('commentaire_motif3')) {
    $commentForHistory = trim($request->input('commentaire_motif3'));
}

// create a history row for each changed slot (use demande_manifestation_id)
if ($old_id_type !== $new_id_type) {
    ImmobilierStatutHistory::create([
        'demande_manifestation_id' => $demande->id_demande,
        'from_statut_id'           => $old_id_type,
        'to_statut_id'             => $statExists($new_id_type) ? $new_id_type : null,
        'changed_by'               => Auth::id() ?? null,
        'commentaire'              => $commentForHistory,
    ]);
}

if ($old_statut_id !== $new_statut_id) {
    ImmobilierStatutHistory::create([
        'demande_manifestation_id' => $demande->id_demande,
        'from_statut_id'           => $old_statut_id,
        'to_statut_id'             => $statExists($new_statut_id) ? $new_statut_id : null,
        'changed_by'               => Auth::id() ?? null,
        'commentaire'              => $commentForHistory,
    ]);
}

if ($old_statut2_id !== $new_statut2_id) {
    ImmobilierStatutHistory::create([
        'demande_manifestation_id' => $demande->id_demande,
        'from_statut_id'           => $old_statut2_id,
        'to_statut_id'             => $statExists($new_statut2_id) ? $new_statut2_id : null,
        'changed_by'               => Auth::id() ?? null,
        'commentaire'              => $commentForHistory,
    ]);
}


    $this->envoyerNotificationEtMail($demande->fresh(['utilisateur']), $nouveauStatut);
}
 
            $attachmentsMap = [
                'assurance_file' => 2, 
                'accord_file' => 7,    
                'technique_file' => 8,
                'demande_file' => 9,
            ];

            foreach ($attachmentsMap as $inputName => $typeId) {
                if ($request->hasFile($inputName)) {

                    $oldAttachment = Attachement::where('id_demande', $demande->id_demande)
                        ->where('id_type_attach', $typeId)
                        ->first();

                    if ($oldAttachment && Storage::disk('public')->exists($oldAttachment->file)) {
                        Storage::disk('public')->delete($oldAttachment->file);
                    }

                    $path = $request->file($inputName)->store('manifestations/', 'public');

                    if ($oldAttachment) {
                        $oldAttachment->update(['file' => $path]);
                    } else {
                        Attachement::create([
                            'id_demande' => $demande->id_demande,
                            'id_type_attach' => $typeId,
                            'file' => $path,
                        ]);
                    }
                }
            }

            $evenement = EvenementSportif::where('demande_id', $demande->id_demande)->first();

            if ($request->has('nom_even')) {
                $evenData = [
                    'nom_even' => $request->nom_even,
                    'objectifs' => $request->objectifs,
                    'contenus' => $request->contenus,
                    'date' => $request->date,
                    'heure' => $request->heure,
                    'instalation_sportives' => $request->instalation_sportives,
                    'reference' => $request->reference,
                    'estimation_evenement' => $request->estimation_evenement,
                    'cout_total_evenement' => $request->cout_total_evenement,
                    'difference_estimee' => $request->difference_estimee,
                    'recompense' => $request->recompense,
                    'moyen_transport' => $request->moyen_transport,
                    'user_id' => Auth::id(),
                    'id_type' => $request->id_type ?? 2,
                    'id_formulaire' => $request->id_formulaire ?? 15,
                    'gouvernorat_id' => $request->gouvernorat_id,
                    'delegation_id' => $request->delegation_id,
                    'demande_id' => $demande->id_demande,
                ];

                if ($evenement) {
                    $evenement->update($evenData);
                } else {
                    $evenement = EvenementSportif::create($evenData);
                }

                MembreOrgInvite::where('event_id', $evenement->id_even)->delete();
                AssocParticipante::where('event_id', $evenement->id_even)->delete();
                AthleteIndividuel::where('event_id', $evenement->id_even)->delete();
                SportifParticipant::where('even_id', $evenement->id_even)->delete();

                if ($request->has('membres')) {
                    foreach ($request->membres as $membre) {
                        MembreOrgInvite::create([
                            'nom_prenom' => $membre['nom_prenom'],
                            'role' => $membre['role'] ?? null,
                            'nationalite' => $membre['nationalite'] ?? null,
                            'event_id' => $evenement->id_even,
                            'type_membre_id' => 1,
                        ]);
                    }
                }

                if ($request->has('invites')) {
                    foreach ($request->invites as $invite) {
                        MembreOrgInvite::create([
                            'nom_prenom' => $invite['nom_prenom'],
                            'role' => $invite['role'] ?? null,
                            'nationalite' => $invite['nationalite'] ?? null,
                            'event_id' => $evenement->id_even,
                            'type_membre_id' => 2,
                        ]);
                    }
                }

                if ($request->has('assoc_participantes')) {
                    foreach ($request->assoc_participantes as $assoc) {
                        $num_h = isset($assoc['num_athlete_h']) ? (int) $assoc['num_athlete_h'] : 0;
                        $num_f = isset($assoc['num_athlete_f']) ? (int) $assoc['num_athlete_f'] : 0;
                        $accomp = isset($assoc['accompagnants']) ? (int) $assoc['accompagnants'] : 0;
                        $total = $num_h + $num_f + $accomp;

                        AssocParticipante::create([
                            'nom' => $assoc['nom'] ?? null,
                            'num_athlete_h' => $num_h,
                            'num_athlete_f' => $num_f,
                            'accompagnants' => $accomp,
                            'total' => $total,
                            'event_id' => $evenement->id_even,
                        ]);
                    }
                }

                if ($request->has('athletes')) {
                    foreach ($request->athletes as $athlete) {
                        AthleteIndividuel::create([
                            'nom_prenom_sportif' => $athlete['nom_prenom_sportif'],
                            'num_athlete_h' => $athlete['num_athlete_h'] ?? 0,
                            'num_athlete_f' => $athlete['num_athlete_f'] ?? 0,
                            'accompagnants' => $athlete['accompagnants'] ?? 0,
                            'event_id' => $evenement->id_even,
                        ]);
                    }
                }

                if ($request->has('sportifs')) {
                    foreach ($request->sportifs as $sportif) {
                        SportifParticipant::create([
                            'nom_prenom' => $sportif['nom_prenom'],
                            'tranch_age' => $sportif['tranch_age'] ?? null,
                            'date_naissance' => $sportif['date_naissance'] ?? null,
                            'num_cin' => $sportif['num_cin'] ?? null,
                            'invitee' => $sportif['invitee'] ?? null,
                            'even_id' => $evenement->id_even,
                            'jeune_id' => $sportif['jeune_id'] ?? null,
                        ]);
                    }
                }
            }
            DB::commit();

        session()->forget('_old_input');
            return redirect()->route('admin.manifestation.index', $demande->id_demande)
                            ->with('success', 'تم تحديث الطلب بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->withInput()->with('error', 'حدث خطأ أثناء التحديث: ' . $e->getMessage());
        }
    }

    public function destroy($id)
    {
        $demande = DemandeManifestation::findOrFail($id);

        foreach ($demande->attachements as $attachement) {
            if ($attachement->chemin && Storage::exists($attachement->chemin)) {
                Storage::delete($attachement->chemin);
            }
            $attachement->delete();
        }

        if ($demande->teamsAssociations) {
            foreach ($demande->teamsAssociations as $team) {
                $team->delete();
            }
        }

        if ($demande->evenementSportif) {
            $demande->evenementSportif->delete();
        }

        if ($demande->organismeNonSportif) {
            $demande->organismeNonSportif->delete();
        }

        $demande->delete();

        return redirect()->route('admin.manifestation.index')->with('success', 'تم حذف الطلب بنجاح.');
    }


    public function toggleStatut(Request $request, $id)
    {
        $demande = DemandeManifestation::with(['utilisateur', 'evenementSportifs', 'organismeNonSportif'])->findOrFail($id);

        $nouveauStatut = $request->input('id_statut') ?? $request->input('statut_id') ?? $request->input('statut2_id');

        if ($request->has('id_statut')) {
            $demande->id_type = $request->id_statut;
            if ($request->filled('commentaire_motif')) {
                $demande->commentaire_motif = $request->commentaire_motif;
            }
        }

        if ($request->has('statut_id')) {
            $demande->statut_id = $request->statut_id;
            if ($request->filled('commentaire_motif2')) {
                $demande->commentaire_motif2 = $request->commentaire_motif2;
            }
        }

        if ($request->has('statut2_id')) {
            $demande->statut2_id = $request->statut2_id;
            if ($request->filled('commentaire_motif3')) {
                $demande->commentaire_motif3 = $request->commentaire_motif3;
            }
        }

        $date = now()->format('Y-m-d');
        $statut_ar = $nouveauStatut == 1 ? 'قبول' : 'رفض';

        $idPad = str_pad($demande->id_demande, 3, '0', STR_PAD_LEFT);
        $cin = $demande->utilisateur->num_cin ?? $demande->utilisateur->cin ?? null;
        $cinDigits = $cin ? preg_replace('/\D+/', '', $cin) : '';
        $cinPad = $cinDigits ? str_pad($cinDigits, 9, '0', STR_PAD_LEFT) : '000000000';
        $year = $demande->created_at ? $demande->created_at->format('Y') : now()->format('Y');
        $num_dossier = "{$idPad}-{$cinPad}-{$year}";
        $nomUtilisateur = $demande->utilisateur->nom_prenom ?? 'المستخدم';

        if ($nouveauStatut == 1) {
            $messageStatus = "مرحباً <strong>{$nomUtilisateur}</strong>،<br>" .
                                "رقم الملف: <strong>{$num_dossier}</strong><br>" .
                                "تم قبول طلبكم \" طلب في اقامة تظاهرة رياضية وطنية بالبالد التونسية \" ، بتاريخ: <strong>{$date}</strong>.<br>نهنئكم على هذا القبول.";
        } else {
            $messageStatus = "مرحباً <strong>{$nomUtilisateur}</strong>،<br>" .
                            "رقم الملف: <strong>{$num_dossier}</strong><br>" .
                            "تم رفض طلبكم \" طلب في اقامة تظاهرة رياضية وطنية بالبالد التونسية \" ، بتاريخ: <strong>{$date}</strong>.<br>" .
                            "التعليق: {$demande->commentaire_motif}{$demande->commentaire_motif2}{$demande->commentaire_motif3}.";
        }

        if ($request->hasFile('file')) {
        $file = $request->file('file');
        $filename = time() . '_' . $file->getClientOriginalName();
        $path = $file->storeAs('public/files', $filename);
        $demande->file = $filename;
    }

        $demande->save();

        if ($demande->utilisateur?->email) {
            Mail::to($demande->utilisateur->email)->send(
                new StatutDemandeMail($demande->utilisateur, $num_dossier, $messageStatus)
            );
        }

        if ($demande->utilisateur) {
            $demande->utilisateur->notify(
                new \App\Notifications\StatutDemandeNotification($num_dossier, $messageStatus)
            );
        }

        return redirect()->back()->with('success', "تم تغيير الحالة إلى $statut_ar");
    }
public function getDelegations_manifestation($gouvernoratId)    {        $delegations = Delegation::where('gouver_id', $gouvernoratId)->get();        return response()->json($delegations);    }

}