<?php
namespace App\Http\Controllers\Investissement;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;
use App\Models\DocumentsFormulaire;
use App\Models\Utilisateur;
use App\Models\TypeFormulaire;
use App\Models\Identite_personnel;
use App\Models\TpStatut;

use App\Mail\DocumentStatusMail;
use App\Notifications\DocumentStatusNotification;
use Illuminate\Support\Facades\Notification;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use Barryvdh\DomPDF\Facade\Pdf;
use Mpdf\Mpdf;
use App\Models\ImmobilierStatutHistory;
use Illuminate\Support\Facades\Auth;

class ExonarationController extends Controller
{
    public function index(Request $request)
{
    $query = DocumentsFormulaire::with(['utilisateur', 'statut']);

    if ($request->filled('search_option') && $request->search_option === 'cahier' && $request->filled('id_type_formulaire')) {
        $query->where('id_type_formulaire', $request->id_type_formulaire);
    }

    if ($request->filled('search_option') && $request->search_option === 'date' && $request->filled('date')) {
        $query->whereDate('created_at', $request->date);
    }

    if ($request->filled('search_option') && $request->search_option === 'identite' && $request->filled('type_identite')) {
        $type = $request->type_identite;

        if ($type == 1 && $request->filled('cin')) {
            $query->whereHas('utilisateur', fn($q) => $q->where('cin', $request->cin));
        }
        if ($type == 2 && $request->filled('passport')) {
            $query->whereHas('utilisateur', fn($q) => $q->where('passport', $request->passport));
        }
        if ($type == 3 && $request->filled('num_fiscal')) {
            $query->whereHas('utilisateur', fn($q) => $q->where('num_fiscal', $request->num_fiscal));
        }
    }

    $documents = $query->latest()->paginate(10)->withQueryString();

    $typesFormulaires = TypeFormulaire::whereIn('id_type', [1,3,4,21,5])->get();

    return view('backoffice.investissement.gestion_exoneration.index', compact('documents', 'typesFormulaires'));
}


private function filteredQuery(Request $request)
{
    $query = DocumentsFormulaire::with(['utilisateur', 'statut']);

    if ($request->search_option === 'cahier' && $request->filled('id_type_formulaire')) {
        $query->where('id_type_formulaire', $request->id_type_formulaire);
    }

    if ($request->search_option === 'date' && $request->filled('date')) {
        $query->whereDate('created_at', $request->date);
    }

    if ($request->search_option === 'identite' && $request->filled('type_identite')) {
        if ($request->type_identite == 1 && $request->filled('cin')) {
            $query->whereHas('utilisateur', fn($q) => $q->where('cin', $request->cin));
        }

        if ($request->type_identite == 2 && $request->filled('passport')) {
            $query->whereHas('utilisateur', fn($q) => $q->where('passport', $request->passport));
        }

        if ($request->type_identite == 3 && $request->filled('num_fiscal')) {
            $query->whereHas('utilisateur', fn($q) => $q->where('num_fiscal', $request->num_fiscal));
        }
    }

    return $query;
}


public function exportPdf(Request $request)
{
    $documents = $this->filteredQuery($request)
        ->latest()
        ->get();

    $html = view(
        'backoffice.investissement.gestion_exoneration.exports.exoneration_pdf',
        compact('documents')
    )->render();

    $mpdf = new Mpdf([
        'mode' => 'utf-8',
        'format' => 'A4',
        'default_font' => 'dejavusans',
        'directionality' => 'rtl',
    ]);

    $mpdf->WriteHTML($html);

    return response($mpdf->Output('exoneration.pdf', 'S'))
        ->header('Content-Type', 'application/pdf');
}

public function exportExcel(Request $request)
{
    $documents = $this->filteredQuery($request)
        ->latest()
        ->get();

    $spreadsheet = new Spreadsheet();
    $sheet = $spreadsheet->getActiveSheet();
    $sheet->setRightToLeft(true);

    // Header
    $sheet->fromArray([
        'المعرف',
        'الاسم و اللقب',
        'اسم المشروع',
        'مكان المشروع',
        'وضعية المشروع',
    ], null, 'A1');

    $sheet->getStyle('A1:E1')->getFont()->setBold(true);

    // Data
    $row = 2;
    foreach ($documents as $doc) {
        $sheet->setCellValue('A'.$row, $doc->id_doc);
        $sheet->setCellValue('B'.$row, $doc->utilisateur?->nom_prenom ?? '---');
        $sheet->setCellValue('C'.$row, $doc->nom_projet);
        $sheet->setCellValue('D'.$row, $doc->adresse_projet);
        $sheet->setCellValue('E'.$row, $doc->statut?->statut_ar ?? 'في الانتظار');
        $row++;
    }

    foreach (range('A', 'E') as $col) {
        $sheet->getColumnDimension($col)->setAutoSize(true);
    }

    $writer = new Xlsx($spreadsheet);

    return response()->streamDownload(
        fn () => $writer->save('php://output'),
        'exoneration.xlsx'
    );
}


    public function store(Request $request)
    {
        DB::beginTransaction();
        try {
            $request->validate([
                // Utilisateur
                'nom_prenom'       => 'required|string|max:255',
                'id_type_identite' => 'required|integer|exists:tp_type_identite_pers,id_type',
                'cin'              => 'nullable|string|max:20',
                'passport'         => 'nullable|string|max:50',
                'num_fiscal'       => 'nullable|string|max:50',
                'tel'              => 'nullable|string|max:20',
                'adresse'          => 'nullable|string|max:255',
                'email'            => 'required|email|unique:utilisateur,email',
                'date_naissance'   => 'nullable|date',
                'lieu_naissance'   => 'nullable|string|max:100',                

                // Documents
                'nom_projet'       => 'required|string|max:255',
                'adresse_projet'   => 'required|string|max:255',
                'copie_cahier_des_charges'           => 'nullable|file|mimes:pdf,jpg,jpeg,png',
                'copie_originale'                    => 'nullable|file|mimes:pdf,jpg,jpeg,png',
                'facture'                            => 'nullable|file|mimes:pdf,jpg,jpeg,png',
                'contrat'                            => 'nullable|file|mimes:pdf,jpg,jpeg,png',
                'fichier_arrivage'                   => 'nullable|file|mimes:pdf,jpg,jpeg,png',
                'fichier_colisage'                   => 'nullable|file|mimes:pdf,jpg,jpeg,png',
                'copie_registre_fiscal'              => 'nullable|file|mimes:pdf,jpg,jpeg,png',
                'copie_registre_national_institutions' => 'nullable|file|mimes:pdf,jpg,jpeg,png',
                'copie_certificat_propriete'         => 'nullable|file|mimes:pdf,jpg,jpeg,png',
                'engagement'                         => 'nullable|file|mimes:pdf,jpg,jpeg,png',
                'exemple_projet'                     => 'nullable|file|mimes:pdf,jpg,jpeg,png',
            ]);

            $utilisateur = Utilisateur::create([
                'nom_prenom'          => $request->nom_prenom,
                'id_type_identite'    => $request->id_type_identite,
                'cin'                 => $request->cin,
                'passport'            => $request->passport,
                'num_fiscal'          => $request->num_fiscal,
                'adresse'             => $request->adresse,
                'tel'                 => $request->tel,
                'email'               => $request->email,
                'date_naissance'      => $request->date_naissance,
                'lieu_naissance'      => $request->lieu_naissance,
            ]);

            $files = [
                'copie_cahier_des_charges',
                'copie_originale',
                'facture',
                'contrat',
                'fichier_arrivage',
                'fichier_colisage',
                'copie_registre_fiscal',
                'copie_registre_national_institutions',
                'copie_certificat_propriete',
                'engagement',
                'exemple_projet'
            ];

            $fileData = [];
            foreach ($files as $fileField) {
                if ($request->hasFile($fileField)) {
                    $path = $request->file($fileField)->store('documents/exoneration', 'public');
                    $fileData[$fileField] = $path;
                }
            }

            DocumentsFormulaire::create(array_merge($fileData, [
                'id_user'      => $utilisateur->id,
                'id_formulaire'=> $request->id_formulaire,
                'nom_projet'   => $request->nom_projet,
                'adresse_projet'=> $request->adresse_projet,
                'id_statut'    => 2,
            ]));

           DB::commit();
            return redirect()->route('admin.exoneration.index')
                            ->with('success', 'تم تسجيل المشروع بنجاح');

            } catch (\Exception $e) {
                DB::rollBack();
                return back()->withErrors(['error' => 'حدث خطأ أثناء التسجيل: '.$e->getMessage()])
                            ->withInput();
        }
    }

    public function edit($id)
    {
        $document = DocumentsFormulaire::with('utilisateur')->findOrFail($id);
        $typesIdentite = Identite_personnel::all();
        $formulaire = TypeFormulaire::find($document->id_formulaire);
        $statuts = TpStatut::orderBy('id_statut')->get();
        return view('backoffice.investissement.gestion_exoneration.edit', compact(
            'document','typesIdentite','formulaire', 'statuts'
        ));
    }

    public function update(Request $request, $id)
    {
        DB::beginTransaction();
        try {
            $document = DocumentsFormulaire::with('utilisateur')->findOrFail($id);
            $utilisateur = $document->utilisateur;

            $request->validate([
                // Utilisateur
                'nom_prenom'       => 'required|string|max:255',
                'id_type_identite' => 'required|integer|exists:tp_type_identite_pers,id_type',
                'cin'              => 'nullable|string|max:20',
                'passport'         => 'nullable|string|max:50',
                'num_fiscal'       => 'nullable|string|max:50',
                'tel'              => 'nullable|string|max:20',
                'adresse'          => 'nullable|string|max:255',
                'email'            => 'required|email|unique:utilisateur,email,' . $utilisateur->id,
                'date_naissance'   => 'nullable|date',
                'lieu_naissance'   => 'nullable|string|max:100',

                // Documents
                'nom_projet'       => 'required|string|max:255',
                'adresse_projet'   => 'required|string|max:255',
                'id_statut'        => 'required|integer',
                'commentaire_motif'        => 'nullable|string',
                'commentaire_acceptation'  => 'nullable|string',
                'commentaire_avis'         => 'nullable|string',
                'copie_cahier_des_charges'           => 'nullable|file|mimes:pdf,jpg,jpeg,png',
                'copie_originale'                    => 'nullable|file|mimes:pdf,jpg,jpeg,png',
                'facture'                            => 'nullable|file|mimes:pdf,jpg,jpeg,png',
                'contrat'                            => 'nullable|file|mimes:pdf,jpg,jpeg,png',
                'fichier_arrivage'                   => 'nullable|file|mimes:pdf,jpg,jpeg,png',
                'fichier_colisage'                   => 'nullable|file|mimes:pdf,jpg,jpeg,png',
                'copie_registre_fiscal'              => 'nullable|file|mimes:pdf,jpg,jpeg,png',
                'copie_registre_national_institutions' => 'nullable|file|mimes:pdf,jpg,jpeg,png',
                'copie_certificat_propriete'         => 'nullable|file|mimes:pdf,jpg,jpeg,png',
                'engagement'                         => 'nullable|file|mimes:pdf,jpg,jpeg,png',
                'exemple_projet'                     => 'nullable|file|mimes:pdf,jpg,jpeg,png',
            ]);

            $utilisateur->update([
                'nom_prenom'       => $request->nom_prenom,
                'id_type_identite' => $request->id_type_identite,
                'cin'              => $request->cin,
                'passport'         => $request->passport,
                'num_fiscal'       => $request->num_fiscal,
                'adresse'          => $request->adresse,
                'tel'              => $request->tel,
                'email'            => $request->email,
                'date_naissance'   => $request->date_naissance,
                'lieu_naissance'   => $request->lieu_naissance,
            ]);

            $files = [
                'copie_cahier_des_charges',
                'copie_originale',
                'facture',
                'contrat',
                'fichier_arrivage',
                'fichier_colisage',
                'copie_registre_fiscal',
                'copie_registre_national_institutions',
                'copie_certificat_propriete',
                'engagement',
                'exemple_projet'
            ];

            foreach ($files as $fileField) {
                if ($request->hasFile($fileField)) {
                    $path = $request->file($fileField)->store('documents/exoneration', 'public');
                    $document->$fileField = $path;
                }
            }

          

            if ($request->id_statut != 2 && $utilisateur && !empty($utilisateur->email)) {
                $messageStatus = '';

                switch ($request->id_statut) {
                    case 1:
                        $messageStatus = " تم <b>قبول</b> طلبكم بنجاح. نهنئكم على هذا القبول.";
                        break;

                    case 3:
                        $messageStatus = " تم <b>رفض</b> طلبكم.<br>السبب: <b>" . e($request->commentaire_motif) . "</b>";
                        break;

                    case 4:
                        $messageStatus = " طلبكم <b>غير مكتمل</b>.<br>يرجى استكمال النواقص التالية: <b>" . e($request->commentaire_acceptation) . "</b>";
                        break;

                    case 5:
                        $messageStatus = " تم <b>إرجاء طلبكم لإبداء الرأي</b>.<br>الملاحظة: <b>" . e($request->commentaire_avis) . "</b>";
                        break;
                }

            //Mail::to($utilisateur->email)->send(new DocumentStatusMail($utilisateur, $request->id_statut, $messageStatus));
$oldStatus = (int) ($document->id_statut ?? 0);
$newStatus = (int) $request->input('id_statut', $oldStatus);

// choisir le commentaire selon le type de statut
$commentRefus   = $request->input('commentaire_motif');
$commentAccept  = $request->input('commentaire_acceptation');
$commentAvis    = $request->input('commentaire_avis');

     $comment = null;
        if ($request->filled('commentaire_refus')) {
            $comment = trim($request->input('commentaire_refus'));
        } elseif ($request->filled('commentaire_status')) {
            $comment = trim($request->input('commentaire_status'));
        } elseif ($request->filled('commentaire_motif')) {
            $comment = trim($request->input('commentaire_motif'));
        }
        if ($comment === '') $comment = null;
		
$document->fill([
    'nom_projet'              => $request->nom_projet,
    'adresse_projet'          => $request->adresse_projet,
    'id_statut'               => $request->id_statut,
    'commentaire_motif'       => $request->commentaire_motif,
    'commentaire_acceptation' => $request->commentaire_acceptation,
    'commentaire_avis'        => $request->commentaire_avis,
    'etat_formulaire'         => $request->id_statut != 2 ? 'reponse' : 'remplir',
])->save(); 	


// si le statut a changé, créer l'historique AVANT (ou APRÈS) la mise à jour
if ($oldStatus !== $newStatus) {
    ImmobilierStatutHistory::create([
        'documents_formulaire_id' => $document->id_doc,
        'from_statut_id'          => $oldStatus ?: null,
        'to_statut_id'            => $newStatus,
        'changed_by'              => Auth::id() ?? null,
        'commentaire'             => $comment,
    ]);
}
// ensuite tu peux mettre à jour le document (ou faire la création d'historique après en utilisant getOriginal())
$document->update([
    'nom_projet'    => $request->nom_projet,
    'adresse_projet'=> $request->adresse_projet,
    'id_statut'     => $request->id_statut,
    'commentaire_motif'      => $request->commentaire_motif,
    'commentaire_acceptation'=> $request->commentaire_acceptation,
    'commentaire_avis'       => $request->commentaire_avis,
    'etat_formulaire' => $request->id_statut != 2 ? 'reponse' : 'remplir',
]);


            Mail::to($utilisateur->email)->send(
                new DocumentStatusMail($utilisateur, $request->id_statut, $messageStatus, $document->num_dossier)
            );

            //$utilisateur->notify(new DocumentStatusNotification(strip_tags($messageStatus)));
            $utilisateur->notify(new DocumentStatusNotification(
                    strip_tags($messageStatus),
                    $document->num_dossier
                ));
        }

            DB::commit();

            return redirect()->route('admin.exoneration.index')
                            ->with('success', 'تم تحديث المشروع بنجاح ');

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->withErrors(['error' => 'حدث خطأ أثناء التحديث: ' . $e->getMessage()])
                        ->withInput();
        }
    }

    public function show($id)
    {
        $document = DocumentsFormulaire::with('utilisateur')->findOrFail($id);
        $formulaire = TypeFormulaire::find($document->id_formulaire);
        $statuts = TpStatut::orderBy('id_statut')->get();
        return view('backoffice.investissement.gestion_exoneration.show', compact('document','formulaire','statuts'));
    }


    public function destroy($id)
    {
        DB::beginTransaction();
        try {
            $document = DocumentsFormulaire::findOrFail($id);

            $files = [
                'copie_cahier_des_charges','copie_originale','facture','contrat','fichier_arrivage','fichier_colisage',
                'copie_registre_fiscal','copie_registre_national_institutions','copie_certificat_propriete','engagement','exemple_projet'
            ];

            foreach ($files as $file) {
                if (!empty($document->$file) && \Storage::disk('public')->exists($document->$file)) {
                    \Storage::disk('public')->delete($document->$file);
                }
            }

            $document->utilisateur()->delete();

            $document->delete();

            DB::commit();
            return redirect()->route('admin.exoneration.index')
                            ->with('success', 'تم حذف المشروع بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->withErrors(['error' => 'حدث خطأ أثناء الحذف: '.$e->getMessage()]);
        }
    }

    public function toggleStatut(Request $request, $id)
    {
        $request->validate([
            'id_statut' => 'required|integer|in:1,3,4,5',
            'commentaire_motif' => 'nullable|string',
            'commentaire_acceptation' => 'nullable|string',
            'commentaire_avis' => 'nullable|string',
        ]);

        DB::beginTransaction();
        try {
            $document = DocumentsFormulaire::findOrFail($id);

            $document->id_statut = $request->id_statut;

            if ($request->id_statut == 3) {
                $document->commentaire_motif = $request->commentaire_motif;
                $document->commentaire_acceptation = null;
                $document->commentaire_avis = null;
            } elseif ($request->id_statut == 4) {
                $document->commentaire_acceptation = $request->commentaire_acceptation;
                $document->commentaire_motif = null;
                $document->commentaire_avis = null;
            } elseif ($request->id_statut == 5) {
                $document->commentaire_avis = $request->commentaire_avis;
                $document->commentaire_motif = null;
                $document->commentaire_acceptation = null;
            } else {
                $document->commentaire_motif = null;
                $document->commentaire_acceptation = null;
                $document->commentaire_avis = null;
            }

            if ($document->id_statut != 2) {
                $document->etat_formulaire = 'reponse';
            }

            $document->save();
            DB::commit();

            $utilisateur = Utilisateur::find($document->id_user); 

            if ($utilisateur && !empty($utilisateur->email)) {
                $messageStatus = '';

                switch ($request->id_statut) {
                case 1:
                    $messageStatus = " تم <b>قبول</b> طلبكم بنجاح. نهنئكم على هذا القبول.";
                    break;

                case 3:
                    $messageStatus = " تم <b>رفض</b> طلبكم.<br>السبب: <b>" . e($request->commentaire_motif) . "</b>";
                    break;

                case 4:
                    $messageStatus = " طلبكم <b>غير مكتمل</b>.<br>يرجى استكمال النواقص التالية: <b>" . e($request->commentaire_acceptation) . "</b>";
                    break;

                case 5:
                    $messageStatus = " تم <b>إرجاء طلبكم لإبداء الرأي</b>.<br>الملاحظة: <b>" . e($request->commentaire_avis) . "</b>";
                    break;
            }


                //Mail::to($utilisateur->email)->send(new DocumentStatusMail($utilisateur, $request->id_statut, $messageStatus));

                Mail::to($utilisateur->email)->send(
                    new DocumentStatusMail($utilisateur, $request->id_statut, $messageStatus, $document->num_dossier)
                );

                //$utilisateur->notify(new DocumentStatusNotification(strip_tags($messageStatus)));
                $utilisateur->notify(new DocumentStatusNotification(
                    strip_tags($messageStatus),
                    $document->num_dossier
                ));
            }


            return redirect()->back()->with('success', 'تم تغيير الوضعية بنجاح');

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->withErrors(['error' => 'حدث خطأ أثناء تغيير الوضعية: '.$e->getMessage()]);
        }
    }

}
