<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use App\Models\Type;
use App\Models\Guide;
use App\Models\Builder;
class GestionFrontController extends Controller
{
    public function index(Request $request)
{
    $search = trim($request->get('search'));

    $words = $search ? preg_split('/\s+/', $search) : [];

    $guides = Guide::with('children')
        ->when($words, function ($query) use ($words) {
            foreach ($words as $word) {
               
                $word = str_replace(['\\', '%', '_'], ['\\\\', '\\%', '\\_'], $word);

                $query->where(function ($q) use ($word) {
                    $q->where('titre_G', 'LIKE', "%{$word}%")
                      ->orWhere('titre_fr', 'LIKE', "%{$word}%")
                      ->orWhere('titre_en', 'LIKE', "%{$word}%");
                });
            }
        })
        ->orderBy('ordre', 'asc')
        ->paginate(10)
        ->appends(['search' => $search]); // Important pour pagination

    return view('backoffice.gestion_guide.listeguides', compact('guides'));
}



    public function create()
    {
        $guides = Guide::orderBy('ordre', 'asc')->get();
        $parentGuides = $this->buildGuideTree($guides);
        $builders = Builder::orderBy('titre_ar')->get();
        return view('backoffice.gestion_guide.create', compact('parentGuides','builders'));
    }

    private function buildGuideTree($guides, $parentId = null, $prefix = '')
    {
        $tree = [];
        foreach ($guides->where('id_parent', $parentId) as $guide) {
            $tree[] = [
                'id_G' => $guide->id_G,
                'titre_G' => $prefix . $guide->titre_G
            ];
            $children = $this->buildGuideTree($guides, $guide->id_G, $prefix . '--');
            $tree = array_merge($tree, $children);
        }
        return $tree;
    }

    public function store(Request $request)
    {
        $request->merge([
            'type_contenu' => $request->input('type_contenu_menu') ?? $request->input('type_contenu_sous'),
            'lien' => $request->input('lien_menu') ?? $request->input('lien_sous'),
            'ordre' => $request->input('ordre') ?? $request->input('ordre_sous'),
            'builder_id' => $request->input('builder_id_menu') ?? $request->input('builder_id_sous'),
        ]);

        $validator = Validator::make($request->all(), [
            'titre_G' => 'required|string|max:255',
            'titre_fr' => 'required|string|max:255',
            'titre_en' => 'required|string|max:255',
            'ordre' => 'required|integer',
            'type_contenu' => 'required|in:none,lien,builder',
            'id_type' => 'nullable|integer|exists:tp_type,id_type',
            'id_parent' => 'nullable|integer|exists:guides,id_G',
            'type' => 'required|in:menu_principal,sous_menu',
            'lien' => 'nullable|string|max:2050',
            'open_new_tab' => 'nullable|boolean',
            'builder_id' => 'nullable|integer|exists:builder,id_builder',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        $data = $validator->validated();

        if ($data['type_contenu'] !== 'builder') {
            $data['builder_id'] = null;
        }
        if ($data['type_contenu'] !== 'lien') {
            $data['lien'] = null;
            $data['open_new_tab'] = 0;
        }
        Guide::create($data);

        return redirect()->route('guides.index')->with('success', 'تمت إضافة الدليل بنجاح!');
    }

    public function edit($id)
    {
        $guide = Guide::findOrFail($id);
        $allGuides = Guide::where('id_G', '!=', $guide->id_G)
                        ->orderBy('ordre', 'asc')
                        ->get();
        $parentGuides = $this->buildGuideTree($allGuides);
		$builders = Builder::orderBy('titre_ar')->get();

        return view('backoffice.gestion_guide.edit', compact('guide', 'builders','parentGuides'));
    }

    /*public function update(Request $request, $id)
{
    $guide = Guide::findOrFail($id);

    $request->merge([
        'type_contenu' => $request->input('type_contenu_menu') ?? $request->input('type_contenu_sous'),
        'ordre' => $request->input('ordre_menu') ?? $request->input('ordre_sous'),
        'lien' => $request->input('lien_menu') ?? $request->input('lien_sous'),
        'open_new_tab' => $request->has('open_new_tab') ? 1 : 0,
        'builder_id' => $request->input('builder_id_menu') ?? $request->input('builder_id_sous'),
    ]);

    $validator = Validator::make($request->all(), [
        'titre_G' => 'required|string|max:255',
        'titre_fr' => 'required|string|max:255',
        'titre_en' => 'required|string|max:255',
        'ordre' => 'required|integer',
        'type' => 'required|in:menu_principal,sous_menu',
        'type_contenu' => 'required|in:none,lien,builder',
        'id_type' => 'nullable|integer|exists:tp_type,id_type',
        'id_parent' => 'nullable|integer|exists:guides,id_G',
        'lien' => 'nullable|string|max:2050',
        'open_new_tab' => 'nullable|boolean',
        'builder_id' => 'nullable|integer|exists:builder,id_builder',
    ]);

    if ($validator->fails()) {
        return redirect()->back()->withErrors($validator)->withInput();
    }

    $data = $validator->validated();

    // Si ce n'est pas un builder, on enlève builder_id
    if (($data['type_contenu'] ?? null) !== 'builder') {
        $data['builder_id'] = null;
    }

    // Si ce n'est pas un lien, on enlève lien et on force open_new_tab à 0
    if (($data['type_contenu'] ?? null) !== 'lien') {
        $data['lien'] = null;
        $data['open_new_tab'] = 0;
    }

    $guide->update($data);

    return redirect()->route('guides.index')->with('success', 'تم تحديث الدليل بنجاح!');
}*/

public function update(Request $request, $id)
{
    $guide = Guide::findOrFail($id);

    // Déterminer le champ source selon type
    $type = $request->input('type');

    $type_contenu = $type === 'menu_principal'
        ? $request->input('type_contenu_menu')
        : $request->input('type_contenu_sous');

    $ordre = $type === 'menu_principal'
        ? $request->input('ordre_menu')
        : $request->input('ordre_sous');

    $lien = $type_contenu === "lien"
        ? ($type === 'menu_principal' ? $request->input('lien_menu') : $request->input('lien_sous'))
        : null;

    $open_new_tab = $type === 'menu_principal'
        ? $request->boolean('open_new_tab_menu')
        : $request->boolean('open_new_tab_sous');

    $builder_id = $type_contenu === "builder"
        ? ($type === 'menu_principal' ? $request->input('builder_id_menu') : $request->input('builder_id_sous'))
        : null;

    // Validation
    $validator = Validator::make([
        'titre_G' => $request->titre_G,
        'titre_fr' => $request->titre_fr,
        'titre_en' => $request->titre_en,
        'type' => $type,
        'type_contenu' => $type_contenu,
        'ordre' => $ordre,
        'lien' => $lien,
        'open_new_tab' => $open_new_tab,
        'builder_id' => $builder_id,
        'id_parent' => $request->id_parent,
        'id_type' => $request->id_type
    ], [
        'titre_G' => 'required|string|max:255',
        'titre_fr' => 'required|string|max:255',
        'titre_en' => 'required|string|max:255',
        'type' => 'required|in:menu_principal,sous_menu',
        'type_contenu' => 'required|in:none,lien,builder',
        'ordre' => 'required|integer',
        'lien' => 'nullable|string|max:2050',
        'open_new_tab' => 'boolean',
        'builder_id' => 'nullable|integer|exists:builder,id_builder',
        'id_parent' => 'nullable|integer|exists:guides,id_G',
        'id_type' => 'nullable|integer|exists:tp_type,id_type',
    ]);

    if ($validator->fails()) {
        return back()->withErrors($validator)->withInput();
    }

    // Mise à jour
    $guide->update($validator->validated());

    return redirect()->route('guides.index')
        ->with('success', 'تم تحديث الدليل بنجاح!');
}



    public function destroy($id)
    {
        $guide = Guide::findOrFail($id);

        if ($guide->children()->count() > 0) {
            return redirect()->back()->with('error', 'لا يمكن حذف دليل يحتوي على أدلة فرعية.');
        }

        $guide->delete();

        return redirect()->route('guides.index')->with('success', 'تم حذف الدليل بنجاح!');
    }
}
