<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Concours;
use App\Models\FieldDefinition;
use App\Models\Candidate;
use Illuminate\Validation\Rule;
use Illuminate\Support\Str;

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use App\Models\FieldValue;
use App\Models\FieldFile;

class ConcoursController extends Controller
{
	
	
	
	


public function index(Request $request)
{
    $q = (string) $request->input('q', '');

    $query = Concours::withCount('candidates')
                     ->with('user:id,nom_prenom'); 

    if ($q !== '') {
        $query->where(function ($qb) use ($q) {
            $qb->where('titre_ar', 'like', "%{$q}%")
               ->orWhere('titre_fr', 'like', "%{$q}%")
               ->orWhere('titre_en', 'like', "%{$q}%")
               ->orWhere('etat', 'like', "%{$q}%");
        });
    }

    $concours = $query->orderBy('date_debut', 'desc')->paginate(15)->withQueryString();

    return view('backoffice.concours.index', compact('concours', 'q'));
}


    // Show create form
    public function create()
    {
        // In create mode we do NOT pass the global list of fields.
        // The admin will create fields inline; show core fields only.
        return view('backoffice.concours.form', [
            'concours' => new Concours(),
            'fields' => collect(), // empty collection
            'method' => 'create'
        ]);
    }

    /**
 * Normalize an options input that may be an array or a string (comma/newline separated).
 * Returns an array of trimmed non-empty values.
 */
protected function normalizeOptions($input): array
{
    if (is_array($input)) {
        $items = $input;
    } else {
        $items = preg_split('/\r\n|\n|\r|,/', (string) $input);
    }

    return array_values(array_filter(array_map('trim', $items), fn($v) => $v !== ''));
}


    // Persist new concours
public function store(Request $request)
{
      $data = $request->validate([
        // accept either per-language titles or the old single 'titre'
        'titre'     => ['nullable','string','max:255'],
        'titre_ar'  => ['nullable','string','max:255'],
        'titre_fr'  => ['nullable','string','max:255'],
        'titre_en'  => ['nullable','string','max:255'],

        'etat' => ['required','string', Rule::in(['draft','published','closed'])],
        'date_debut' => ['nullable','date'],
        'date_fin' => ['nullable','date'],
        'fields' => ['nullable','array'],
        'fields.*' => ['integer','exists:field_definitions,id'],
        'core_fields' => ['nullable','array'],
        'core_fields.*' => ['string'],
        'core_required' => ['nullable','array'],
        'core_required.*' => ['string'],
        'new_fields' => ['nullable','array'],
        'new_fields.*.label' => ['required_with:new_fields','string','max:255'],
        'new_fields.*.input_type' => ['required_with:new_fields','string'],
        'new_fields.*.is_required' => ['nullable'],
        'new_fields.*.is_visible' => ['nullable'],
        'new_fields.*.options' => ['nullable'],
        'new_fields_payload' => ['nullable','string'],
    ]);
  if (empty($data['titre_ar'] ?? '') && empty($data['titre_fr'] ?? '') && empty($data['titre_en'] ?? '') && empty($data['titre'] ?? '')) {
        return back()->withInput()->withErrors(['titre' => 'Please provide at least one title (Arabic, French or English).']);
    }
    \DB::beginTransaction();
    try {
     $data['id_user'] = auth()->id() ?? $request->input('id_user');

        // choose sensible fallbacks:
        // if a specific language is missing but the old 'titre' exists, use it for that language
        $titre_ar = $data['titre_ar'] ?? ($data['titre'] ?? null);
        $titre_fr = $data['titre_fr'] ?? ($data['titre'] ?? null);
        $titre_en = $data['titre_en'] ?? ($data['titre'] ?? null);

        $concours = Concours::create([
            'titre_ar' => $titre_ar,
            'titre_fr' => $titre_fr,
            'titre_en' => $titre_en,
            'etat' => $data['etat'],
            'date_debut' => $data['date_debut'] ?? null,
            'date_fin' => $data['date_fin'] ?? null,
            'id_user' => $data['id_user'] ?? null,
        ]);
        // Defensive normalizer (arrays, json string, comma/newline string, nested arrays)
        $normalizeOptions = function($input) {
            // If a JSON string representing an array, decode it.
            if (is_string($input)) {
                $decoded = json_decode($input, true);
                if (is_array($decoded)) {
                    $items = $decoded;
                } else {
                    // fallback: split on newline/comma
                    $items = preg_split('/\r\n|\n|\r|,/', $input);
                }
            } elseif (is_array($input)) {
                $items = $input;
            } else {
                $items = [];
            }

            // flatten one level and trim
            $flat = [];
            array_walk_recursive($items, function($v) use (&$flat) {
                $v = is_null($v) ? '' : trim((string)$v);
                if ($v !== '') $flat[] = $v;
            });

            // preserve order, remove duplicates
            $seen = [];
            $out = [];
            foreach ($flat as $v) {
                if (!isset($seen[$v])) {
                    $seen[$v] = true;
                    $out[] = $v;
                }
            }
            return $out;
        };

        // Prefer JSON payload created by JS if present
        $newFieldsPayload = [];
        if ($request->filled('new_fields_payload')) {
            $decoded = json_decode($request->input('new_fields_payload'), true);
            if (is_array($decoded)) {
                $newFieldsPayload = $decoded;
            } else {
                \Log::warning('Concours.store: new_fields_payload present but invalid JSON', ['payload' => substr($request->input('new_fields_payload'),0,1000)]);
            }
        }
        // fallback to classic form inputs
        if (empty($newFieldsPayload) && $request->has('new_fields')) {
            $newFieldsPayload = $request->input('new_fields', []);
        }

        // DEBUG: dump incoming structure so we can inspect server-side
        \Log::debug('Concours.store: incoming new_fields (raw)', ['raw' => $request->input('new_fields')]);
        \Log::debug('Concours.store: incoming new_fields_payload (decoded)', ['decoded_payload' => $newFieldsPayload]);

        $newFieldIds = [];

        foreach ($newFieldsPayload as $idx => $nf) {
            // defensive access (could be object-like from JSON)
            $label = isset($nf['label']) ? trim((string)$nf['label']) : '';
            if ($label === '') {
                \Log::debug("Concours.store: skipping new_field[$idx] because label empty", ['nf' => $nf]);
                continue;
            }

            $input_type = isset($nf['input_type']) ? (string)$nf['input_type'] : 'text';
            $is_required = (!empty($nf['is_required']) && (int)$nf['is_required'] === 1) ? 1 : 0;
            $is_visible = isset($nf['is_visible']) ? ((int)$nf['is_visible'] === 1 ? 1 : 0) : 1;

            // raw options value (could be array or string)
            $rawOptions = $nf['options'] ?? [];
            \Log::debug("Concours.store: new_field[$idx] rawOptions type", ['type' => gettype($rawOptions), 'raw' => $rawOptions]);

            $opts = $normalizeOptions($rawOptions);
            \Log::debug("Concours.store: new_field[$idx] normalized options", ['opts' => $opts]);

            // keep options only for multiple-choice types
            if (!in_array($input_type, ['select','radio','checkbox'])) {
                $opts = [];
            }

            $key = Str::slug(mb_substr($label,0,40) . '-' . time() . '-' . Str::random(4), '_');

            $fdData = [
                'key' => $key,
                'label' => $label,
                'input_type' => $input_type,
                'is_visible' => (bool)$is_visible,
                'is_required' => (bool)$is_required,
                'sort_order' => 100,
            ];

            if (!empty($opts)) {
                // encode before creating
                $fdData['options_json'] = json_encode(array_values($opts), JSON_UNESCAPED_UNICODE);
            } else {
                $fdData['options_json'] = null;
            }

            \Log::debug("Concours.store: creating FieldDefinition", ['index' => $idx, 'fdData' => $fdData]);

            $fd = FieldDefinition::create($fdData);

            if ($fd && $fd->id) {
                $newFieldIds[] = $fd->id;
                \Log::debug("Concours.store: FieldDefinition created", ['id' => $fd->id, 'options_json' => $fd->options_json]);
            } else {
                \Log::warning("Concours.store: FieldDefinition create returned falsy", ['index' => $idx, 'fd' => $fd]);
            }
        }

        // attach existing selected fields
        $selectedFieldIds = array_values(array_unique((array)$request->input('fields', [])));

        // If admin toggled visibility/required for existing fields, update FieldDefinition rows accordingly.
        $fieldVisibleMap = $request->input('field_visible', []);
        $fieldRequiredMap = $request->input('field_required', []);

        foreach ($selectedFieldIds as $fid) {
            $update = [];
            if (isset($fieldVisibleMap[$fid])) $update['is_visible'] = (int)$fieldVisibleMap[$fid] === 1 ? 1 : 0;
            if (isset($fieldRequiredMap[$fid])) $update['is_required'] = (int)$fieldRequiredMap[$fid] === 1 ? 1 : 0;
            if (!empty($update)) {
                FieldDefinition::where('id', $fid)->update($update);
            }
        }

        $allFieldIds = array_values(array_unique(array_merge($selectedFieldIds, $newFieldIds)));
        $concours->fields()->sync($allFieldIds);

        \DB::commit();

        return redirect()->route('concours.index')->with('success', 'تم إنشاء المسابقة بنجاح.');
    } catch (\Throwable $e) {
        \DB::rollBack();
        \Log::error('Concours store error: ' . $e->getMessage(), ['trace' => $e->getTraceAsString(), 'request_all' => substr(json_encode($request->all()),0,2000)]);
        return back()->withInput()->withErrors(['general' => 'حدث خطأ أثناء إنشاء المسابقة. حاول مرة أخرى أو تواصل مع الدعم.']);
    }
}



    // Show a concours with fields and candidates
    public function show(Concours $concours)
    {
        $concours->load(['fields', 'candidates']);
        $candidates = $concours->candidates()->paginate(25);
        return view('backoffice.concours.show', compact('concours','candidates'));
    }

    // Edit form
    public function edit(Concours $concours)
    {
        // Load only fields attached to this concours, ordered by sort_order
        $concours->load('fields');
        $attachedFields = $concours->fields()->orderBy('sort_order')->get();

        return view('backoffice.concours.form', [
            'concours' => $concours,
            'fields' => $attachedFields, // only attached fields
            'method' => 'edit'
        ]);
    }

    // Update concours

public function update(Request $request, Concours $concours)
{
    $data = $request->validate([
        'titre'     => ['nullable','string','max:255'],
        'titre_ar'  => ['nullable','string','max:255'],
        'titre_fr'  => ['nullable','string','max:255'],
        'titre_en'  => ['nullable','string','max:255'],

        'etat' => ['required','string', Rule::in(['draft','published','closed'])],
        'date_debut' => ['nullable','date'],
        'date_fin' => ['nullable','date'],
        'fields' => ['nullable','array'],
        'fields.*' => ['integer','exists:field_definitions,id'],
        'edit_fields' => ['nullable','array'],
        'new_fields' => ['nullable','array'],
        'new_fields.*.label' => ['required_with:new_fields','string','max:255'],
        'new_fields.*.input_type' => ['required_with:new_fields','string'],
        'new_fields.*.is_required' => ['nullable'],
        'new_fields.*.options' => ['nullable'],
        'delete_fields' => ['nullable','array'],
        'delete_fields.*' => ['integer'],
    ]);

    // require at least one title (fallback rules)
    if (empty($data['titre_ar'] ?? '') && empty($data['titre_fr'] ?? '') && empty($data['titre_en'] ?? '') && empty($data['titre'] ?? '')) {
        return back()->withInput()->withErrors(['titre' => 'Please provide at least one title (Arabic, French or English).']);
    }

    DB::beginTransaction();
    try {
        // update titles with fallbacks
        $titre_ar = $data['titre_ar'] ?? ($data['titre'] ?? $concours->titre_ar);
        $titre_fr = $data['titre_fr'] ?? ($data['titre'] ?? $concours->titre_fr);
        $titre_en = $data['titre_en'] ?? ($data['titre'] ?? $concours->titre_en);

        $concours->update([
            'titre_ar' => $titre_ar,
            'titre_fr' => $titre_fr,
            'titre_en' => $titre_en,
            'etat' => $data['etat'],
            'date_debut' => $data['date_debut'] ?? null,
            'date_fin' => $data['date_fin'] ?? null,
        ]);

        // -------------------------
        // Handle inline edits to existing FieldDefinition rows (edit_fields[id][...])
        // -------------------------
        if ($request->filled('edit_fields')) {
            foreach ($request->input('edit_fields') as $fid => $ef) {
                $fid = (int)$fid;
                $fd = FieldDefinition::find($fid);
                if (!$fd) continue;

                $update = [];

                if (isset($ef['label'])) $update['label'] = trim((string)$ef['label']);
                if (isset($ef['input_type'])) $update['input_type'] = (string)$ef['input_type'];
                if (isset($ef['is_required'])) $update['is_required'] = (int)$ef['is_required'] === 1 ? 1 : 0;
                if (isset($ef['is_visible'])) $update['is_visible'] = (int)$ef['is_visible'] === 1 ? 1 : 0;

                if (array_key_exists('options', $ef)) {
                    $rawOptions = $ef['options'];
                    $opts = $this->normalizeOptions($rawOptions);
                    $effectiveType = $update['input_type'] ?? $fd->input_type;
                    if (!in_array($effectiveType, ['select','radio','checkbox'])) {
                        $opts = [];
                    }
                    $update['options_json'] = empty($opts) ? null : json_encode(array_values($opts), JSON_UNESCAPED_UNICODE);
                }

                if (!empty($update)) {
                    $fd->update($update);
                }
            }
        }

        // -------------------------
        // Handle delete requests (delete_fields[]) - safe cascade
        // -------------------------
        if ($request->filled('delete_fields')) {
            $delIds = array_values(array_unique(array_map('intval', (array)$request->input('delete_fields', []))));
            if (!empty($delIds)) {
                // 1) Detach from concours_field pivot (this concours and any others)
                DB::table('concours_field')->whereIn('field_definition_id', $delIds)->delete();

                // 2) Delete associated files (FieldFile) and remove stored files if present
                $files = FieldFile::whereIn('field_definition_id', $delIds)->get();
                foreach ($files as $f) {
                    // if file_path exists on disk, delete it
                    if (!empty($f->file_path) && Storage::exists($f->file_path)) {
                        try { Storage::delete($f->file_path); } catch (\Throwable $e) { \Log::warning('Failed deleting stored file: '.$f->file_path); }
                    }
                }
                FieldFile::whereIn('field_definition_id', $delIds)->delete();

                // 3) Delete associated field values
                FieldValue::whereIn('field_definition_id', $delIds)->delete();

                // 4) Finally delete the FieldDefinition rows
                FieldDefinition::whereIn('id', $delIds)->delete();
            }
        }

        // -------------------------
        // Create new fields (if any) and collect their IDs to attach
        // -------------------------
        $newFieldIds = [];
        if ($request->filled('new_fields')) {
            foreach ($request->input('new_fields') as $nf) {
                $label = trim($nf['label'] ?? '');
                if ($label === '') continue;
                $input_type = $nf['input_type'] ?? 'text';
                $is_required = (!empty($nf['is_required']) && (int)$nf['is_required'] === 1) ? 1 : 0;
                $options_input = $nf['options'] ?? '';

                $opts = $this->normalizeOptions($options_input);
                if (!in_array($input_type, ['select','radio','checkbox'])) {
                    $opts = [];
                }

                $fd = FieldDefinition::create([
                    'key' => Str::slug(mb_substr($label,0,40).'-'.time(), '_'),
                    'label' => $label,
                    'input_type' => $input_type,
                    'options_json' => empty($opts) ? null : json_encode($opts, JSON_UNESCAPED_UNICODE),
                    'is_visible' => true,
                    'is_required' => $is_required,
                    'sort_order' => 100,
                ]);

                if ($fd && $fd->id) $newFieldIds[] = $fd->id;
            }
        }

        // -------------------------
        // Update visibility/required for selected existing fields
        // -------------------------
        $selectedFieldIds = array_values(array_unique((array)$request->input('fields', [])));
        $fieldVisibleMap = $request->input('field_visible', []);
        $fieldRequiredMap = $request->input('field_required', []);
        foreach ($selectedFieldIds as $fid) {
            $update = [];
            if (isset($fieldVisibleMap[$fid])) $update['is_visible'] = (int)$fieldVisibleMap[$fid] === 1 ? 1 : 0;
            if (isset($fieldRequiredMap[$fid])) $update['is_required'] = (int)$fieldRequiredMap[$fid] === 1 ? 1 : 0;
            if (!empty($update)) {
                FieldDefinition::where('id', (int)$fid)->update($update);
            }
        }

        // -------------------------
        // Final attach list: keep only selected + newly created
        // -------------------------
        $finalAttach = array_values(array_unique(array_merge($selectedFieldIds, $newFieldIds)));
        $concours->fields()->sync($finalAttach);

        DB::commit();

        return redirect()->route('concours.index')->with('success', 'تم حفظ التغييرات.');
    } catch (\Throwable $e) {
        DB::rollBack();
        \Log::error('Concours update error: ' . $e->getMessage(), ['trace' => $e->getTraceAsString(), 'request_all' => substr(json_encode($request->all()),0,2000)]);
        return back()->withInput()->withErrors(['general' => 'حدث خطأ أثناء حفظ التغييرات. حاول مرة أخرى أو تواصل مع الدعم.']);
    }
}


/**
 * Remove the specified FieldDefinition from storage (safe cascade).
 *
 * @param  \App\Models\FieldDefinition  $fieldDefinition
 * @return \Illuminate\Http\RedirectResponse
 */
 public function destroy(Concours $concours)
    {
        // detach candidates relation if you want keep candidates rows but unlink:
        $concours->candidates()->detach();

        // optionally delete pivot entries with fields (sync([]) will remove)
        $concours->fields()->sync([]);

        $concours->delete();

        return redirect()->route('concours.index')->with('success', 'تم حذف المناظرة بنجاح');
    }
   // Quick change state (AJAX or normal form)
    public function changeState(Request $request, Concours $concours)
    {
        $data = $request->validate([
            'etat' => ['required', Rule::in(['draft','published','closed'])],
        ]);

        $concours->etat = $data['etat'];
        $concours->save();

        if ($request->wantsJson() || $request->acceptsJson()) {
            return response()->json(['success' => true, 'etat' => $concours->etat, 'label' => $concours->etat_label]);
        }

        return redirect()->back()->with('success', 'تم تحديث الحالة.');
    }
}
