<?php

namespace App\Http\Controllers;


use Illuminate\Http\Request;
use App\Models\Auberge;
use App\Models\ChambreTente;
use App\Models\TypeChambre;
use App\Models\Disponibilite;

class ChambreController extends Controller
{
    public function index($auberge_id)
    {
        $auberge = Auberge::findOrFail($auberge_id);

        $chambres = ChambreTente::where('auberge_id', $auberge_id)->get();

        return view('backoffice.auberges.chambres.index', compact('auberge', 'chambres'));
    }


    public function create($auberge_id)
    {
        $auberge = Auberge::findOrFail($auberge_id);

        $types = TypeChambre::all();

        return view('backoffice.auberges.chambres.create', compact('auberge', 'types'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'num_chambre' => 'required|string|max:11',
            //'nbr_chambre_tente' => 'required|integer',
            'capacite' => 'required|integer',
            'auberge_id' => 'required|exists:auberge,id',
            'type_chambre_id' => 'required|exists:tp_type_chambre,id_type',
            'type' => 'required|string|max:50',
        ]);

        ChambreTente::create($request->all());
        return redirect()->route('chambres.index', $request->auberge_id)
                        ->with('success', 'تمت إضافة الغرفة بنجاح');
    }

     public function show($id_chambre)
    {
        $chambre = ChambreTente::findOrFail($id_chambre);
        return view('backoffice.auberges.chambres.show', compact('chambre'));
    }

    public function edit($id_chambre)
    {
        $chambre = ChambreTente::findOrFail($id_chambre);
        $types = TypeChambre::all();
        return view('backoffice.auberges.chambres.edit', compact('chambre', 'types'));
    }

    public function update(Request $request, $id_chambre)
    {
        $request->validate([
            'num_chambre' => 'required|string|max:11',
            'capacite' => 'required|integer',
            'type_chambre_id' => 'required|exists:tp_type_chambre,id_type',
            'type' => 'required|string|in:chambre,tente',
        ]);

        $chambre = ChambreTente::findOrFail($id_chambre);
        $chambre->update($request->all());

        return redirect()->route('chambres.index', $chambre->auberge_id)
                         ->with('success', 'تم تعديل الغرفة بنجاح');
    }

    public function destroy($id_chambre)
    {
        $chambre = ChambreTente::findOrFail($id_chambre);
        $auberge_id = $chambre->auberge_id;
        $chambre->delete();

        return redirect()->route('chambres.index', $auberge_id)
                         ->with('success', 'تم حذف الغرفة بنجاح');
    }


public function disponibilite(Request $request, $id)
{
    $chambre = ChambreTente::findOrFail($id);

    // Initialiser lits
    $lits = [];
    for ($i = 1; $i <= $chambre->capacite; $i++) {
        $lits[$i] = [
            'est_dispo' => true,
            'sexe_occupe' => null,
        ];
    }

    $date = $request->date ?? date('Y-m-d');

    $dispos = Disponibilite::where('chambre_id', $chambre->id_chambre_tente)
                ->where('date', $date)
                ->get();

    foreach ($dispos as $dispo) {
        $litNum = (int)$dispo->lit_num;
        $lits[$litNum]['est_dispo'] = (bool)$dispo->est_dispo;
        $lits[$litNum]['sexe_occupe'] = $dispo->est_dispo ? null : $dispo->sexe;
    }

    // Calcul disponibilité globale
    $globalDisponible = collect($lits)->contains(fn($l) => $l['est_dispo']);

    return view('backoffice.auberges.chambres.disponibilite', compact('chambre','lits','globalDisponible','date'));
}

public function toggleDisponibilite(Request $request)
{
    $request->validate([
        'chambre_id' => 'required|integer|exists:chambre_tente,id_chambre_tente',
        'lit_num' => 'required|integer|min:1',
        'sexe' => 'nullable|in:homme,femme',
        'est_dispo' => 'required|boolean',
        'date' => 'required|date',
    ]);

    $dispo = Disponibilite::updateOrCreate(
        [
            'chambre_id' => $request->chambre_id,
            'lit_num' => $request->lit_num,
            'date' => $request->date,
        ],
        [
            'est_dispo' => $request->est_dispo,
            'sexe' => $request->est_dispo ? null : $request->sexe,
        ]
    );

    return response()->json([
        'success' => true,
        'est_dispo' => (bool)$dispo->est_dispo,
        'message' => $dispo->est_dispo ? 'تم تفعيل السرير' : 'تم تعطيل السرير'
    ]);
}

public function blockRoom(Request $request, $id)
{
    $request->validate([
        'est_dispo' => 'required|boolean',
        'date' => 'required|date',
    ]);

    $chambre = ChambreTente::findOrFail($id);
    $est = (bool)$request->est_dispo; // true = ouvrir, false = bloquer
    $date = $request->date;

    for ($i = 1; $i <= (int)$chambre->capacite; $i++) {
        $dispo = Disponibilite::firstOrNew([
            'chambre_id' => $chambre->id_chambre_tente,
            'lit_num' => $i,
            'date' => $date,
        ]);

        // Si le lit est déjà occupé, on ne change rien
        if (!$dispo->est_dispo && $dispo->sexe !== null) {
            continue;
        }

        // Sinon on applique la modification de blocage/déblocage
        $dispo->est_dispo = $est; // 1 si ouverture, 0 si blocage
        $dispo->sexe = $dispo->sexe ?? null; // reste null si vide
        $dispo->save();
    }

    return response()->json([
        'success' => true,
        'message' => $est ? 'تم فتح كل الأسرة المتاحة' : 'تم غلق كل الأسرة المتاحة'
    ]);
}




}