<?php

namespace App\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Notifications\Notification;

class StatutDemandeNotification extends Notification
{
    use Queueable;

    protected $demande;
    protected $type;           // accepted|rejected|incomplete|deferred|status_changed
    protected $params;         // assoc array for translation params
    protected $referenceId;
    protected $translationKey; // optional override

    /**
     * @param  \App\Models\DemandeManifestation  $demande
     * @param  string  $type  normalized type (accepted|rejected|incomplete|deferred|status_changed)
     * @param  array   $params translation parameters
     * @param  mixed   $referenceId optional related id (eg. demande id)
     * @param  string|null $translationKey optional explicit key
     */
    public function __construct($demande, string $type, array $params = [], $referenceId = null, ?string $translationKey = null)
    {
        $this->demande = $demande;
        $this->type = $type;
        $this->params = $params;
        $this->referenceId = $referenceId;
        $this->translationKey = $translationKey;
    }

    public function via($notifiable)
    {
        // only database here: mail is sent manually in controllers (keeps behavior consistent)
        return ['database'];
    }

    public function toDatabase($notifiable)
    {
        // build dossier number consistent with your toggle implementation
        $idPad = str_pad($this->demande->id_demande, 3, '0', STR_PAD_LEFT);
        $cin = $this->demande->utilisateur->num_cin ?? $this->demande->utilisateur->cin ?? null;
        $cinDigits = $cin ? preg_replace('/\D+/', '', $cin) : '';
        $cinPad = $cinDigits ? str_pad($cinDigits, 9, '0', STR_PAD_LEFT) : '000000000';
        $year = $this->demande->created_at ? $this->demande->created_at->format('Y') : now()->format('Y');
        $numDossier = "{$idPad}-{$cinPad}-{$year}";

        // explicit key if provided otherwise build from manifestation namespace
        $translationKey = $this->translationKey ?? "site.notifications.manifestation.{$this->type}";

        return [
            'translation_key'    => $translationKey,
            'translation_params' => $this->params,
            'num_dossier'        => $numDossier,
            'date'               => now()->toDateTimeString(),
            'reference_id'       => $this->referenceId,
        ];
    }

    public function toArray($notifiable)
    {
        return [
            'type' => $this->type,
            'params' => $this->params,
            'reference_id' => $this->referenceId,
        ];
    }
}
