<?php

namespace App\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Notifications\Notification;

class StatutDemande2Notification extends Notification
{
    use Queueable;

    protected $demande;
    protected $type;           // accepted|rejected|incomplete|deferred|status_changed
    protected $params;         // assoc array for translation params (reason, date, missing, note, reference)
    protected $referenceId;
    protected $translationKey; // optional explicit key override

    /**
     * @param  \App\Models\DemandeManifestation  $demande
     * @param  string  $type  normalized type (accepted|rejected|incomplete|deferred|status_changed)
     * @param  array   $params translation parameters
     * @param  mixed   $referenceId optional related id (eg. demande id)
     * @param  string|null $translationKey optional explicit key
     */
    public function __construct($demande, string $type, array $params = [], $referenceId = null, ?string $translationKey = null)
    {
        $this->demande = $demande;
        $this->type = $type;
        $this->params = $params;
        $this->referenceId = $referenceId;
        $this->translationKey = $translationKey;
    }

    public function via($notifiable)
    {
        // only DB here — mail is sent by controller to reuse your existing templates
        return ['database'];
    }

    /**
     * toDatabase — store translation key + params so view can localize client-side
     */
    public function toDatabase($notifiable)
    {
        // build dossier number consistent with controller logic
        $idPad = str_pad($this->demande->id_demande, 3, '0', STR_PAD_LEFT);
        $cin = $this->demande->utilisateur->num_cin ?? $this->demande->utilisateur->cin ?? null;
        $cinDigits = $cin ? preg_replace('/\D+/', '', $cin) : '';
        $cinPad = $cinDigits ? str_pad($cinDigits, 9, '0', STR_PAD_LEFT) : '000000000';
        $year = $this->demande->created_at ? $this->demande->created_at->format('Y') : now()->format('Y');
        $numDossier = "{$idPad}-{$cinPad}-{$year}";

        // explicit key if provided otherwise build from manifestation namespace
        $translationKey = $this->translationKey ?? "site.notifications.manifestation.{$this->type}";

        // ensure `reference` param exists (useful for mail & display)
        $params = $this->params;
        if (!array_key_exists('reference', $params)) {
            $params['reference'] = $numDossier;
        }

        // store minimal usable data in DB: translation key + params + reference and timestamp
        return [
            'translation_key'    => $translationKey,
            'translation_params' => $params,
            'num_dossier'        => $numDossier,
            'date'               => now()->toDateTimeString(),
            'reference_id'       => $this->referenceId,
        ];
    }

    /**
     * toArray — legacy/fallback data structure
     */
    public function toArray($notifiable)
    {
        return [
            'type' => $this->type,
            'params' => $this->params,
            'reference_id' => $this->referenceId,
        ];
    }
}
