<?php

namespace App\Http\Controllers;

use App\Models\TropheeSportif;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;

class TropheeSportifController extends Controller
{
   public function index(Request $request)
{
    $search = trim($request->get('search'));
    $words = $search ? preg_split('/\s+/', $search) : [];

    $trophee_sportif = TropheeSportif::when($words, function ($query) use ($words) {
        foreach ($words as $word) {
            $word = str_replace(['\\', '%', '_'], ['\\\\', '\\%', '\\_'], $word);

            $query->where(function ($q) use ($word) {
                $q->where('title', 'LIKE', "%{$word}%")
                  ->orWhere('title_en', 'LIKE', "%{$word}%")
                  ->orWhere('title_fr', 'LIKE', "%{$word}%");
            });
        }
    })
    ->orderBy('created_at','desc')
    ->paginate(10)
    ->appends(['search' => $search]); // يحافظ على البحث أثناء التنقل بين الصفحات

    // Helper closure to compute image source (data:base64 or URL) for each item
    $computeImageSrc = function ($stored) {
        if (empty($stored)) return null;

        // If it's already a full URL, use it directly
        if (filter_var($stored, FILTER_VALIDATE_URL)) {
            return $stored;
        }

        // Build candidate filesystem paths/urls to test (in order)
        $candidates = [];

        // as-stored value first (it may be absolute path or relative)
        $candidates[] = $stored;

        // try prefixing with server root you use
        $serverRoot = '/www/mjs';
        if (strpos($stored, $serverRoot) !== 0) {
            $candidates[] = rtrim($serverRoot, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . ltrim($stored, DIRECTORY_SEPARATOR);
        }

        // try the common shared uploads images folder (basename fallback)
        $candidates[] = rtrim($serverRoot, DIRECTORY_SEPARATOR) . '/shared_uploads/images/' . basename($stored);

        // try public path
        $candidates[] = public_path(ltrim($stored, '/'));

        // try storage path
        $candidates[] = storage_path('app/' . ltrim($stored, '/'));

        // dedupe & normalize
        $seen = [];
        $candidates = array_values(array_filter(array_map(function ($p) use (&$seen) {
            $p = (string)$p;
            if ($p === '') return null;
            // strip file:// wrapper
            $p = preg_replace('#^file://#', '', $p);
            if (isset($seen[$p])) return null;
            $seen[$p] = true;
            return $p;
        }, $candidates)));

        // Try to find a readable file among candidates
        foreach ($candidates as $path) {
            if (@file_exists($path) && @is_file($path) && @is_readable($path)) {
                $mime = @mime_content_type($path) ?: 'image/jpeg';
                try {
                    $contents = @file_get_contents($path);
                } catch (\Throwable $e) {
                    $contents = false;
                }
                if ($contents !== false && $contents !== null) {
                    $b64 = base64_encode($contents);
                    return "data:{$mime};base64,{$b64}";
                }
            }
        }

        // As a fallback: if stored path begins with a slash, expose it as an absolute URL relative to app
        if (strpos($stored, '/') === 0) {
            return url(ltrim($stored, '/'));
        }

        // Nothing found
        return null;
    };

    // Compute imageSrc for each model in the paginator
    foreach ($trophee_sportif as $item) {
        // compute and attach a transient attribute imageSrc that the view can use
        $item->imageSrc = $computeImageSrc($item->image);
    }

    return view('backoffice.trophee_sportif.index', compact('trophee_sportif'));
}



    public function create()
    {
        return view('backoffice.trophee_sportif.create');
    }

    private function youtubeEmbedUrl($url)
    {
        if (preg_match('/(youtu\.be\/|v=)([A-Za-z0-9_\-]+)/', $url, $m)) {
            return 'https://www.youtube.com/embed/' . $m[2];
        }
        return null;
    }


    public function store(Request $request)
    {
            $data = $request->validate([
            'date' => 'required|date',
            'title' => 'required|string|max:255',
            'title_fr' => 'required|string|max:255',
            'title_en' => 'required|string|max:255',
            'content' => 'required|string',
            'content_fr' => 'required|string',
            'content_en' => 'required|string',
            'resume_ar' => 'required|string|max:255',
            'resume_fr' => 'required|string|max:255',
            'resume_en' => 'required|string|max:255',
            'image' => 'nullable|image|max:2048',
            'youtube_url' => 'nullable|url',
            'published' => 'sometimes|boolean',
            ], [
                'date.required' => 'الرجاء تحديد التاريخ.',
                'title.required' => 'الرجاء إدخال العنوان بالعربية.',
                'title_fr.required' => 'الرجاء إدخال العنوان بالفرنسية.',
                'title_en.required' => 'الرجاء إدخال العنوان بالإنجليزية.',
                'content.required' => 'الرجاء إدخال المحتوى بالعربية.',
                'content_fr.required' => 'الرجاء إدخال المحتوى بالفرنسية.',
                'content_en.required' => 'الرجاء إدخال المحتوى بالإنجليزية.',
                'resume_ar.required' => 'الرجاء إدخال الملخّص بالعربية.',
                'resume_fr.required' => 'الرجاء إدخال الملخّص بالفرنسية.',
                'resume_en.required' => 'الرجاء إدخال الملخّص بالإنجليزية.',
                'image.image' => 'يجب أن يكون الملف صورة.',
                'image.max' => 'حجم الصورة يجب ألا يتجاوز 2 ميغابايت.',
                'youtube_url.url' => 'الرجاء إدخال رابط YouTube صالح.',
                'published.boolean' => 'قيمة النشر غير صحيحة.',
            ]);


            // Candidate directories to try for saving uploaded images (ordered)
            $candidates = [
                '/www/mjs/shared_uploads/images',
                '/home/preprov/www/mjs/shared_uploads/images',
                base_path('shared_uploads/images'),
                base_path('../shared_uploads/images'),
                public_path('../../shared_uploads/images'),
            ];

            $usable = null;
            $tried = [];

            if ($request->hasFile('image')) {
                $file = $request->file('image');

                foreach ($candidates as $cand) {
                    $rp = @realpath($cand);
                    $tried[$cand] = [
                        'realpath' => $rp ?: null,
                        'exists' => @file_exists($cand) ? 'yes' : 'no',
                        'is_dir' => @is_dir($cand) ? 'yes' : 'no',
                        'is_writable' => @is_writable($cand) ? 'yes' : 'no'
                    ];
                    if ($rp && is_dir($rp) && is_writable($rp)) {
                        $usable = $rp;
                        break;
                    }
                }

                if (!$usable) {
                    $diag = [
                        'cwd' => @getcwd(),
                        'open_basedir' => ini_get('open_basedir') ?: '(not set)',
                        'tried' => $tried,
                        'get_current_user' => @get_current_user(),
                    ];
                    \Log::warning('No usable upload folder found for TropheeSportif image upload.', $diag);

                    $msg = "Upload folder not accessible by PHP. Checked these candidates: ";
                    foreach ($tried as $c => $info) {
                        $msg .= "[{$c} -> realpath=" . ($info['realpath'] ?? 'null') . ", exists={$info['exists']}, is_dir={$info['is_dir']}, writable={$info['is_writable']}] ";
                    }
                    return back()->withInput()->withErrors(['image' => $msg]);
                }

                // sanitize original filename (keep original name where possible)
                $originalName = $file->getClientOriginalName();
                $originalName = basename($originalName);
                $originalName = preg_replace('/[\0\/\\\\<>:"|?*\x00-\x1F]+/u', '-', $originalName);
                if ($originalName === '' || $originalName === '.' || $originalName === '..') {
                    return back()->withInput()->withErrors(['image' => 'اسم الملف الأصلي غير صالح.']);
                }

                $targetPath = rtrim($usable, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . $originalName;

                // avoid overwriting: append timestamp if exists
                if (file_exists($targetPath)) {
                    $dotPos = strrpos($originalName, '.');
                    if ($dotPos !== false) {
                        $base = substr($originalName, 0, $dotPos);
                        $ext  = substr($originalName, $dotPos);
                    } else {
                        $base = $originalName;
                        $ext = '';
                    }
                    $originalName = $base . '-' . time() . $ext;
                    $targetPath = rtrim($usable, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . $originalName;
                }

                try {
                    // move the uploaded file into the chosen existing directory
                    $file->move($usable, $originalName);

                    // store the absolute path in DB
                    $data['image'] = $targetPath;
                } catch (\Throwable $e) {
                    \Log::error('TropheeSportif image move failed: ' . $e->getMessage(), ['exception' => $e, 'usable' => $usable]);
                    return back()->withInput()->withErrors(['image' => 'فشل حفظ الصورة على الخادم. (راجع سجلات الخادم).']);
                }
            } else {
                $data['image'] = null;
            }

            $data['published'] = $request->input('published', 0);
            $data['slug'] = \Illuminate\Support\Str::slug($data['title']) . '-' . \Illuminate\Support\Str::random(6);
            $data['id_user'] = auth()->id();

            \App\Models\TropheeSportif::create($data);

           return redirect()->route('trophee_sportif.index')->with('success', 'تم إنشاء التتويج الرياضي بنجاح.');

    }

    public function show(TropheeSportif $trophee_sportif)
    {
        $embed = $this->youtubeEmbedUrl($trophee_sportif->youtube_url);
        return view('backoffice.trophee_sportif.show', compact('trophee_sportif', 'embed'));
    }

   public function edit(TropheeSportif $trophee_sportif)
{
    // compute a usable image source (asset URL, data: URI, or absolute URL)
    $computeImageSrc = function ($stored) {
        if (empty($stored)) {
            return null;
        }

        // If already a full URL, return it
        if (filter_var($stored, FILTER_VALIDATE_URL)) {
            return $stored;
        }

        // If it's binary data (old format), return data URI
        if (!is_string($stored)) {
            try {
                return "data:image/jpeg;base64," . base64_encode($stored);
            } catch (\Throwable $e) {
                // continue to file checks
            }
        }

        $basename = basename($stored);

        // Candidate filesystem locations to check (ordered)
        $candidates = [
            $stored, // as-stored (maybe absolute path)
            '/www/mjs/shared_uploads/images/' . $basename,
            '/home/preprov/www/mjs/shared_uploads/images/' . $basename,
            public_path('shared_uploads/images/' . $basename),
            storage_path('app/' . $basename),
        ];

        // normalize & dedupe while preserving order
        $seen = [];
        $unique = [];
        foreach ($candidates as $p) {
            if (!$p) continue;
            $p = preg_replace('#^file://#', '', (string)$p);
            if (isset($seen[$p])) continue;
            $seen[$p] = true;
            $unique[] = $p;
        }

        // If the file exists and is readable, prefer returning an asset URL (requires a symlink or alias)
        foreach ($unique as $path) {
            if (@file_exists($path) && @is_file($path) && @is_readable($path)) {
                // If path is under the shared_uploads folder or public, return an asset URL so the browser can fetch it
                if (strpos($path, '/www/mjs/shared_uploads/images') === 0
                    || strpos($path, public_path('shared_uploads/images')) === 0
                    || strpos($path, public_path()) === 0
                ) {
                    // asset('shared_uploads/images/...') will resolve to:
                    // http(s)://<host>/<base-path>/shared_uploads/images/{basename}
                    return asset('shared_uploads/images/' . $basename);
                }

                // Otherwise read file and return data URI
                try {
                    $mime = @mime_content_type($path) ?: 'image/jpeg';
                    $contents = @file_get_contents($path);
                    if ($contents !== false) {
                        return "data:{$mime};base64," . base64_encode($contents);
                    }
                } catch (\Throwable $e) {
                    // continue
                }
            }
        }

        // If stored string contains /shared_uploads, expose it as a URL
        if (preg_match('#(/shared_uploads/.*)$#', $stored, $m)) {
            return asset(ltrim($m[1], '/'));
        }

        // Final fallback: return an asset URL assuming the file will live at shared_uploads/images/{basename}
        return asset('shared_uploads/images/' . $basename);
    };

    $trophee_sportif->imageSrc = $computeImageSrc($trophee_sportif->image);

    return view('backoffice.trophee_sportif.edit', compact('trophee_sportif'));
}

   public function update(Request $request, TropheeSportif $trophee_sportif)
{
    $data = $request->validate([
        'date' => 'required|date',
        'title' => 'required|string|max:255',
        'title_fr' => 'required|string|max:255',
        'title_en' => 'required|string|max:255',
        'content' => 'required|string',
        'content_fr' => 'required|string',
        'content_en' => 'required|string',
        'resume_ar' => 'required|string|max:255',
        'resume_fr' => 'required|string|max:255',
        'resume_en' => 'required|string|max:255',
        'image' => 'nullable|image|max:2048',
        'youtube_url' => 'nullable|url',
        'published' => 'sometimes|boolean',
    ], [
        'date.required' => 'الرجاء تحديد التاريخ.',
        'title.required' => 'الرجاء إدخال العنوان بالعربية.',
        'title_fr.required' => 'الرجاء إدخال العنوان بالفرنسية.',
        'title_en.required' => 'الرجاء إدخال العنوان بالإنجليزية.',
        'content.required' => 'الرجاء إدخال المحتوى بالعربية.',
        'content_fr.required' => 'الرجاء إدخال المحتوى بالفرنسية.',
        'content_en.required' => 'الرجاء إدخال المحتوى بالإنجليزية.',
        'resume_ar.required' => 'الرجاء إدخال الملخّص بالعربية.',
        'resume_fr.required' => 'الرجاء إدخال الملخّص بالفرنسية.',
        'resume_en.required' => 'الرجاء إدخال الملخّص بالإنجليزية.',
        'image.image' => 'يجب أن يكون الملف صورة.',
        'image.max' => 'حجم الصورة يجب ألا يتجاوز 2 ميغابايت.',
        'youtube_url.url' => 'الرجاء إدخال رابط YouTube صالح.',
        'published.boolean' => 'قيمة النشر غير صحيحة.',
    ]);

    // Candidate directories to try for saving uploaded images (ordered)
    $candidates = [
        '/www/mjs/shared_uploads/images',
        '/home/preprov/www/mjs/shared_uploads/images',
        base_path('shared_uploads/images'),
        base_path('../shared_uploads/images'),
        public_path('shared_uploads/images'),
    ];

    // Helper: resolve an absolute filesystem path for a stored DB value if it exists/readable.
    $resolveExistingFilePath = function ($stored) use ($candidates) {
        if (empty($stored)) return null;
        if (!is_string($stored)) return null;
        if (preg_match('#^data:#', $stored)) return null;
        if (filter_var($stored, FILTER_VALIDATE_URL)) return null;

        $basename = basename($stored);
        $tries = [];

        // try stored as-is
        $tries[] = $stored;

        // if stored starts with /shared_uploads try common prefixes
        if (preg_match('#^/shared_uploads/#', $stored)) {
            $serverPrefixes = ['/www/mjs', '/home/preprov/www/mjs'];
            foreach ($serverPrefixes as $prefix) {
                $tries[] = rtrim($prefix, DIRECTORY_SEPARATOR) . $stored;
            }
            $tries[] = public_path(ltrim($stored, '/'));
        }

        // try basenames inside known candidate dirs
        foreach ($candidates as $cand) {
            $tries[] = rtrim($cand, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . $basename;
        }

        // normalize & dedupe
        $unique = [];
        foreach ($tries as $p) {
            if (!$p) continue;
            $p = preg_replace('#^file://#', '', (string)$p);
            if (!isset($unique[$p])) $unique[$p] = $p;
        }

        foreach ($unique as $path) {
            if (@file_exists($path) && @is_file($path) && @is_readable($path)) {
                return realpath($path) ?: $path;
            }
        }

        return null;
    };

    // If a new image was uploaded, handle moving it into a usable directory and deleting the old one.
    if ($request->hasFile('image')) {
        $file = $request->file('image');

        // find a usable directory
        $usable = null;
        $tried = [];
        foreach ($candidates as $cand) {
            $rp = @realpath($cand);
            $tried[$cand] = [
                'realpath' => $rp ?: null,
                'exists' => @file_exists($cand) ? 'yes' : 'no',
                'is_dir' => @is_dir($cand) ? 'yes' : 'no',
                'is_writable' => @is_writable($cand) ? 'yes' : 'no'
            ];
            if ($rp && is_dir($rp) && is_writable($rp)) {
                $usable = $rp;
                break;
            }
        }

        if (!$usable) {
            \Log::warning('No usable upload folder found for TropheeSportif image update.', [
                'cwd' => @getcwd(),
                'open_basedir' => ini_get('open_basedir') ?: '(not set)',
                'tried' => $tried,
                'get_current_user' => @get_current_user(),
            ]);

            $msg = "Upload folder not accessible by PHP. Checked these candidates: ";
            foreach ($tried as $c => $info) {
                $msg .= "[{$c} -> realpath=" . ($info['realpath'] ?? 'null') . ", exists={$info['exists']}, is_dir={$info['is_dir']}, writable={$info['is_writable']}] ";
            }
            return back()->withInput()->withErrors(['image' => $msg]);
        }

        // sanitize filename
        $originalName = $file->getClientOriginalName();
        $originalName = basename($originalName);
        $originalName = preg_replace('/[\0\/\\\\<>:"|?*\x00-\x1F]+/u', '-', $originalName);
        if ($originalName === '' || $originalName === '.' || $originalName === '..') {
            return back()->withInput()->withErrors(['image' => 'اسم الملف الأصلي غير صالح.']);
        }

        $targetPath = rtrim($usable, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . $originalName;

        // avoid overwrite
        if (file_exists($targetPath)) {
            $dotPos = strrpos($originalName, '.');
            if ($dotPos !== false) {
                $base = substr($originalName, 0, $dotPos);
                $ext  = substr($originalName, $dotPos);
            } else {
                $base = $originalName;
                $ext = '';
            }
            $originalName = $base . '-' . time() . $ext;
            $targetPath = rtrim($usable, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . $originalName;
        }

        // move uploaded file
        try {
            $file->move($usable, $originalName);
        } catch (\Throwable $e) {
            \Log::error('TropheeSportif image move failed on update: ' . $e->getMessage(), ['exception' => $e, 'usable' => $usable]);
            return back()->withInput()->withErrors(['image' => 'فشل حفظ الصورة على الخادم. (راجع سجلات الخادم).']);
        }

        // Attempt to delete the old file safely
        try {
            $oldStored = $trophee_sportif->image;
            $oldPath = $resolveExistingFilePath($oldStored);
            if ($oldPath) {
                // Allowed prefixes (realpath if available)
                $allowedPrefixes = [
                    realpath('/www/mjs/shared_uploads/images') ?: '/www/mjs/shared_uploads/images',
                    realpath('/home/preprov/www/mjs/shared_uploads/images') ?: '/home/preprov/www/mjs/shared_uploads/images',
                    realpath(public_path('shared_uploads/images')) ?: public_path('shared_uploads/images'),
                ];

                $oldReal = realpath($oldPath) ?: $oldPath;
                $isAllowed = false;
                foreach ($allowedPrefixes as $prefix) {
                    if (!$prefix) continue;
                    $prefixReal = realpath($prefix) ?: $prefix;
                    if ($prefixReal && strpos($oldReal, $prefixReal) === 0) {
                        $isAllowed = true;
                        break;
                    }
                }

                if ($isAllowed && @file_exists($oldReal) && @is_file($oldReal)) {
                    try {
                        @unlink($oldReal);
                    } catch (\Throwable $e) {
                        \Log::warning('Old TropheeSportif image deletion failed: ' . $e->getMessage(), ['old' => $oldReal]);
                    }
                } else {
                    \Log::info('Skipping deletion of old TropheeSportif image (not in allowed folders or not found).', [
                        'oldStored' => $oldStored,
                        'resolvedOldPath' => $oldReal,
                    ]);
                }
            }
        } catch (\Throwable $e) {
            \Log::warning('Exception while deleting old TropheeSportif image: ' . $e->getMessage(), ['exception' => $e]);
            // do not block update on deletion errors
        }

        // store absolute path to the new file in DB
        $data['image'] = $targetPath;
    } else {
        // keep existing image if none uploaded
        $data['image'] = $trophee_sportif->image;
    }

    // other fields
    $data['id_user'] = auth()->id();
    $data['published'] = $request->input('published', 0);

    // update model
    $trophee_sportif->update($data);

    return redirect()->route('trophee_sportif.index')->with('success', 'تم تحديث التتويج الرياضي بنجاح.');
}

    public function destroy(TropheeSportif $trophee_sportif)
    {
        if ($trophee_sportif->image && Storage::disk('public')->exists($trophee_sportif->image)) {
            Storage::disk('public')->delete($trophee_sportif->image);
        }
        $trophee_sportif->delete();

        return redirect()->route('trophee_sportif.index')->with('success', 'تم حذف التتويج الرياضي بنجاح.');
    }


}
