<?php

namespace App\Http\Controllers;


use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;
use App\Models\News;
use Illuminate\Support\Facades\Log;

class NewsController extends Controller
{

    public function index(Request $request)
{
    $search = trim($request->get('search'));
    $words = $search ? preg_split('/\s+/', $search) : [];

    $news = News::when($words, function ($query) use ($words) {
        foreach ($words as $word) {
            // Échapper caractères spéciaux pour LIKE
            $word = str_replace(['\\', '%', '_'], ['\\\\', '\\%', '\\_'], $word);

            $query->where(function ($q) use ($word) {
                $q->where('title', 'LIKE', "%{$word}%")
                  ->orWhere('title_en', 'LIKE', "%{$word}%")
                  ->orWhere('title_fr', 'LIKE', "%{$word}%");
            });
        }
    })
    ->orderBy('created_at','desc')
    ->paginate(10)
    ->appends(['search' => $search]); // يحافظ على البحث أثناء الانتقال بين الصفحات

    // Closure to compute an image source usable in <img src="...">
    $computeImageSrc = function ($stored) {
        if (empty($stored)) return null;

        // If already a URL, return it
        if (filter_var($stored, FILTER_VALIDATE_URL)) {
            return $stored;
        }

        $candidates = [];

        // as-stored value (may be absolute path or relative)
        $candidates[] = $stored;

        // try prefixing with server root you use
        $serverRoot = '/www/mjs';
        if (strpos($stored, $serverRoot) !== 0) {
            $candidates[] = rtrim($serverRoot, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . ltrim($stored, DIRECTORY_SEPARATOR);
        }

        // try shared uploads images folder with basename
        $candidates[] = rtrim($serverRoot, DIRECTORY_SEPARATOR) . '/shared_uploads/images/' . basename($stored);

        // try public path
        $candidates[] = public_path(ltrim($stored, '/'));

        // try storage path
        $candidates[] = storage_path('app/' . ltrim($stored, '/'));

        // normalize/dedupe
        $seen = [];
        $candidates = array_values(array_filter(array_map(function ($p) use (&$seen) {
            $p = (string)$p;
            if ($p === '') return null;
            $p = preg_replace('#^file://#', '', $p);
            if (isset($seen[$p])) return null;
            $seen[$p] = true;
            return $p;
        }, $candidates)));

        // Try to load a readable file and return a data:...;base64, string
        foreach ($candidates as $path) {
            if (@file_exists($path) && @is_file($path) && @is_readable($path)) {
                $mime = @mime_content_type($path) ?: 'image/jpeg';
                try {
                    $contents = @file_get_contents($path);
                } catch (\Throwable $e) {
                    $contents = false;
                }
                if ($contents !== false && $contents !== null) {
                    $b64 = base64_encode($contents);
                    return "data:{$mime};base64,{$b64}";
                }
            }
        }

        // fallback: if stored path starts with slash, expose as URL
        if (strpos($stored, '/') === 0) {
            return url(ltrim($stored, '/'));
        }

        return null;
    };

    // attach computed imageSrc to each item in the paginator
    foreach ($news as $item) {
        $item->imageSrc = $computeImageSrc($item->image);
    }

    return view('backoffice.news.index', compact('news'));
}




    public function create()
    {
        return view('backoffice.news.create');
    }

    private function youtubeEmbedUrl($url)
    {
        if (preg_match('/(youtu\.be\/|v=)([A-Za-z0-9_\-]+)/', $url, $m)) {
            return 'https://www.youtube.com/embed/' . $m[2];
        }
        return null;
    }


public function store(Request $request)
{
    $data = $request->validate([
        'date' => 'required|date',
        'title' => 'required|string|max:255',
        'title_fr' => 'required|string|max:255',
        'title_en' => 'required|string|max:255',
        'content' => 'required|string',
        'content_fr' => 'required|string',
        'content_en' => 'required|string',
        'resume_ar' => 'required|string|max:255',
        'resume_fr' => 'required|string|max:255',
        'resume_en' => 'required|string|max:255',
        'image' => 'nullable|image|max:2048',
        'youtube_url' => 'nullable|url',
        'published' => 'sometimes|boolean',
    ], [
        'date.required' => 'الرجاء تحديد التاريخ.',
        'title.required' => 'الرجاء إدخال العنوان بالعربية.',
        'title_fr.required' => 'الرجاء إدخال العنوان بالفرنسية.',
        'title_en.required' => 'الرجاء إدخال العنوان بالإنجليزية.',
        'content.required' => 'الرجاء إدخال المحتوى بالعربية.',
        'content_fr.required' => 'الرجاء إدخال المحتوى بالفرنسية.',
        'content_en.required' => 'الرجاء إدخال المحتوى بالإنجليزية.',
        'resume_ar.required' => 'الرجاء إدخال الملخّص بالعربية.',
        'resume_fr.required' => 'الرجاء إدخال الملخّص بالفرنسية.',
        'resume_en.required' => 'الرجاء إدخال الملخّص بالإنجليزية.',
        'image.image' => 'يجب أن يكون الملف صورة.',
        'image.max' => 'حجم الصورة يجب ألا يتجاوز 2 ميغابايت.',
        'youtube_url.url' => 'الرجاء إدخال رابط YouTube صالح.',
        'published.boolean' => 'قيمة النشر غير صحيحة.',
    ]);

    $candidates = [
        '/www/mjs/shared_uploads/images',
        '/home/preprov/www/mjs/shared_uploads/images',
        base_path('shared_uploads/images'),
        base_path('../shared_uploads/images'),
        public_path('../../shared_uploads/images'),
    ];

    $usable = null;
    $tried = [];

    if ($request->hasFile('image')) {
        $file = $request->file('image');

        foreach ($candidates as $cand) {
            $rp = @realpath($cand);
            $tried[$cand] = [
                'realpath' => $rp ?: null,
                'exists' => @file_exists($cand) ? 'yes' : 'no',
                'is_dir' => @is_dir($cand) ? 'yes' : 'no',
                'is_writable' => @is_writable($cand) ? 'yes' : 'no'
            ];
            if ($rp && is_dir($rp) && is_writable($rp)) {
                $usable = $rp;
                break;
            }
        }

        if (!$usable) {
            $diag = [
                'cwd' => @getcwd(),
                'open_basedir' => ini_get('open_basedir') ?: '(not set)',
                'tried' => $tried,
                'get_current_user' => @get_current_user(),
            ];
            \Log::warning('No usable upload folder found for image upload.', $diag);

            $msg = "Upload folder not accessible by PHP. Checked these candidates: ";
            foreach ($tried as $c => $info) {
                $msg .= "[{$c} -> realpath=" . ($info['realpath'] ?? 'null') . ", exists={$info['exists']}, is_dir={$info['is_dir']}, writable={$info['is_writable']}] ";
            }
            return back()->withInput()->withErrors(['image' => $msg]);
        }

        $originalName = $file->getClientOriginalName();
        $originalName = basename($originalName); 
        $originalName = preg_replace('/[\0\/\\\\<>:"|?*\x00-\x1F]+/u', '-', $originalName);
        if ($originalName === '' || $originalName === '.' || $originalName === '..') {
            return back()->withInput()->withErrors(['image' => 'اسم الملف الأصلي غير صالح.']);
        }

        $targetPath = rtrim($usable, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . $originalName;

        if (file_exists($targetPath)) {
            $dotPos = strrpos($originalName, '.');
            if ($dotPos !== false) {
                $base = substr($originalName, 0, $dotPos);
                $ext  = substr($originalName, $dotPos);
            } else {
                $base = $originalName;
                $ext = '';
            }
            $originalName = $base . '-' . time() . $ext;
            $targetPath = rtrim($usable, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . $originalName;
        }

        try {
           
            $file->move($usable, $originalName);

            $data['image'] = $targetPath;
        } catch (\Throwable $e) {
            \Log::error('Image move failed: ' . $e->getMessage(), ['exception' => $e, 'usable' => $usable]);
            return back()->withInput()->withErrors(['image' => 'فشل حفظ الصورة على الخادم. (راجع سجلات الخادم).']);
        }
    } else {
        $data['image'] = null;
    }

    $data['published'] = $request->input('published', 0);
    $data['slug'] = \Illuminate\Support\Str::slug($data['title']) . '-' . \Illuminate\Support\Str::random(6);
    $data['id_user'] = auth()->id();

    \App\Models\News::create($data);

    return redirect()->route('news.index')->with('success', 'تم إنشاء الخبر بنجاح.');
}


    public function show(News $news)
    {
        $embed = $this->youtubeEmbedUrl($news->youtube_url);
        return view('backoffice.news.show', compact('news', 'embed'));
    }

   public function edit(News $news)
{
    $computeImageSrc = function ($stored) {
        if (empty($stored)) {
            return null;
        }

        if (filter_var($stored, FILTER_VALIDATE_URL)) {
            return $stored;
        }

        if (!is_string($stored)) {
            try {
                $b64 = base64_encode($stored);
                return "data:image/jpeg;base64,{$b64}";
            } catch (\Throwable $e) {
            }
        }

        $basename = basename($stored);

        $candidates = [
            $stored,
            '/www/mjs/shared_uploads/images/' . $basename,
            '/home/preprov/www/mjs/shared_uploads/images/' . $basename,
            public_path('shared_uploads/images/' . $basename),
            storage_path('app/' . $basename),
        ];

        $seen = [];
        $unique = [];
        foreach ($candidates as $p) {
            if (!$p) continue;
            $p = (string) $p;
            $p = preg_replace('#^file://#', '', $p);
            if (isset($seen[$p])) continue;
            $seen[$p] = true;
            $unique[] = $p;
        }

        
        foreach ($unique as $path) {
            
            if (@file_exists($path) && @is_file($path) && @is_readable($path)) {
                if (strpos($path, '/www/mjs/shared_uploads/images') === 0 ||
                    strpos($path, public_path('shared_uploads/images')) === 0 ||
                    strpos($path, public_path()) === 0
                ) {
                    return asset('shared_uploads/images/' . $basename);
                }

                try {
                    $mime = @mime_content_type($path) ?: 'image/jpeg';
                    $contents = @file_get_contents($path);
                    if ($contents !== false) {
                        return "data:{$mime};base64," . base64_encode($contents);
                    }
                } catch (\Throwable $e) {
                    // ignore and continue
                }
            }
        }
        if (preg_match('#(/shared_uploads/.*)$#', $stored, $m)) {
            return asset(ltrim($m[1], '/'));
        }
        return asset('shared_uploads/images/' . $basename);
    };

    $news->imageSrc = $computeImageSrc($news->image);

    return view('backoffice.news.edit', compact('news'));
}


  
	
public function update(Request $request, News $news)
{
    $data = $request->validate([
        'date' => 'required|date',
        'title' => 'required|string|max:255',
        'title_fr' => 'required|string|max:255',
        'title_en' => 'required|string|max:255',
        'content' => 'required|string',
        'content_fr' => 'required|string',
        'content_en' => 'required|string',
        'resume_ar' => 'required|string|max:255',
        'resume_fr' => 'required|string|max:255',
        'resume_en' => 'required|string|max:255',
        'image' => 'nullable|image|max:2048',
        'youtube_url' => 'nullable|url',
        'published' => 'sometimes|boolean',
    ], [
        'date.required' => 'الرجاء تحديد التاريخ.',
        'title.required' => 'الرجاء إدخال العنوان بالعربية.',
        'title_fr.required' => 'الرجاء إدخال العنوان بالفرنسية.',
        'title_en.required' => 'الرجاء إدخال العنوان بالإنجليزية.',
        'content.required' => 'الرجاء إدخال المحتوى بالعربية.',
        'content_fr.required' => 'الرجاء إدخال المحتوى بالفرنسية.',
        'content_en.required' => 'الرجاء إدخال المحتوى بالإنجليزية.',
        'resume_ar.required' => 'الرجاء إدخال الملخّص بالعربية.',
        'resume_fr.required' => 'الرجاء إدخال الملخّص بالفرنسية.',
        'resume_en.required' => 'الرجاء إدخال الملخّص بالالإنجليزية.',
        'image.image' => 'يجب أن يكون الملف صورة.',
        'image.max' => 'حجم الصورة يجب ألا يتجاوز 2 ميغابايت.',
        'youtube_url.url' => 'الرجاء إدخال رابط YouTube صالح.',
        'published.boolean' => 'قيمة النشر غير صحيحة.',
    ]);

    // Candidate directories to try for saving uploaded images (same order as store)
    $candidates = [
        '/www/mjs/shared_uploads/images',
        '/home/preprov/www/mjs/shared_uploads/images',
        base_path('shared_uploads/images'),
        base_path('../shared_uploads/images'),
        public_path('shared_uploads/images'),
    ];

    // Helper: resolve an absolute filesystem path for an existing stored DB value (if possible).
    // Returns absolute path string if a readable file exists, otherwise null.
    $resolveExistingFilePath = function ($stored) use ($candidates) {
        if (empty($stored)) return null;

        // If stored is a data URI or a remote URL, nothing to resolve on disk
        if (!is_string($stored)) return null;
        if (preg_match('#^data:#', $stored)) return null;
        if (filter_var($stored, FILTER_VALIDATE_URL)) return null;

        $basename = basename($stored);
        $tries = [];

        // try stored value as-is
        $tries[] = $stored;

        // if stored is a web path like /shared_uploads/..., try mapping it to common server roots
        if (preg_match('#^/shared_uploads/#', $stored)) {
            // common server prefixes
            $serverPrefixes = ['/www/mjs', '/home/preprov/www/mjs'];
            foreach ($serverPrefixes as $prefix) {
                $tries[] = rtrim($prefix, DIRECTORY_SEPARATOR) . $stored;
            }
            // also try front/public + stored
            $tries[] = public_path(ltrim($stored, '/'));
        }

        // try basenames located in candidate upload dirs
        foreach ($candidates as $cand) {
            $tries[] = rtrim($cand, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . $basename;
        }

        // dedupe
        $unique = [];
        foreach ($tries as $p) {
            if (!$p) continue;
            $p = preg_replace('#^file://#', '', (string)$p);
            if (!isset($unique[$p])) $unique[$p] = $p;
        }

        // return the first readable file we find
        foreach ($unique as $path) {
            if (@file_exists($path) && @is_file($path) && @is_readable($path)) {
                return realpath($path);
            }
        }

        return null;
    };

    // If a new image was uploaded, handle move + delete old
    if ($request->hasFile('image')) {
        $file = $request->file('image');

        // find a usable directory from candidates
        $usable = null;
        $tried = [];
        foreach ($candidates as $cand) {
            $rp = @realpath($cand);
            $tried[$cand] = [
                'realpath' => $rp ?: null,
                'exists' => @file_exists($cand) ? 'yes' : 'no',
                'is_dir' => @is_dir($cand) ? 'yes' : 'no',
                'is_writable' => @is_writable($cand) ? 'yes' : 'no'
            ];
            if ($rp && is_dir($rp) && is_writable($rp)) {
                $usable = $rp;
                break;
            }
        }

        if (!$usable) {
            \Log::warning('No usable upload folder found for image upload (update).', [
                'cwd' => @getcwd(),
                'open_basedir' => ini_get('open_basedir') ?: '(not set)',
                'tried' => $tried,
                'get_current_user' => @get_current_user(),
            ]);

            $msg = "Upload folder not accessible by PHP. Checked these candidates: ";
            foreach ($tried as $c => $info) {
                $msg .= "[{$c} -> realpath=" . ($info['realpath'] ?? 'null') . ", exists={$info['exists']}, is_dir={$info['is_dir']}, writable={$info['is_writable']}] ";
            }
            return back()->withInput()->withErrors(['image' => $msg]);
        }

        // sanitize original filename
        $originalName = $file->getClientOriginalName();
        $originalName = basename($originalName);
        $originalName = preg_replace('/[\0\/\\\\<>:"|?*\x00-\x1F]+/u', '-', $originalName);
        if ($originalName === '' || $originalName === '.' || $originalName === '..') {
            return back()->withInput()->withErrors(['image' => 'اسم الملف الأصلي غير صالح.']);
        }

        // prepare target path and avoid overwrite
        $targetPath = rtrim($usable, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . $originalName;
        if (file_exists($targetPath)) {
            $dotPos = strrpos($originalName, '.');
            if ($dotPos !== false) {
                $base = substr($originalName, 0, $dotPos);
                $ext  = substr($originalName, $dotPos);
            } else {
                $base = $originalName;
                $ext = '';
            }
            $originalName = $base . '-' . time() . $ext;
            $targetPath = rtrim($usable, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . $originalName;
        }

        // move uploaded file
        try {
            $file->move($usable, $originalName);
        } catch (\Throwable $e) {
            \Log::error('Image move failed (update): ' . $e->getMessage(), ['exception' => $e, 'usable' => $usable]);
            return back()->withInput()->withErrors(['image' => 'فشل حفظ الصورة على الخادم. (راجع سجلات الخادم).']);
        }

        // Delete old image safely if it exists and resolves to a file inside expected locations
        try {
            $oldStored = $news->image;
            $oldPath = $resolveExistingFilePath($oldStored);
            if ($oldPath) {
                // extra safety: only unlink if the old path is inside one of the approved upload directories
                $allowedPrefixes = [
                    realpath('/www/mjs/shared_uploads/images') ?: '/www/mjs/shared_uploads/images',
                    realpath('/home/preprov/www/mjs/shared_uploads/images') ?: '/home/preprov/www/mjs/shared_uploads/images',
                    realpath(public_path('shared_uploads/images')) ?: public_path('shared_uploads/images'),
                ];
                $oldPathReal = realpath($oldPath);

                $isAllowed = false;
                foreach ($allowedPrefixes as $prefix) {
                    if (!$prefix) continue;
                    $prefixReal = realpath($prefix) ?: $prefix;
                    if ($prefixReal && strpos($oldPathReal, $prefixReal) === 0) {
                        $isAllowed = true;
                        break;
                    }
                }

                if ($isAllowed && @file_exists($oldPathReal) && @is_file($oldPathReal)) {
                    try {
                        @unlink($oldPathReal);
                    } catch (\Throwable $e) {
                        \Log::warning('Old image deletion failed during news update: ' . $e->getMessage(), ['old' => $oldPathReal]);
                    }
                } else {
                    // log possible suspicious path or non-allowed deletion attempt
                    \Log::info('Skipping deletion of old image (not in allowed folders) or not found.', [
                        'oldStored' => $oldStored,
                        'resolvedOldPath' => $oldPathReal ?? $oldPath,
                        'allowedPrefixes' => $allowedPrefixes,
                    ]);
                }
            }
        } catch (\Throwable $e) {
            \Log::warning('Exception while trying to delete old image: ' . $e->getMessage(), ['exception' => $e]);
            // do not block the update if deletion failed
        }

        // store absolute path to the moved file in DB
        $data['image'] = $targetPath;
    } else {
        // keep existing DB value if no new upload
        $data['image'] = $news->image;
    }

    // other fields
    $data['id_user'] = auth()->id();
    $data['published'] = $request->input('published', 0);

    // perform update
    $news->update($data);

    return redirect()->route('news.index')->with('success', 'تم تحديث الخبر بنجاح.');
}

    public function destroy(News $news)
    {
        if ($news->image && Storage::disk('public')->exists($news->image)) {
            Storage::disk('public')->delete($news->image);
        }
        $news->delete();

        return redirect()->route('news.index')->with('success', 'تم حذف الخبر بنجاح.');
    }


}
