<?php

namespace App\Http\Controllers\ManifestationSport;

use Illuminate\Support\Facades\Auth;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use App\Models\Utilisateur;

use App\Models\DemandeManifestation;
use App\Models\TpAttachement;
use App\Models\TpTypeMembre;
use App\Models\TpDomaineEducation;
use App\Models\TpTypeEtrangere;
use App\Models\TeamsAssociation;
use App\Models\OrganismeNonSportif;
use App\Models\EvenementSportif;
use App\Models\MembreOrgInvite;
use App\Models\AssocParticipante;
use App\Models\SportifParticipant;
use App\Models\Etrangere;
use App\Models\TypeFormulaire;
use App\Models\Gouvernorat;
use App\Models\Delegation;
use App\Models\TpStatut;
use App\Models\AthleteIndividuel;
use App\Models\Attachement;
use App\Models\Identite_personnel;
use Illuminate\Support\Facades\Mail;
use App\Mail\StatutDemandeMail;
use App\Notifications\StatutDemandeNotification;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use Mpdf\Mpdf;
use App\Models\ImmobilierStatutHistory;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

class ManifestationSportController extends Controller
{
    public function index(Request $request)
    {
        $typesIdentite = Identite_personnel::all();

        $query = DemandeManifestation::with([
            'utilisateur',
            'organismeNonSportif',
            'evenementSportifs.gouvernorat',
            'evenementSportifs.delegation',
            'statut'
        ])
        ->where('type_formulaire_id', 12) 
        ->orderByDesc('id_demande');

        if ($request->filled('date')) {
            $query->whereDate('created_at', $request->date);
        }

        if ($request->filled('type_identite') && $request->filled('cin')) {
            $query->whereHas('utilisateur', function($q) use ($request) {
                $q->where('type_identite_id', $request->type_identite)
                ->where('num_cin', $request->cin);
            });
        }

        $demandes = $query->paginate(10);

        $rows = $demandes->map(function($d) {
            $idPad = str_pad($d->id_demande, 3, '0', STR_PAD_LEFT);

            $cin = $d->utilisateur->num_cin ?? $d->utilisateur->cin ?? null;
            $cinDigits = $cin ? preg_replace('/\D+/', '', $cin) : '';
            $cinPad = $cinDigits ? str_pad($cinDigits, 9, '0', STR_PAD_LEFT) : '000000000';

            $year = $d->created_at ? $d->created_at->format('Y') : now()->format('Y');
            $num_dossier = "{$idPad}-{$cinPad}-{$year}";

            $even = $d->evenementSportifs->first() ?? null;
            $nom_evenement = $even->nom_even ?? null;
            $date_even = $even->date ?? null;
            $heure_even = $even->heure ?? null;

            $gouvernorat = $even?->gouvernorat?->name ?? $even?->gouvernorat?->gouvernorat ?? null;
            $delegation  = $even?->delegation?->name ?? $even?->delegation?->delegation ?? null;

            $nom_organisme = $d->organismeNonSportif ? ($d->organismeNonSportif->nom ?? $d->organismeNonSportif->raison_sociale ?? null) : null;

            return (object) [
                'id_demande' => $d->id_demande,
                'num_dossier' => $num_dossier,
                'nom_organisme' => $nom_organisme,
                'nom_evenement' => $nom_evenement,
                'date_even' => $date_even ? \Carbon\Carbon::parse($date_even)->format('Y-m-d') : null,
                'heure_even' => $heure_even ? \Carbon\Carbon::parse($heure_even)->format('H:i') : null,
                'gouvernorat' => $gouvernorat,
                'delegation' => $delegation,
                'statut' => $d->statut,
                'raw' => $d,
            ];
        });

        return view('backoffice.manifestation_sport.demande.index', compact('demandes', 'rows', 'typesIdentite'));
    }


    private function filteredQuery(Request $request)
{
    $query = DemandeManifestation::with([
        'utilisateur',
        'organismeNonSportif',
        'evenementSportifs.gouvernorat',
        'evenementSportifs.delegation',
        'statut'
    ])->where('type_formulaire_id', 12)
      ->orderByDesc('id_demande');

    if ($request->filled('date')) {
        $query->whereDate('created_at', $request->date);
    }

    if ($request->filled('type_identite') && $request->filled('cin')) {
        $query->whereHas('utilisateur', function($q) use ($request) {
            $q->where('type_identite_id', $request->type_identite)
              ->where('num_cin', $request->cin);
        });
    }

    return $query;
}

public function exportExcel(Request $request)
{
    $demandes = $this->filteredQuery($request)->get();

    $rows = $demandes->map(function ($d) {

        // ===============================
        // 🔹 رقم الملف
        // ===============================
        $idPad = str_pad($d->id_demande, 3, '0', STR_PAD_LEFT);

        $cin = $d->utilisateur->num_cin ?? $d->utilisateur->cin ?? null;
        $cinDigits = $cin ? preg_replace('/\D+/', '', $cin) : '';
        $cinPad = $cinDigits
            ? str_pad($cinDigits, 9, '0', STR_PAD_LEFT)
            : '000000000';

        $year = $d->created_at
            ? $d->created_at->format('Y')
            : now()->format('Y');

        $num_dossier = "{$idPad}-{$cinPad}-{$year}";

        // ===============================
        // 🔹 الحدث الرياضي
        // ===============================
        $even = $d->evenementSportifs->first();

        // ===============================
        // 🔹 الولاية / البلدية (✔ تصحيح)
        // ===============================
        $gouvernorat = $even?->gouvernorat?->nom_ar
            ?? $even?->gouvernorat?->name
            ?? $even?->gouvernorat?->gouvernorat
            ?? '-';

        $delegation = $even?->delegation?->nom_ar
            ?? $even?->delegation?->name
            ?? $even?->delegation?->delegation
            ?? '-';

        // ===============================
        // 🔹 اسم الهيكل
        // ===============================
        $nom_organisme = $d->organismeNonSportif->nom
            ?? $d->organismeNonSportif->raison_sociale
            ?? '-';

        return [
            $num_dossier,
            $nom_organisme,
            $even?->nom_even ?? '-',
            $even?->date
                ? \Carbon\Carbon::parse($even->date)->format('Y-m-d')
                : '-',
            $gouvernorat,
            $delegation,
            $d->statut?->statut_ar ?? 'في الانتظار',
        ];
    });

    // ===============================
    // 🔹 Excel
    // ===============================
    $spreadsheet = new Spreadsheet();
    $sheet = $spreadsheet->getActiveSheet();
    $sheet->setRightToLeft(true);

    $sheet->fromArray([
        'رقم الملف',
        'اسم الهيكل',
        'اسم التظاهرة',
        'التاريخ',
        'الولاية',
        'البلدية',
        'الحالة',
    ], null, 'A1');

    $sheet->getStyle('A1:G1')->getFont()->setBold(true);

    $rowNum = 2;
    foreach ($rows as $row) {
        $sheet->fromArray($row, null, 'A' . $rowNum++);
    }

    foreach (range('A', 'G') as $col) {
        $sheet->getColumnDimension($col)->setAutoSize(true);
    }

    $writer = new Xlsx($spreadsheet);

    return response()->streamDownload(
        fn () => $writer->save('php://output'),
        'manifestation.xlsx'
    );
}
public function exportPdf(Request $request)
{
    $demandes = $this->filteredQuery($request)->get();

    $rows = $demandes->map(function ($d) {

        $idPad = str_pad($d->id_demande, 3, '0', STR_PAD_LEFT);

        $cin = $d->utilisateur->num_cin ?? $d->utilisateur->cin ?? null;
        $cinDigits = $cin ? preg_replace('/\D+/', '', $cin) : '';
        $cinPad = $cinDigits
            ? str_pad($cinDigits, 9, '0', STR_PAD_LEFT)
            : '000000000';

        $year = $d->created_at
            ? $d->created_at->format('Y')
            : now()->format('Y');

        $num_dossier = "{$idPad}-{$cinPad}-{$year}";

        $even = $d->evenementSportifs->first();

        // ✔ تصحيح الولاية / البلدية
        $gouvernorat = $even?->gouvernorat?->nom_ar
            ?? $even?->gouvernorat?->name
            ?? $even?->gouvernorat?->gouvernorat
            ?? '-';

        $delegation = $even?->delegation?->nom_ar
            ?? $even?->delegation?->name
            ?? $even?->delegation?->delegation
            ?? '-';

        $nom_organisme = $d->organismeNonSportif->nom
            ?? $d->organismeNonSportif->raison_sociale
            ?? '-';

        return (object)[
            'num_dossier'   => $num_dossier,
            'nom_organisme' => $nom_organisme,
            'nom_evenement' => $even?->nom_even ?? '-',
            'date_even'     => $even?->date
                ? \Carbon\Carbon::parse($even->date)->format('Y-m-d')
                : '-',
            'heure_even'    => $even?->heure
                ? \Carbon\Carbon::parse($even->heure)->format('H:i')
                : '-',
            'gouvernorat'   => $gouvernorat,
            'delegation'    => $delegation,
            'statut'        => $d->statut,
        ];
    });

    $html = view(
        'backoffice.manifestation_sport.demande.exports.manifestation_pdf',
        compact('rows')
    )->render();

    $mpdf = new \Mpdf\Mpdf([
        'mode' => 'utf-8',
        'format' => 'A4',
        'default_font' => 'dejavusans',
        'directionality' => 'rtl',
    ]);

    $mpdf->WriteHTML($html);

    return response(
        $mpdf->Output('manifestation.pdf', 'S')
    )->header('Content-Type', 'application/pdf');
}


    public function create()
    {
        $formulaire   = TypeFormulaire::find(12);
        $attachments  = TpAttachement::orderBy('type_ar', 'asc')->get();
        $formulaire2  = TypeFormulaire::find(14);
        $formulaire3  = TypeFormulaire::find(15);
        $gouvernorats = Gouvernorat::all();
        $delegations  = Delegation::all();
        $formulaire4  = TypeFormulaire::find(16);
        $type_membre  = TpTypeMembre::all();
        $statut       = TpStatut::all();
        $formulaire5  = TypeFormulaire::find(17);
        $formulaire6  = TypeFormulaire::find(23);
        $formulaire7  = TypeFormulaire::find(18);
        

        return view('backoffice.manifestation_sport.demande.create', compact('formulaire', 'attachments','formulaire2','formulaire3',
            'gouvernorats','delegations','type_membre','statut','formulaire4','formulaire5','formulaire6','formulaire7'));
    }

    public function store(Request $request)
    {
        
        $request->validate([
            //demande
            'sujet' => 'nullable|string|max:255',
            'sport_type' => 'nullable|string|max:255',
            'event_name' => 'nullable|string|max:255',
            'objectif' => 'nullable|string',
            'commentaire_motif' => 'nullable|string',
            'type_formulaire_id' => 'required|integer',
            'user_id' => 'required|integer',
            'id_type' => 'nullable|integer',
            'organisme_non_sportif_id' => 'nullable|integer',
            'role' => 'nullable|string',
            'periode_min' => 'nullable|date',
            'periode_max' => 'nullable|date|after_or_equal:periode_min',
            'lieu' => 'nullable|string|max:255',

            // Organisme non sportif
            'org.nom' => 'required|string|max:255',
            'org.num_visa' => 'nullable|integer',
            'org.date_enregistrement' => 'nullable|date',
            'org.num_enregistrement' => 'nullable|string|max:50',
            'org.identifiant_fiscal' => 'nullable|int',
            'org.num_compte_bancaire' => 'nullable|string|max:50',
            'org.adresse' => 'nullable|string|max:255',
            'org.tel' => 'nullable|string|max:20',
            'org.fax' => 'nullable|string|max:20',
            'org.email' => 'nullable|email|max:255',
            'org.type_formulaire_id' => 'nullable|integer',

            // EvenementSportif
            'nom_even' => 'required|string|max:100',
            'objectifs' => 'nullable|string',
            'contenus' => 'nullable|string',
            'date' => 'required|date',
            'heure' => 'nullable|date_format:H:i',
            'instalation_sportives' => 'nullable|string|max:200',
            'reference' => 'nullable|string|max:100',
            'estimation_evenement' => 'nullable|numeric',
            'cout_total_evenement' => 'nullable|numeric',
            'difference_estimee' => 'nullable|numeric',
            'recompense' => 'nullable|string',
            'moyen_transport' => 'nullable|string|max:200',
            'user_id' => 'required|integer',
            'id_type' => 'nullable|integer',
            'id_formulaire' => 'nullable|integer',
            'gouvernorat_id' => 'required|integer',
            'delegation_id' => 'required|integer',
            'demande_id' => 'nullable|integer',

            // MembreOrgInvite
            'membres.*.nom_prenom' => 'required|string|max:255',
            'membres.*.role' => 'nullable|string',
            'membres.*.nationalite' => 'nullable|string',
            'membres.*.type_membre_id' => 'nullable|integer',

            //MembreOrgInvite2
            'invites.*.nom_prenom' => 'required|string|max:255',
            'invites.*.role' => 'nullable|string',
            'invites.*.nationalite' => 'nullable|string',
            'invites.*.type_membre_id' => 'nullable|integer',

            // AssocParticipante
            'assoc_participantes.*.nom' => 'required|string|max:255',
            'assoc_participantes.*.num_athlete_h' => 'nullable|integer',
            'assoc_participantes.*.num_athlete_f' => 'nullable|integer',
            'assoc_participantes.*.accompagnants' => 'nullable|integer',
            'assoc_participantes.*.total' => 'nullable|integer',

            // AthleteIndividuel
            'athletes.*.nom_prenom_sportif' => 'nullable|string|max:255',
            'athletes.*.num_athlete_h' => 'nullable|integer',
            'athletes.*.num_athlete_f' => 'nullable|integer',
            'athletes.*.accompagnants' => 'nullable|integer',
            'athletes.*.total' => 'nullable|integer',

            // SportifParticipant
            'sportifs.*.nom_prenom' => 'required|string|max:255',
            'sportifs.*.tranch_age' => 'nullable|string|max:50',
            'sportifs.*.date_naissance' => 'nullable|date',
            'sportifs.*.num_cin' => 'nullable|string|max:50',
            'sportifs.*.invitee' => 'nullable|string|max:255',
            'sportifs.*.jeune_id' => 'nullable|integer',

            // Attachements
            'assurance_file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
            'technique_file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
            'accord_file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
            'demande_file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
            
        ]);

        //OrganismeNonSportif
        $organisme = null;
        if ($request->has('org')) {
            $organismeData = $request->input('org');
            $organismeData['type_formulaire_id'] = 14; 
            $organisme = OrganismeNonSportif::create($organismeData);
        }

        // DemandeManifestation
        $demande = DemandeManifestation::create([
            'sujet' => $request->sujet,
            'sport_type' => $request->sport_type,
            'event_name' => $request->event_name,
            'objectif' => $request->objectif,
            'commentaire_motif' => $request->commentaire_motif,
            'type_formulaire_id' => 12,
            'user_id' => Auth::id(),
            'id_type' => 2,
            'organisme_non_sportif_id' => $organisme?->id_org,
            'role' => $request->role,
            'periode_min' => $request->periode_min,
            'periode_max' => $request->periode_max,
            'lieu' => $request->lieu,
        ]);

        

        if ($request->hasFile('assurance_file')) {
            $filePath = $request->file('assurance_file')->store('manifestations/', 'public');

            Attachement::create([
                'id_type_attach' => 2, 
                'file' => $filePath,
                'id_demande' => $demande->id_demande,
            ]);
        }

        if ($request->hasFile('accord_file')) {
            $filePath = $request->file('accord_file')->store('manifestations/', 'public');

            Attachement::create([
                'id_type_attach' => 7, 
                'file' => $filePath,
                'id_demande' => $demande->id_demande,
            ]);
        }

        if ($request->hasFile('technique_file')) {
            $filePath = $request->file('technique_file')->store('manifestations/', 'public');

            Attachement::create([
                'id_type_attach' => 8, 
                'file' => $filePath,
                'id_demande' => $demande->id_demande,
            ]);
        }

        if ($request->hasFile('demande_file')) {
            $filePath = $request->file('demande_file')->store('manifestations/', 'public');

            Attachement::create([
                'id_type_attach' => 9, 
                'file' => $filePath,
                'id_demande' => $demande->id_demande,
            ]);
        }

        // EvenementSportif
        if ($request->has('nom_even')) {
            $evenement = EvenementSportif::create([
                'nom_even' => $request->nom_even,
                'objectifs' => $request->objectifs,
                'contenus' => $request->contenus,
                'date' => $request->date,
                'heure' => $request->heure,
                'instalation_sportives' => $request->instalation_sportives,
                'reference' => $request->reference,
                'estimation_evenement' => $request->estimation_evenement,
                'cout_total_evenement' => $request->cout_total_evenement,
                'difference_estimee' => $request->difference_estimee,
                'recompense' => $request->recompense,
                'moyen_transport' => $request->moyen_transport,
                'user_id' => Auth::id(),
                'id_type' => 2,
                'id_formulaire' => 15,
                'gouvernorat_id' => $request->gouvernorat_id,
                'delegation_id' => $request->delegation_id,
                'demande_id' => $demande->id_demande,
            ]);

            // MembreOrgInvite
            if ($request->has('membres')) {
                foreach ($request->membres as $membre) {
                    MembreOrgInvite::create([
                        'nom_prenom' => $membre['nom_prenom'],
                        'role' => $membre['role'] ?? null,
                        'nationalite' => $membre['nationalite'] ?? null,
                        'event_id' => $evenement->id_even,
                        'type_membre_id' => 1,
                    ]);
                }
            }

            // MembreOrgInvite2
            if ($request->has('invites')) {
                foreach ($request->invites as $invite) {
                    MembreOrgInvite::create([
                        'nom_prenom' => $invite['nom_prenom'],
                        'role' => $invite['role'] ?? null,
                        'nationalite' => $invite['nationalite'] ?? null,
                        'event_id' => $evenement->id_even,
                        'type_membre_id' => 2,
                    ]);
                }
            }

            //AssocParticipante
            if ($request->has('assoc_participantes')) {
                foreach ($request->assoc_participantes as $assoc) {
                    // On calcule le total directement ici
                    $num_h = isset($assoc['num_athlete_h']) ? (int) $assoc['num_athlete_h'] : 0;
                    $num_f = isset($assoc['num_athlete_f']) ? (int) $assoc['num_athlete_f'] : 0;
                    $accomp = isset($assoc['accompagnants']) ? (int) $assoc['accompagnants'] : 0;
                    $total = $num_h + $num_f + $accomp;

                    AssocParticipante::create([
                        'nom' => $assoc['nom'] ?? null,
                        'num_athlete_h' => $num_h,
                        'num_athlete_f' => $num_f,
                        'accompagnants' => $accomp,
                        'total' => $total,
                        'event_id' => $evenement->id_even,
                    ]);
                }
            }

            //AthleteIndividuel
            if ($request->has('athletes')) {
                foreach ($request->athletes as $athlete) {
                    AthleteIndividuel::create([
                        'nom_prenom_sportif' => $athlete['nom_prenom_sportif'],
                        'num_athlete_h' => $athlete['num_athlete_h'] ?? 0,
                        'num_athlete_f' => $athlete['num_athlete_f'] ?? 0,
                        'accompagnants' => $athlete['accompagnants'] ?? 0,
                        'event_id' => $evenement->id_even,
                    ]);
                }
            }

            //SportifParticipant
            if ($request->has('sportifs')) {
                foreach ($request->sportifs as $sportif) {
                    SportifParticipant::create([
                        'nom_prenom' => $sportif['nom_prenom'],
                        'tranch_age' => $sportif['tranch_age'] ?? null,
                        'date_naissance' => $sportif['date_naissance'] ?? null,
                        'num_cin' => $sportif['num_cin'] ?? null,
                        'invitee' => $sportif['invitee'] ?? null,
                        'even_id' => $evenement->id_even,
                        'jeune_id' => $sportif['jeune_id'] ?? null,
                    ]);
                }
            }
        }

        return redirect()->route('admin.manifestation.index')
                    ->with('success', 'تم تسجيل الطلب بنجاح');
    }

public function show($id)
{
    // Nettoyer les anciennes données de session pour éviter les conflits
    session()->forget('_old_input');
    
    // Charger la demande avec toutes les relations nécessaires
    $demande = DemandeManifestation::with([
        'organismeNonSportif',
        'evenementSportifs' => function($query) {
            $query->with([
                'membres' => function($q) {
                    $q->orderBy('type_membre_id')->orderBy('created_at');
                },
                'assocParticipantes',
                'athletesIndividuels',
                'sportifs',
                'gouvernorat',
                'delegation'
            ]);
        },
        'attachements.type',
        'statutHistories' => function($query) {
            $query->with(['fromStatut', 'toStatut', 'changer'])
                  ->orderBy('created_at', 'desc');
        },
        'assuranceFile',
        'accordFile',
        'techniqueFile',
        'demandeFile',
        'utilisateur'
    ])->findOrFail($id);

    // Récupérer l'organisme associé à la demande (pas avec $id directement)
    $org = $demande->organismeNonSportif ?? new OrganismeNonSportif();
    
    // Récupérer l'événement sportif (premier de la collection)
    $evenement = $demande->evenementSportifs->first();
    
    // Récupérer les membres organisateurs et invités séparément
    $membres = collect();
    $invites = collect();
    $assocParticipantes = collect();
    $athletesIndividuels = collect();
    $sportifs = collect();
    
    if ($evenement) {
        $membres = $evenement->membres->where('type_membre_id', 1)->values();
        $invites = $evenement->membres->where('type_membre_id', 2)->values();
        $assocParticipantes = $evenement->assocParticipantes ?? collect();
        $athletesIndividuels = $evenement->athletesIndividuels ?? collect();
        $sportifs = $evenement->sportifs ?? collect();
    }

    // Charger les données de référence
    $formulaire   = TypeFormulaire::find(12);  // Formulaire principal
    $formulaire2  = TypeFormulaire::find(14);  // Organisme non sportif
    $formulaire3  = TypeFormulaire::find(15);  // Événement sportif
    $formulaire4  = TypeFormulaire::find(16);  // Membres organisation
    $formulaire5  = TypeFormulaire::find(17);  // Associations participantes
    $formulaire6  = TypeFormulaire::find(23);  // Athlètes individuels
    $formulaire7  = TypeFormulaire::find(18);  // Sportifs participants
    
    $attachments  = TpAttachement::orderBy('type_ar', 'asc')->get();
    $gouvernorats = Gouvernorat::all();
    $delegations  = Delegation::all();
    $type_membre  = TpTypeMembre::all();
    $statut       = TpStatut::orderBy('id_statut')->get();
    $statuts      = TpStatut::orderBy('id_statut')->get();

    // Vérifier les données de délégation pour l'événement
    if ($evenement && $evenement->gouvernorat_id) {
        $delegations = Delegation::where('gouver_id', $evenement->gouvernorat_id)->get();
    }

    return view('backoffice.manifestation_sport.demande.show', compact(
        'demande',
        'org',
        'evenement',
        'membres',
        'invites',
        'assocParticipantes',
        'athletesIndividuels',
        'sportifs',
        'formulaire',
        'formulaire2',
        'formulaire3',
        'formulaire4',
        'formulaire5',
        'formulaire6',
        'formulaire7',
        'attachments',
        'gouvernorats',
        'delegations',
        'type_membre',
        'statut',
        'statuts'
    ));
}


       public function pdf($id)
{
    ini_set('pcre.backtrack_limit', '10000000');
    ini_set('pcre.recursion_limit', '10000000');
    ini_set('memory_limit', '512M');

    $demande = DemandeManifestation::with([
        'organismeNonSportif',
        'evenementSportifs',
        'evenementSportifs.membres',
        'evenementSportifs.invites',
        'evenementSportifs.assocParticipantes',
        'evenementSportifs.athletesIndividuels',
        'evenementSportifs.sportifs',
        'attachements'
    ])->findOrFail($id);

    //$org           = OrganismeNonSportif::find($id);
    $org = $demande->organismeNonSportif;
    $formulaire    = TypeFormulaire::find(12);
    $attachments   = TpAttachement::orderBy('type_ar', 'asc')->get();
    $formulaire2   = TypeFormulaire::find(14);
    $formulaire3   = TypeFormulaire::find(15);
    $gouvernorats  = Gouvernorat::all();
    $delegations   = Delegation::all();
    $formulaire4   = TypeFormulaire::find(16);
    $type_membre   = TpTypeMembre::all();
    $statut        = TpStatut::all();
    $formulaire5   = TypeFormulaire::find(17);
    $formulaire6   = TypeFormulaire::find(23);
    $formulaire7   = TypeFormulaire::find(18);

    // Définir la langue et la direction
   $lang = 'ar';
$direction = 'rtl';

    $html = view(
        'backoffice.manifestation_sport.demande.pdf',
        compact(
            'demande','formulaire','attachments','formulaire2','formulaire3',
            'gouvernorats','delegations','type_membre','statut','formulaire4',
            'formulaire5','formulaire6','formulaire7','org',
            'lang','direction' // <-- ajouter ici
        )
    )->render();

    $mpdf = new Mpdf([
        'mode'              => 'utf-8',
        'format'            => 'A4',
        'default_font'      => 'dejavusans',
        'directionality'    => $direction,
        'autoScriptToLang'  => true,
        'autoLangToFont'    => true,
        'tempDir'           => storage_path('app/mpdf'),
    ]);

    $mpdf->setFooter('{PAGENO} / {nbpg}');
    $mpdf->WriteHTML($html);

    return response(
        $mpdf->Output('demande_manifestation_'.$id.'.pdf', 'S'),
        200,
        [
            'Content-Type'        => 'application/pdf',
            'Content-Disposition' => 'inline; filename="demande_manifestation_'.$id.'.pdf"',
        ]
    );
}


    /*public function edit($id)
    {
        $demande = DemandeManifestation::with([
            'organismeNonSportif',
            'evenementSportifs',
            'evenementSportifs.membres',
            'evenementSportifs.invites',
            'evenementSportifs.assocParticipantes',
            'evenementSportifs.athletesIndividuels',
            'evenementSportifs.sportifs',
            'attachements',
            'statutHistories.fromStatut',
            'statutHistories.toStatut',
            'statutHistories.changer'
        ])->findOrFail($id);

        $org = OrganismeNonSportif::find($id);
        $formulaire   = TypeFormulaire::find(12);
        $attachments  = TpAttachement::orderBy('type_ar', 'asc')->get();
        $formulaire2  = TypeFormulaire::find(14);
        $formulaire3  = TypeFormulaire::find(15);
        $gouvernorats = Gouvernorat::all();
        $delegations  = Delegation::all();
        $formulaire4  = TypeFormulaire::find(16);
        $type_membre  = TpTypeMembre::all();
        $statut       = TpStatut::all();
        $formulaire5  = TypeFormulaire::find(17);
        $formulaire6  = TypeFormulaire::find(23);
        $formulaire7  = TypeFormulaire::find(18);
        $statuts = TpStatut::orderBy('id_statut')->get();
        return view('backoffice.manifestation_sport.demande.edit', compact(
            'demande', 'formulaire', 'attachments','formulaire2','formulaire3',
            'gouvernorats','delegations','type_membre','statut','formulaire4',
            'formulaire5','formulaire6','formulaire7','org','statuts'
        ));
    }*/

public function edit($id)
{
    // Nettoyer les anciennes données de session pour éviter les conflits
    session()->forget('_old_input');
    
    // Charger la demande avec toutes les relations nécessaires
    $demande = DemandeManifestation::with([
        'organismeNonSportif',
        'evenementSportifs' => function($query) {
            $query->with([
                'membres' => function($q) {
                    $q->orderBy('type_membre_id')->orderBy('created_at');
                },
                'assocParticipantes',
                'athletesIndividuels',
                'sportifs',
                'gouvernorat',
                'delegation'
            ]);
        },
        'attachements.type',
        'statutHistories' => function($query) {
            $query->with(['fromStatut', 'toStatut', 'changer'])
                  ->orderBy('created_at', 'desc');
        },
        'assuranceFile',
        'accordFile',
        'techniqueFile',
        'demandeFile',
        'utilisateur'
    ])->findOrFail($id);

    // Récupérer l'organisme associé à la demande (pas avec $id directement)
    $org = $demande->organismeNonSportif ?? new OrganismeNonSportif();
    
    // Récupérer l'événement sportif (premier de la collection)
    $evenement = $demande->evenementSportifs->first();
    
    // Récupérer les membres organisateurs et invités séparément
    $membres = collect();
    $invites = collect();
    $assocParticipantes = collect();
    $athletesIndividuels = collect();
    $sportifs = collect();
    
    if ($evenement) {
        $membres = $evenement->membres->where('type_membre_id', 1)->values();
        $invites = $evenement->membres->where('type_membre_id', 2)->values();
        $assocParticipantes = $evenement->assocParticipantes ?? collect();
        $athletesIndividuels = $evenement->athletesIndividuels ?? collect();
        $sportifs = $evenement->sportifs ?? collect();
    }

    // Charger les données de référence
    $formulaire   = TypeFormulaire::find(12);  // Formulaire principal
    $formulaire2  = TypeFormulaire::find(14);  // Organisme non sportif
    $formulaire3  = TypeFormulaire::find(15);  // Événement sportif
    $formulaire4  = TypeFormulaire::find(16);  // Membres organisation
    $formulaire5  = TypeFormulaire::find(17);  // Associations participantes
    $formulaire6  = TypeFormulaire::find(23);  // Athlètes individuels
    $formulaire7  = TypeFormulaire::find(18);  // Sportifs participants
    
    $attachments  = TpAttachement::orderBy('type_ar', 'asc')->get();
    $gouvernorats = Gouvernorat::all();
    $delegations  = Delegation::all();
    $type_membre  = TpTypeMembre::all();
    $statut       = TpStatut::orderBy('id_statut')->get();
    $statuts      = TpStatut::orderBy('id_statut')->get();

    // Vérifier les données de délégation pour l'événement
    if ($evenement && $evenement->gouvernorat_id) {
        $delegations = Delegation::where('gouver_id', $evenement->gouvernorat_id)->get();
    }

    return view('backoffice.manifestation_sport.demande.edit', compact(
        'demande',
        'org',
        'evenement',
        'membres',
        'invites',
        'assocParticipantes',
        'athletesIndividuels',
        'sportifs',
        'formulaire',
        'formulaire2',
        'formulaire3',
        'formulaire4',
        'formulaire5',
        'formulaire6',
        'formulaire7',
        'attachments',
        'gouvernorats',
        'delegations',
        'type_membre',
        'statut',
        'statuts'
    ));
}


    /**
 * Envoie mail + notification selon le statut
 */
private function envoyerNotificationEtMail(DemandeManifestation $demande, $nouveauStatut)
{
    $date = now()->format('Y-m-d');
    $statut_ar = $nouveauStatut == 1 ? 'قبول' : 'رفض';

    $idPad = str_pad($demande->id_demande, 3, '0', STR_PAD_LEFT);
    $cin = $demande->utilisateur->num_cin ?? $demande->utilisateur->cin ?? null;
    $cinDigits = $cin ? preg_replace('/\D+/', '', $cin) : '';
    $cinPad = $cinDigits ? str_pad($cinDigits, 9, '0', STR_PAD_LEFT) : '000000000';
    $year = $demande->created_at ? $demande->created_at->format('Y') : now()->format('Y');
    $num_dossier = "{$idPad}-{$cinPad}-{$year}";
    $nomUtilisateur = $demande->utilisateur->nom_prenom ?? 'المستخدم';

    if ($nouveauStatut == 1) {
        $messageStatus = "مرحباً <strong>{$nomUtilisateur}</strong>،<br>" .
            "رقم الملف: <strong>{$num_dossier}</strong><br>" .
            "تم قبول طلبكم \" طلب في اقامة تظاهرة رياضية وطنية بالبالد التونسية \" ، بتاريخ: <strong>{$date}</strong>.<br>نهنئكم على هذا القبول.";
    } else {
        $messageStatus = "مرحباً <strong>{$nomUtilisateur}</strong>،<br>" .
            "رقم الملف: <strong>{$num_dossier}</strong><br>" .
            "تم رفض طلبكم \" طلب في اقامة تظاهرة رياضية وطنية بالبالد التونسية \" ، بتاريخ: <strong>{$date}</strong>.<br>" .
            "التعليق: {$demande->commentaire_motif}{$demande->commentaire_motif2}{$demande->commentaire_motif3}.";
    }

    if ($demande->utilisateur?->email) {
        Mail::to($demande->utilisateur->email)->send(
            new \App\Mail\StatutDemandeMail($demande->utilisateur, $num_dossier, $messageStatus)
        );
    }

    if ($demande->utilisateur) {
        $demande->utilisateur->notify(
            new \App\Notifications\StatutDemandeNotification($num_dossier, $messageStatus)
        );
    }
}





/*
public function update(Request $request, $id)
{
    $request->validate([
        'sujet' => 'nullable|string|max:255',
        'sport_type' => 'nullable|string|max:255',
        'event_name' => 'nullable|string|max:255',
        'objectif' => 'nullable|string',
        'commentaire_motif' => 'nullable|string',
        'type_formulaire_id' => 'required|integer',
        'statut_id' => 'nullable|integer',
        'statut2_id' => 'nullable|integer',
        'commentaire_motif2' => 'nullable|string|max:255',
        'commentaire_motif3' => 'nullable|string|max:255',
        'file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:4096',
        'id_type' => 'nullable|integer',
        'organisme_non_sportif_id' => 'nullable|integer',
        'role' => 'nullable|string',
        'periode_min' => 'nullable|date',
        'periode_max' => 'nullable|date|after_or_equal:periode_min',
        'lieu' => 'nullable|string|max:255',

        // Organisme non sportif : valider org comme array
        'org' => 'nullable|array',
        'org.nom' => 'nullable|string|max:255',
        'org.num_visa' => 'nullable|integer',
        'org.date_enregistrement' => 'nullable|date',
        'org.num_enregistrement' => 'nullable|string|max:50',
        'org.identifiant_fiscal' => 'nullable|numeric',
        'org.num_compte_bancaire' => 'nullable|string|max:50',
        'org.adresse' => 'nullable|string|max:255',
        'org.tel' => 'nullable|string|max:20',
        'org.fax' => 'nullable|string|max:20',
        'org.email' => 'nullable|email|max:255',
        'org.type_formulaire_id' => 'nullable|integer',

        // EvenementSportif
        'nom_even' => 'required_with:date|string|max:100',
        'objectifs' => 'nullable|string',
        'contenus' => 'nullable|string',
        'date' => 'required_with:nom_even|date',
        'heure' => 'nullable|date_format:H:i',
        'instalation_sportives' => 'nullable|string|max:200',
        'reference' => 'nullable|string|max:100',
        'estimation_evenement' => 'nullable|numeric',
        'cout_total_evenement' => 'nullable|numeric',
        'difference_estimee' => 'nullable|numeric',
        'recompense' => 'nullable|string',
        'moyen_transport' => 'nullable|string|max:200',
        'gouvernorat_id' => 'required_with:nom_even|integer',
        'delegation_id' => 'required_with:nom_even|integer',

        // collections
        'membres.*.nom_prenom' => 'required_with:membres|string|max:255',
        'invites.*.nom_prenom' => 'required_with:invites|string|max:255',
        'assoc_participantes.*.nom' => 'required_with:assoc_participantes|string|max:255',
        'athletes.*.nom_prenom_sportif' => 'nullable|string|max:255',
        'sportifs.*.nom_prenom' => 'required_with:sportifs|string|max:255',

        // attachements
        'assurance_file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
        'technique_file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
        'accord_file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
        'demande_file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
    ]);

    DB::beginTransaction();

    try {
        $demande = DemandeManifestation::findOrFail($id);

        $ancienStatut = $demande->getOriginal('id_type');
        $ancienStatut1 = $demande->getOriginal('statut_id');
        $ancienStatut2 = $demande->getOriginal('statut2_id');

        $organisme = null;
        $orgInput = $request->input('org', null);

        if (is_array($orgInput) && count(array_filter($orgInput)) > 0) {
            $orgInput['type_formulaire_id'] = $orgInput['type_formulaire_id'] ?? 14;

            if ($demande->organisme_non_sportif_id) {
                $organisme = OrganismeNonSportif::find($demande->organisme_non_sportif_id);
                if ($organisme) {
                    $organisme->fill($orgInput);
                    $organisme->save();
                } else {
                    $organisme = OrganismeNonSportif::create($orgInput);
                }
            } else {
                $organisme = OrganismeNonSportif::create($orgInput);
            }

            $pkName = $organisme->getKeyName();
            $demande->organisme_non_sportif_id = $organisme->{$pkName};
        }

        
		
		
        $uploadDir = '/home/preprov/www/mjs/shared_uploads/images';
        if (!is_dir($uploadDir) || !is_writable($uploadDir)) {
            abort(500, 'Dossier upload non accessible');
        }

        $filePath = $demande->file; 
        $fileName = $demande->file_name ?? null;
        $fileType = $demande->file_type ?? null;

        if ($request->hasFile('file')) {
            $file = $request->file('file');
            
            if (!empty($demande->file) && file_exists($demande->file)) {
                @unlink($demande->file);
            }
            
            $original = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
            $ext = $file->getClientOriginalExtension();
            $safe = Str::slug(mb_substr($original, 0, 40)) ?: 'document';
            
            $filename = 'demande_' . $demande->id_demande . '_' . $safe . '-' . time() . '-' . Str::random(6) . '.' . $ext;
            $targetPath = $uploadDir . DIRECTORY_SEPARATOR . $filename;
            
            $file->move($uploadDir, $filename);
            
            $realPath = realpath($targetPath);
            if ($realPath === false) {
                abort(500, 'Erreur lors de la sauvegarde du fichier');
            }
            
            $filePath = $realPath;
            $fileName = $file->getClientOriginalName();
            $fileType = $file->getMimeType();
        }
		

        $commentForHistory = null;
        if ($request->filled('commentaire_motif')) {
            $commentForHistory = trim($request->input('commentaire_motif'));
        } elseif ($request->filled('commentaire_motif2')) {
            $commentForHistory = trim($request->input('commentaire_motif2'));
        } elseif ($request->filled('commentaire_motif3')) {
            $commentForHistory = trim($request->input('commentaire_motif3'));
        }

        $demande->update([
            'sujet' => $request->sujet,
            'sport_type' => $request->sport_type,
            'event_name' => $request->event_name,
            'objectif' => $request->objectif,
            'commentaire_motif' => $request->commentaire_motif,
            'commentaire_motif2' => $request->commentaire_motif2,
            'commentaire_motif3' => $request->commentaire_motif3,
            'type_formulaire_id' => $request->type_formulaire_id ?? $demande->type_formulaire_id,
            //'user_id' => Auth::id(),
            'user_id' => $demande->user_id,
            'id_type' => $request->id_type ?? $demande->id_type,

            'statut_id' => $request->statut_id ?? $demande->statut_id,
            'statut2_id' => $request->statut2_id ?? $demande->statut2_id,
            'role' => $request->role,
            'periode_min' => $request->periode_min,
            'periode_max' => $request->periode_max,
            'lieu' => $request->lieu,
            'organisme_non_sportif_id' => $demande->organisme_non_sportif_id,
            
			
			'file' => $filePath,
            'file_name' => $fileName,
            'file_type' => $fileType,
        ]);

        // Normalize integers helper
        $normalize = function ($v) {
            if (is_null($v) || $v === '') return null;
            $i = (int)$v;
            return ($i > 0) ? $i : null;
        };

        // compute old and new values for comparison
        $old_id_type = $normalize($ancienStatut);
        $old_statut_id = $normalize($ancienStatut1);
        $old_statut2_id = $normalize($ancienStatut2);

        // reload fresh model to get saved values
        $demande->refresh();

        $new_id_type = $normalize($demande->id_type);
        $new_statut_id = $normalize($demande->statut_id);
        $new_statut2_id = $normalize($demande->statut2_id);

        // helper to check tp_statut existence
        $statExists = function ($id) {
            if (is_null($id)) return false;
            return DB::table('tp_statut')->where('id_statut', $id)->exists();
        };

        // create a history row for each changed slot (use demande_manifestation_id)
        if ($old_id_type !== $new_id_type) {
            ImmobilierStatutHistory::create([
                'demande_manifestation_id' => $demande->id_demande,
                'from_statut_id'           => $old_id_type,
                'to_statut_id'             => $statExists($new_id_type) ? $new_id_type : null,
                'changed_by'               => Auth::id() ?? null,
                'commentaire'              => $commentForHistory,
            ]);
        }

        if ($old_statut_id !== $new_statut_id) {
            ImmobilierStatutHistory::create([
                'demande_manifestation_id' => $demande->id_demande,
                'from_statut_id'           => $old_statut_id,
                'to_statut_id'             => $statExists($new_statut_id) ? $new_statut_id : null,
                'changed_by'               => Auth::id() ?? null,
                'commentaire'              => $commentForHistory,
            ]);
        }

        if ($old_statut2_id !== $new_statut2_id) {
            ImmobilierStatutHistory::create([
                'demande_manifestation_id' => $demande->id_demande,
                'from_statut_id'           => $old_statut2_id,
                'to_statut_id'             => $statExists($new_statut2_id) ? $new_statut2_id : null,
                'changed_by'               => Auth::id() ?? null,
                'commentaire'              => $commentForHistory,
            ]);
        }

        // Determine the "resolved" statut to notify (prefer request values)
        $nouveauStatut = $request->id_type ?? $request->statut_id ?? $request->statut2_id;

        
        $attachmentsMap = [
            'assurance_file' => 2,
            'accord_file'    => 7,
            'technique_file' => 8,
            'demande_file'   => 9,
        ];

        foreach ($attachmentsMap as $inputName => $typeId) {
            if ($request->hasFile($inputName)) {
                $f = $request->file($inputName);
                
                $oldAttachment = Attachement::where('id_demande', $demande->id_demande)
                    ->where('id_type_attach', $typeId)
                    ->first();

                if ($oldAttachment && !empty($oldAttachment->file) && file_exists($oldAttachment->file)) {
                    @unlink($oldAttachment->file);
                }

                $original = pathinfo($f->getClientOriginalName(), PATHINFO_FILENAME);
                $ext = $f->getClientOriginalExtension();
                $safe = Str::slug(mb_substr($original, 0, 40)) ?: 'document';
                
                $filename = 'attachement_' . $demande->id_demande . '_' . $typeId . '_' . $safe . '-' . time() . '-' . Str::random(6) . '.' . $ext;
                $targetPath = $uploadDir . DIRECTORY_SEPARATOR . $filename;
                
                $f->move($uploadDir, $filename);
                
                $realPath = realpath($targetPath);
                if ($realPath === false) {
                    abort(500, 'Erreur lors de la sauvegarde du fichier');
                }

                if ($oldAttachment) {
                    $oldAttachment->update([
                        'file' => $realPath, 
                        'mime_type' => $f->getMimeType(),
                    ]);
                } else {
                    Attachement::create([
                        'id_demande'     => $demande->id_demande,
                        'id_type_attach' => $typeId,
                        'file'           => $realPath, 
                        'mime_type'      => $f->getMimeType(),
                    ]);
                }
            }
        }
		

        // Evenement sportif logic (unchanged)
        $evenement = EvenementSportif::where('demande_id', $demande->id_demande)->first();

        if ($request->has('nom_even')) {
            $evenData = [
                'nom_even' => $request->nom_even,
                'objectifs' => $request->objectifs,
                'contenus' => $request->contenus,
                'date' => $request->date,
                'heure' => $request->heure,
                'instalation_sportives' => $request->instalation_sportives,
                'reference' => $request->reference,
                'estimation_evenement' => $request->estimation_evenement,
                'cout_total_evenement' => $request->cout_total_evenement,
                'difference_estimee' => $request->difference_estimee,
                'recompense' => $request->recompense,
                'moyen_transport' => $request->moyen_transport,
                'user_id' => Auth::id(),
                'id_type' => $request->id_type ?? 2,
                'id_formulaire' => $request->id_formulaire ?? 15,
                'gouvernorat_id' => $request->gouvernorat_id,
                'delegation_id' => $request->delegation_id,
                'demande_id' => $demande->id_demande,
            ];

            if ($evenement) {
                $evenement->update($evenData);
            } else {
                $evenement = EvenementSportif::create($evenData);
            }

            MembreOrgInvite::where('event_id', $evenement->id_even)->delete();
            AssocParticipante::where('event_id', $evenement->id_even)->delete();
            AthleteIndividuel::where('event_id', $evenement->id_even)->delete();
            SportifParticipant::where('even_id', $evenement->id_even)->delete();

            if ($request->has('membres')) {
                foreach ($request->membres as $membre) {
                    MembreOrgInvite::create([
                        'nom_prenom' => $membre['nom_prenom'],
                        'role' => $membre['role'] ?? null,
                        'nationalite' => $membre['nationalite'] ?? null,
                        'event_id' => $evenement->id_even,
                        'type_membre_id' => 1,
                    ]);
                }
            }

            if ($request->has('invites')) {
                foreach ($request->invites as $invite) {
                    MembreOrgInvite::create([
                        'nom_prenom' => $invite['nom_prenom'],
                        'role' => $invite['role'] ?? null,
                        'nationalite' => $invite['nationalite'] ?? null,
                        'event_id' => $evenement->id_even,
                        'type_membre_id' => 2,
                    ]);
                }
            }

            if ($request->has('assoc_participantes')) {
                foreach ($request->assoc_participantes as $assoc) {
                    $num_h = isset($assoc['num_athlete_h']) ? (int)$assoc['num_athlete_h'] : 0;
                    $num_f = isset($assoc['num_athlete_f']) ? (int)$assoc['num_athlete_f'] : 0;
                    $accomp = isset($assoc['accompagnants']) ? (int)$assoc['accompagnants'] : 0;
                    $total = $num_h + $num_f + $accomp;

                    AssocParticipante::create([
                        'nom' => $assoc['nom'] ?? null,
                        'num_athlete_h' => $num_h,
                        'num_athlete_f' => $num_f,
                        'accompagnants' => $accomp,
                        'total' => $total,
                        'event_id' => $evenement->id_even,
                    ]);
                }
            }

            if ($request->has('athletes')) {
                foreach ($request->athletes as $athlete) {
                    AthleteIndividuel::create([
                        'nom_prenom_sportif' => $athlete['nom_prenom_sportif'],
                        'num_athlete_h' => $athlete['num_athlete_h'] ?? 0,
                        'num_athlete_f' => $athlete['num_athlete_f'] ?? 0,
                        'accompagnants' => $athlete['accompagnants'] ?? 0,
                        'event_id' => $evenement->id_even,
                    ]);
                }
            }

            if ($request->has('sportifs')) {
                foreach ($request->sportifs as $sportif) {
                    SportifParticipant::create([
                        'nom_prenom' => $sportif['nom_prenom'],
                        'tranch_age' => $sportif['tranch_age'] ?? null,
                        'date_naissance' => $sportif['date_naissance'] ?? null,
                        'num_cin' => $sportif['num_cin'] ?? null,
                        'invitee' => $sportif['invitee'] ?? null,
                        'even_id' => $evenement->id_even,
                        'jeune_id' => $sportif['jeune_id'] ?? null,
                    ]);
                }
            }
        }

        DB::commit();

try {
    $d = $demande->fresh(['utilisateur']);

    $resolved = (int) ($request->id_type ?? $request->statut_id ?? $request->statut2_id ?? 0);
    $statusMap = [1 => 'accepted', 3 => 'rejected', 4 => 'incomplete', 5 => 'deferred'];
    $type = $statusMap[$resolved] ?? 'status_changed';

    // build the messageComment from the saved model or request
    $messageComment = trim(($d->commentaire_motif ?? '') . ' ' . ($d->commentaire_motif2 ?? '') . ' ' . ($d->commentaire_motif3 ?? ''));
    $messageComment = $messageComment !== '' ? $messageComment : null;

    // dossier number
    $idPad = str_pad($d->id_demande, 3, '0', STR_PAD_LEFT);
    $cin = $d->utilisateur->num_cin ?? $d->utilisateur->cin ?? '';
    $cinDigits = preg_replace('/\D+/', '', $cin);
    $cinPad = $cinDigits ? str_pad($cinDigits, 9, '0', STR_PAD_LEFT) : '000000000';
    $year = $d->created_at ? $d->created_at->format('Y') : now()->format('Y');
    $num_dossier = "{$idPad}-{$cinPad}-{$year}";

    $nomUtilisateur = $d->utilisateur->nom_prenom ?? 'المستخدم';
    $date = now()->format('Y-m-d');

    switch ($resolved) {
        case 1:
            $messageStatus = "مرحباً <strong>{$nomUtilisateur}</strong>،<br>رقم الملف: <strong>{$num_dossier}</strong><br>تم قبول طلبكم بتاريخ: <strong>{$date}</strong>.<br>نهنئكم على هذا القبول.";
            break;
        case 3:
            $reason = e($messageComment ?? ($request->input('commentaire_motif') ?? 'سبب غير محدد'));
            $messageStatus = "مرحباً <strong>{$nomUtilisateur}</strong>،<br>رقم الملف: <strong>{$num_dossier}</strong><br>تم رفض طلبكم.<br>التعليق: <b>{$reason}</b>.";
            break;
        case 4:
            $missing = e($request->input('commentaire_acceptation') ?? ($messageComment ?? 'يرجى استكمال الوثائق المطلوبة'));
            $messageStatus = "مرحباً <strong>{$nomUtilisateur}</strong>،<br>رقم الملف: <strong>{$num_dossier}</strong><br>طلبكم <strong>غير مكتمل</strong>.<br>التوضيح: <b>{$missing}</b>.";
            break;
        case 5:
            $note = e($request->input('commentaire_avis') ?? ($messageComment ?? 'لا يوجد ملاحظات'));
            $messageStatus = "مرحباً <strong>{$nomUtilisateur}</strong>،<br>رقم الملف: <strong>{$num_dossier}</strong><br>تم إرجاء طلبكم.<br>الملاحظة: <b>{$note}</b>.";
            break;
        default:
            $messageStatus = "مرحباً <strong>{$nomUtilisateur}</strong>،<br>رقم الملف: <strong>{$num_dossier}</strong><br>تم تحديث حالة ملفكم.";
    }

    // send mail
    if (!empty($d->utilisateur->email)) {
        try {
            Mail::to($d->utilisateur->email)->send(
                new \App\Mail\StatutDemandeMail($d->utilisateur, $num_dossier, $messageStatus, $type, $messageComment)
            );
        } catch (\Exception $ex) {
            Log::error("Mail error after update for DemandeManifestation {$d->id_demande}: " . $ex->getMessage());
        }
    }

    // DB notification
    if ($d->utilisateur) {
        $translationKey = "site.notifications.manifestation.{$type}";
        $d->utilisateur->notify(new \App\Notifications\StatutDemandeNotification(
            $d, $type, [
                'reason' => $messageComment ?? '',
                'reference' => $num_dossier
            ],
            $d->id_demande, $translationKey
        ));
    }
} catch (\Exception $ex) {
    Log::error("Notification error after update for DemandeManifestation {$demande->id_demande}: " . $ex->getMessage());
}


        session()->forget('_old_input');

        return redirect()->route('admin.manifestation.index', $demande->id_demande)
                         ->with('success', 'تم تحديث الطلب بنجاح');
    } catch (\Exception $e) {
        DB::rollBack();
        return redirect()->back()->withInput()->with('error', 'حدث خطأ أثناء التحديث: ' . $e->getMessage());
    }
}*/













//last function update
/*public function update(Request $request, $id)
{
    $request->validate([
        'sujet' => 'nullable|string|max:255',
        'sport_type' => 'nullable|string|max:255',
        'event_name' => 'nullable|string|max:255',
        'objectif' => 'nullable|string',
        'commentaire_motif' => 'nullable|string',
        'type_formulaire_id' => 'required|integer',
        'statut_id' => 'nullable|integer',
        'statut2_id' => 'nullable|integer',
        'commentaire_motif2' => 'nullable|string|max:255',
        'commentaire_motif3' => 'nullable|string|max:255',
        'file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:4096',
        'id_type' => 'nullable|integer',
        'organisme_non_sportif_id' => 'nullable|integer',
        'role' => 'nullable|string',
        'periode_min' => 'nullable|date',
        'periode_max' => 'nullable|date|after_or_equal:periode_min',
        'lieu' => 'nullable|string|max:255',

        // Organisme non sportif : valider org comme array
        'org' => 'nullable|array',
        'org.nom' => 'nullable|string|max:255',
        'org.num_visa' => 'nullable|integer',
        'org.date_enregistrement' => 'nullable|date',
        'org.num_enregistrement' => 'nullable|string|max:50',
        'org.identifiant_fiscal' => 'nullable|numeric',
        'org.num_compte_bancaire' => 'nullable|string|max:50',
        'org.adresse' => 'nullable|string|max:255',
        'org.tel' => 'nullable|string|max:20',
        'org.fax' => 'nullable|string|max:20',
        'org.email' => 'nullable|email|max:255',
        'org.type_formulaire_id' => 'nullable|integer',

        // EvenementSportif
        'nom_even' => 'required_with:date|string|max:100',
        'objectifs' => 'nullable|string',
        'contenus' => 'nullable|string',
        'date' => 'required_with:nom_even|date',
        'heure' => 'nullable|date_format:H:i',
        'instalation_sportives' => 'nullable|string|max:200',
        'reference' => 'nullable|string|max:100',
        'estimation_evenement' => 'nullable|numeric',
        'cout_total_evenement' => 'nullable|numeric',
        'difference_estimee' => 'nullable|numeric',
        'recompense' => 'nullable|string',
        'moyen_transport' => 'nullable|string|max:200',
        'gouvernorat_id' => 'required_with:nom_even|integer',
        'delegation_id' => 'required_with:nom_even|integer',

        // collections
        'membres.*.nom_prenom' => 'required_with:membres|string|max:255',
        'invites.*.nom_prenom' => 'required_with:invites|string|max:255',
        'assoc_participantes.*.nom' => 'required_with:assoc_participantes|string|max:255',
        'athletes.*.nom_prenom_sportif' => 'nullable|string|max:255',
        'sportifs.*.nom_prenom' => 'required_with:sportifs|string|max:255',

        // attachements
        'assurance_file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
        'technique_file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
        'accord_file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
        'demande_file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
    ]);

    DB::beginTransaction();

    try {
        $demande = DemandeManifestation::findOrFail($id);

        $ancienStatut = $demande->getOriginal('id_type');
        $ancienStatut1 = $demande->getOriginal('statut_id');
        $ancienStatut2 = $demande->getOriginal('statut2_id');

        $organisme = null;
        $orgInput = $request->input('org', null);

        if (is_array($orgInput) && count(array_filter($orgInput)) > 0) {
            $orgInput['type_formulaire_id'] = $orgInput['type_formulaire_id'] ?? 14;

            if ($demande->organisme_non_sportif_id) {
                $organisme = OrganismeNonSportif::find($demande->organisme_non_sportif_id);
                if ($organisme) {
                    $organisme->fill($orgInput);
                    $organisme->save();
                } else {
                    $organisme = OrganismeNonSportif::create($orgInput);
                }
            } else {
                $organisme = OrganismeNonSportif::create($orgInput);
            }

            $pkName = $organisme->getKeyName();
            $demande->organisme_non_sportif_id = $organisme->{$pkName};
        }

        // ============================================
        // CORRECTION CRITIQUE : GESTION DU FICHIER 'file'
        // ============================================
        $uploadDir = '/home/preprov/www/mjs/shared_uploads/images';
        
        // Vérifier et créer le dossier si nécessaire
        if (!is_dir($uploadDir)) {
            if (!mkdir($uploadDir, 0755, true)) {
                abort(500, 'لا يمكن إنشاء مجلد التحميل');
            }
        }
        
        // Vérifier les permissions
        if (!is_writable($uploadDir)) {
            // Essayer de changer les permissions
            if (!chmod($uploadDir, 0755)) {
                abort(500, 'المجلد غير قابل للكتابة ولا يمكن تغيير الصلاحيات');
            }
        }

        // Initialiser les variables
        $filePath = $demande->file; 
        $fileName = $demande->file_name ?? null;
        $fileType = $demande->file_type ?? null;

        if ($request->hasFile('file') && $request->file('file')->isValid()) {
            $file = $request->file('file');
            
            // Supprimer l'ancien fichier s'il existe
            if (!empty($demande->file) && file_exists($demande->file)) {
                try {
                    unlink($demande->file);
                } catch (\Exception $e) {
                    \Log::warning("Cannot delete old file: " . $e->getMessage());
                }
            }
            
            // Générer un nom de fichier sécurisé
            $original = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
            $ext = $file->getClientOriginalExtension();
            $safe = \Illuminate\Support\Str::slug(mb_substr($original, 0, 40)) ?: 'document';
            
            // Nom de fichier plus clair
            $filename = 'licence_' . $demande->id_demande . '_' . $safe . '_' . time() . '.' . $ext;
            $targetPath = $uploadDir . DIRECTORY_SEPARATOR . $filename;
            
            try {
                // Déplacer le fichier
                $file->move($uploadDir, $filename);
                
                // Vérifier si le fichier a été créé
                if (!file_exists($targetPath)) {
                    throw new \Exception('لم يتم إنشاء الملف بعد النقل');
                }
                
                $realPath = realpath($targetPath);
                if ($realPath === false) {
                    throw new \Exception('تعذر الحصول على المسار الحقيقي للملف');
                }
                
                $filePath = $realPath;
                $fileName = $file->getClientOriginalName();
                $fileType = $file->getMimeType();
                
                \Log::info("Fichier téléchargé avec succès: " . $filePath);
                
            } catch (\Exception $e) {
                \Log::error("Erreur lors du téléchargement du fichier: " . $e->getMessage());
                return redirect()->back()
                    ->withInput()
                    ->with('error', 'خطأ في تحميل الملف: ' . $e->getMessage());
            }
        }

        $commentForHistory = null;
        if ($request->filled('commentaire_motif')) {
            $commentForHistory = trim($request->input('commentaire_motif'));
        } elseif ($request->filled('commentaire_motif2')) {
            $commentForHistory = trim($request->input('commentaire_motif2'));
        } elseif ($request->filled('commentaire_motif3')) {
            $commentForHistory = trim($request->input('commentaire_motif3'));
        }

        $demande->update([
            'sujet' => $request->sujet,
            'sport_type' => $request->sport_type,
            'event_name' => $request->event_name,
            'objectif' => $request->objectif,
            'commentaire_motif' => $request->commentaire_motif,
            'commentaire_motif2' => $request->commentaire_motif2,
            'commentaire_motif3' => $request->commentaire_motif3,
            'type_formulaire_id' => $request->type_formulaire_id ?? $demande->type_formulaire_id,
            'user_id' => $demande->user_id,
            'id_type' => $request->id_type ?? $demande->id_type,
            'statut_id' => $request->statut_id ?? $demande->statut_id,
            'statut2_id' => $request->statut2_id ?? $demande->statut2_id,
            'role' => $request->role,
            'periode_min' => $request->periode_min,
            'periode_max' => $request->periode_max,
            'lieu' => $request->lieu,
            'organisme_non_sportif_id' => $demande->organisme_non_sportif_id,
            'file' => $filePath,
            'file_name' => $fileName,
            'file_type' => $fileType,
        ]);

        // Normalize integers helper
        $normalize = function ($v) {
            if (is_null($v) || $v === '') return null;
            $i = (int)$v;
            return ($i > 0) ? $i : null;
        };

        // compute old and new values for comparison
        $old_id_type = $normalize($ancienStatut);
        $old_statut_id = $normalize($ancienStatut1);
        $old_statut2_id = $normalize($ancienStatut2);

        // reload fresh model to get saved values
        $demande->refresh();

        $new_id_type = $normalize($demande->id_type);
        $new_statut_id = $normalize($demande->statut_id);
        $new_statut2_id = $normalize($demande->statut2_id);

        // helper to check tp_statut existence
        $statExists = function ($id) {
            if (is_null($id)) return false;
            return DB::table('tp_statut')->where('id_statut', $id)->exists();
        };

        // create a history row for each changed slot (use demande_manifestation_id)
        if ($old_id_type !== $new_id_type) {
            ImmobilierStatutHistory::create([
                'demande_manifestation_id' => $demande->id_demande,
                'from_statut_id'           => $old_id_type,
                'to_statut_id'             => $statExists($new_id_type) ? $new_id_type : null,
                'changed_by'               => Auth::id() ?? null,
                'commentaire'              => $commentForHistory,
            ]);
        }

        if ($old_statut_id !== $new_statut_id) {
            ImmobilierStatutHistory::create([
                'demande_manifestation_id' => $demande->id_demande,
                'from_statut_id'           => $old_statut_id,
                'to_statut_id'             => $statExists($new_statut_id) ? $new_statut_id : null,
                'changed_by'               => Auth::id() ?? null,
                'commentaire'              => $commentForHistory,
            ]);
        }

        if ($old_statut2_id !== $new_statut2_id) {
            ImmobilierStatutHistory::create([
                'demande_manifestation_id' => $demande->id_demande,
                'from_statut_id'           => $old_statut2_id,
                'to_statut_id'             => $statExists($new_statut2_id) ? $new_statut2_id : null,
                'changed_by'               => Auth::id() ?? null,
                'commentaire'              => $commentForHistory,
            ]);
        }

        // ============================================
        // GESTION DES AUTRES FICHIERS (attachments)
        // ============================================
        $attachmentsMap = [
            'assurance_file' => 2,
            'accord_file'    => 7,
            'technique_file' => 8,
            'demande_file'   => 9,
        ];

        foreach ($attachmentsMap as $inputName => $typeId) {
            if ($request->hasFile($inputName) && $request->file($inputName)->isValid()) {
                $f = $request->file($inputName);
                
                $oldAttachment = Attachement::where('id_demande', $demande->id_demande)
                    ->where('id_type_attach', $typeId)
                    ->first();

                // Supprimer l'ancien fichier
                if ($oldAttachment && !empty($oldAttachment->file) && file_exists($oldAttachment->file)) {
                    try {
                        unlink($oldAttachment->file);
                    } catch (\Exception $e) {
                        \Log::warning("Cannot delete old attachment: " . $e->getMessage());
                    }
                }

                $original = pathinfo($f->getClientOriginalName(), PATHINFO_FILENAME);
                $ext = $f->getClientOriginalExtension();
                $safe = \Illuminate\Support\Str::slug(mb_substr($original, 0, 40)) ?: 'document';
                
                $filename = 'attachment_' . $demande->id_demande . '_' . $typeId . '_' . $safe . '_' . time() . '.' . $ext;
                $targetPath = $uploadDir . DIRECTORY_SEPARATOR . $filename;
                
                try {
                    $f->move($uploadDir, $filename);
                    
                    if (!file_exists($targetPath)) {
                        throw new \Exception('لم يتم إنشاء ملف المرفق');
                    }
                    
                    $realPath = realpath($targetPath);
                    
                    if ($oldAttachment) {
                        $oldAttachment->update([
                            'file' => $realPath,
                            'mime_type' => $f->getMimeType(),
                        ]);
                    } else {
                        Attachement::create([
                            'id_demande'     => $demande->id_demande,
                            'id_type_attach' => $typeId,
                            'file'           => $realPath,
                            'mime_type'      => $f->getMimeType(),
                        ]);
                    }
                    
                } catch (\Exception $e) {
                    \Log::error("Erreur attachment {$inputName}: " . $e->getMessage());
                }
            }
        }

        // Evenement sportif logic (unchanged)
        $evenement = EvenementSportif::where('demande_id', $demande->id_demande)->first();

        if ($request->has('nom_even')) {
            $evenData = [
                'nom_even' => $request->nom_even,
                'objectifs' => $request->objectifs,
                'contenus' => $request->contenus,
                'date' => $request->date,
                'heure' => $request->heure,
                'instalation_sportives' => $request->instalation_sportives,
                'reference' => $request->reference,
                'estimation_evenement' => $request->estimation_evenement,
                'cout_total_evenement' => $request->cout_total_evenement,
                'difference_estimee' => $request->difference_estimee,
                'recompense' => $request->recompense,
                'moyen_transport' => $request->moyen_transport,
                'user_id' => Auth::id(),
                'id_type' => $request->id_type ?? 2,
                'id_formulaire' => $request->id_formulaire ?? 15,
                'gouvernorat_id' => $request->gouvernorat_id,
                'delegation_id' => $request->delegation_id,
                'demande_id' => $demande->id_demande,
            ];

            if ($evenement) {
                $evenement->update($evenData);
            } else {
                $evenement = EvenementSportif::create($evenData);
            }

            MembreOrgInvite::where('event_id', $evenement->id_even)->delete();
            AssocParticipante::where('event_id', $evenement->id_even)->delete();
            AthleteIndividuel::where('event_id', $evenement->id_even)->delete();
            SportifParticipant::where('even_id', $evenement->id_even)->delete();

            if ($request->has('membres')) {
                foreach ($request->membres as $membre) {
                    MembreOrgInvite::create([
                        'nom_prenom' => $membre['nom_prenom'],
                        'role' => $membre['role'] ?? null,
                        'nationalite' => $membre['nationalite'] ?? null,
                        'event_id' => $evenement->id_even,
                        'type_membre_id' => 1,
                    ]);
                }
            }

            if ($request->has('invites')) {
                foreach ($request->invites as $invite) {
                    MembreOrgInvite::create([
                        'nom_prenom' => $invite['nom_prenom'],
                        'role' => $invite['role'] ?? null,
                        'nationalite' => $invite['nationalite'] ?? null,
                        'event_id' => $evenement->id_even,
                        'type_membre_id' => 2,
                    ]);
                }
            }

            if ($request->has('assoc_participantes')) {
                foreach ($request->assoc_participantes as $assoc) {
                    $num_h = isset($assoc['num_athlete_h']) ? (int)$assoc['num_athlete_h'] : 0;
                    $num_f = isset($assoc['num_athlete_f']) ? (int)$assoc['num_athlete_f'] : 0;
                    $accomp = isset($assoc['accompagnants']) ? (int)$assoc['accompagnants'] : 0;
                    $total = $num_h + $num_f + $accomp;

                    AssocParticipante::create([
                        'nom' => $assoc['nom'] ?? null,
                        'num_athlete_h' => $num_h,
                        'num_athlete_f' => $num_f,
                        'accompagnants' => $accomp,
                        'total' => $total,
                        'event_id' => $evenement->id_even,
                    ]);
                }
            }

            if ($request->has('athletes')) {
                foreach ($request->athletes as $athlete) {
                    AthleteIndividuel::create([
                        'nom_prenom_sportif' => $athlete['nom_prenom_sportif'],
                        'num_athlete_h' => $athlete['num_athlete_h'] ?? 0,
                        'num_athlete_f' => $athlete['num_athlete_f'] ?? 0,
                        'accompagnants' => $athlete['accompagnants'] ?? 0,
                        'event_id' => $evenement->id_even,
                    ]);
                }
            }

            if ($request->has('sportifs')) {
                foreach ($request->sportifs as $sportif) {
                    SportifParticipant::create([
                        'nom_prenom' => $sportif['nom_prenom'],
                        'tranch_age' => $sportif['tranch_age'] ?? null,
                        'date_naissance' => $sportif['date_naissance'] ?? null,
                        'num_cin' => $sportif['num_cin'] ?? null,
                        'invitee' => $sportif['invitee'] ?? null,
                        'even_id' => $evenement->id_even,
                        'jeune_id' => $sportif['jeune_id'] ?? null,
                    ]);
                }
            }
        }

        DB::commit();

        try {
            $d = $demande->fresh(['utilisateur']);

            $resolved = (int) ($request->id_type ?? $request->statut_id ?? $request->statut2_id ?? 0);
            $statusMap = [1 => 'accepted', 3 => 'rejected', 4 => 'incomplete', 5 => 'deferred'];
            $type = $statusMap[$resolved] ?? 'status_changed';

            // build the messageComment from the saved model or request
            $messageComment = trim(($d->commentaire_motif ?? '') . ' ' . ($d->commentaire_motif2 ?? '') . ' ' . ($d->commentaire_motif3 ?? ''));
            $messageComment = $messageComment !== '' ? $messageComment : null;

            // dossier number
            $idPad = str_pad($d->id_demande, 3, '0', STR_PAD_LEFT);
            $cin = $d->utilisateur->num_cin ?? $d->utilisateur->cin ?? '';
            $cinDigits = preg_replace('/\D+/', '', $cin);
            $cinPad = $cinDigits ? str_pad($cinDigits, 9, '0', STR_PAD_LEFT) : '000000000';
            $year = $d->created_at ? $d->created_at->format('Y') : now()->format('Y');
            $num_dossier = "{$idPad}-{$cinPad}-{$year}";

            $nomUtilisateur = $d->utilisateur->nom_prenom ?? 'المستخدم';
            $date = now()->format('Y-m-d');

            switch ($resolved) {
                case 1:
                    $messageStatus = "مرحباً <strong>{$nomUtilisateur}</strong>،<br>رقم الملف: <strong>{$num_dossier}</strong><br>تم قبول طلبكم بتاريخ: <strong>{$date}</strong>.<br>نهنئكم على هذا القبول.";
                    break;
                case 3:
                    $reason = e($messageComment ?? ($request->input('commentaire_motif') ?? 'سبب غير محدد'));
                    $messageStatus = "مرحباً <strong>{$nomUtilisateur}</strong>،<br>رقم الملف: <strong>{$num_dossier}</strong><br>تم رفض طلبكم.<br>التعليق: <b>{$reason}</b>.";
                    break;
                case 4:
                    $missing = e($request->input('commentaire_acceptation') ?? ($messageComment ?? 'يرجى استكمال الوثائق المطلوبة'));
                    $messageStatus = "مرحباً <strong>{$nomUtilisateur}</strong>،<br>رقم الملف: <strong>{$num_dossier}</strong><br>طلبكم <strong>غير مكتمل</strong>.<br>التوضيح: <b>{$missing}</b>.";
                    break;
                case 5:
                    $note = e($request->input('commentaire_avis') ?? ($messageComment ?? 'لا يوجد ملاحظات'));
                    $messageStatus = "مرحباً <strong>{$nomUtilisateur}</strong>،<br>رقم الملف: <strong>{$num_dossier}</strong><br>تم إرجاء طلبكم.<br>الملاحظة: <b>{$note}</b>.";
                    break;
                default:
                    $messageStatus = "مرحباً <strong>{$nomUtilisateur}</strong>،<br>رقم الملف: <strong>{$num_dossier}</strong><br>تم تحديث حالة ملفكم.";
            }

            // send mail
            if (!empty($d->utilisateur->email)) {
                try {
                    Mail::to($d->utilisateur->email)->send(
                        new \App\Mail\StatutDemandeMail($d->utilisateur, $num_dossier, $messageStatus, $type, $messageComment)
                    );
                } catch (\Exception $ex) {
                    Log::error("Mail error after update for DemandeManifestation {$d->id_demande}: " . $ex->getMessage());
                }
            }

            // DB notification
            if ($d->utilisateur) {
                $translationKey = "site.notifications.manifestation.{$type}";
                $d->utilisateur->notify(new \App\Notifications\StatutDemandeNotification(
                    $d, $type, [
                        'reason' => $messageComment ?? '',
                        'reference' => $num_dossier
                    ],
                    $d->id_demande, $translationKey
                ));
            }
        } catch (\Exception $ex) {
            Log::error("Notification error after update for DemandeManifestation {$demande->id_demande}: " . $ex->getMessage());
        }

        session()->forget('_old_input');

        return redirect()->route('admin.manifestation.index', $demande->id_demande)
                         ->with('success', 'تم تحديث الطلب بنجاح');
    } catch (\Exception $e) {
        DB::rollBack();
        \Log::error('Update error: ' . $e->getMessage());
        \Log::error('Trace: ' . $e->getTraceAsString());
        
        return redirect()->back()->withInput()->with('error', 'حدث خطأ أثناء التحديث: ' . $e->getMessage());
    }
}*/

/*public function update(Request $request, $id)
{
    $request->validate([
        'sujet' => 'nullable|string|max:255',
        'sport_type' => 'nullable|string|max:255',
        'event_name' => 'nullable|string|max:255',
        'objectif' => 'nullable|string',
        'commentaire_motif' => 'nullable|string',
        'type_formulaire_id' => 'required|integer',
        'statut_id' => 'nullable|integer',
        'statut2_id' => 'nullable|integer',
        'commentaire_motif2' => 'nullable|string|max:255',
        'commentaire_motif3' => 'nullable|string|max:255',
        'file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:4096',
        'id_type' => 'nullable|integer',
        'organisme_non_sportif_id' => 'nullable|integer',
        'role' => 'nullable|string',
        'periode_min' => 'nullable|date',
        'periode_max' => 'nullable|date|after_or_equal:periode_min',
        'lieu' => 'nullable|string|max:255',

        // Organisme non sportif : valider org comme array
        'org' => 'nullable|array',
        'org.nom' => 'nullable|string|max:255',
        'org.num_visa' => 'nullable|integer',
        'org.date_enregistrement' => 'nullable|date',
        'org.num_enregistrement' => 'nullable|string|max:50',
        'org.identifiant_fiscal' => 'nullable|numeric',
        'org.num_compte_bancaire' => 'nullable|string|max:50',
        'org.adresse' => 'nullable|string|max:255',
        'org.tel' => 'nullable|string|max:20',
        'org.fax' => 'nullable|string|max:20',
        'org.email' => 'nullable|email|max:255',
        'org.type_formulaire_id' => 'nullable|integer',

        // EvenementSportif
        'nom_even' => 'required_with:date|string|max:100',
        'objectifs' => 'nullable|string',
        'contenus' => 'nullable|string',
        'date' => 'required_with:nom_even|date',
        'heure' => 'nullable|date_format:H:i',
        'instalation_sportives' => 'nullable|string|max:200',
        'reference' => 'nullable|string|max:100',
        'estimation_evenement' => 'nullable|numeric',
        'cout_total_evenement' => 'nullable|numeric',
        'difference_estimee' => 'nullable|numeric',
        'recompense' => 'nullable|string',
        'moyen_transport' => 'nullable|string|max:200',
        'gouvernorat_id' => 'required_with:nom_even|integer',
        'delegation_id' => 'required_with:nom_even|integer',

        // collections
        'membres.*.nom_prenom' => 'required_with:membres|string|max:255',
        'invites.*.nom_prenom' => 'required_with:invites|string|max:255',
        'assoc_participantes.*.nom' => 'required_with:assoc_participantes|string|max:255',
        'athletes.*.nom_prenom_sportif' => 'nullable|string|max:255',
        'sportifs.*.nom_prenom' => 'required_with:sportifs|string|max:255',

        // attachements
        'assurance_file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
        'technique_file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
        'accord_file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
        'demande_file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
    ]);

    DB::beginTransaction();

    try {
        $demande = DemandeManifestation::findOrFail($id);

        $ancienStatut = $demande->getOriginal('id_type');
        $ancienStatut1 = $demande->getOriginal('statut_id');
        $ancienStatut2 = $demande->getOriginal('statut2_id');

        $organisme = null;
        $orgInput = $request->input('org', null);

        if (is_array($orgInput) && count(array_filter($orgInput)) > 0) {
            $orgInput['type_formulaire_id'] = $orgInput['type_formulaire_id'] ?? 14;

            if ($demande->organisme_non_sportif_id) {
                $organisme = OrganismeNonSportif::find($demande->organisme_non_sportif_id);
                if ($organisme) {
                    $organisme->fill($orgInput);
                    $organisme->save();
                } else {
                    $organisme = OrganismeNonSportif::create($orgInput);
                }
            } else {
                $organisme = OrganismeNonSportif::create($orgInput);
            }

            $pkName = $organisme->getKeyName();
            $demande->organisme_non_sportif_id = $organisme->{$pkName};
        }

        // ============================================
        // GESTION DU FICHIER 'file' - VERSION SIMPLIFIÉE
        // ============================================
        $uploadDir = '/home/preprov/www/mjs/shared_uploads/images';
        
        // Vérifier et créer le dossier si nécessaire
        if (!is_dir($uploadDir)) {
            if (!mkdir($uploadDir, 0755, true)) {
                abort(500, 'لا يمكن إنشاء مجلد التحميل');
            }
        }
        
        // Initialiser la variable avec l'ancienne valeur
        $filePath = $demande->file;

        // DÉBUGAGE : Log pour voir si le fichier est reçu
        \Log::info('File upload check:', [
            'hasFile' => $request->hasFile('file'),
            'isValid' => $request->hasFile('file') ? $request->file('file')->isValid() : false,
            'file_name' => $request->hasFile('file') ? $request->file('file')->getClientOriginalName() : null,
        ]);

        if ($request->hasFile('file') && $request->file('file')->isValid()) {
            $file = $request->file('file');
            
            // Supprimer l'ancien fichier s'il existe
            if (!empty($demande->file) && file_exists($demande->file)) {
                try {
                    unlink($demande->file);
                } catch (\Exception $e) {
                    \Log::warning("Cannot delete old file: " . $e->getMessage());
                }
            }
            
            // Générer un nom de fichier sécurisé
            $original = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
            $ext = $file->getClientOriginalExtension();
            $safe = \Illuminate\Support\Str::slug(mb_substr($original, 0, 40)) ?: 'document';
            
            // Nom de fichier
            $filename = 'licence_' . $demande->id_demande . '_' . $safe . '_' . time() . '.' . $ext;
            $targetPath = $uploadDir . DIRECTORY_SEPARATOR . $filename;
            
            // Déplacer le fichier
            try {
                $file->move($uploadDir, $filename);
                
                if (!file_exists($targetPath)) {
                    throw new \Exception('لم يتم إنشاء الملف بعد النقل');
                }
                
                $filePath = $targetPath;
                
                \Log::info("Fichier téléchargé avec succès", [
                    'path' => $filePath,
                    'name' => $filename
                ]);
                
            } catch (\Exception $e) {
                \Log::error("Erreur lors du téléchargement du fichier: " . $e->getMessage());
                DB::rollBack();
                return redirect()->back()
                    ->withInput()
                    ->with('error', 'خطأ في تحميل الملف: ' . $e->getMessage());
            }
        }

        $commentForHistory = null;
        if ($request->filled('commentaire_motif')) {
            $commentForHistory = trim($request->input('commentaire_motif'));
        } elseif ($request->filled('commentaire_motif2')) {
            $commentForHistory = trim($request->input('commentaire_motif2'));
        } elseif ($request->filled('commentaire_motif3')) {
            $commentForHistory = trim($request->input('commentaire_motif3'));
        }

        // DÉBUGAGE : Log des valeurs avant update
        \Log::info('Updating demande with file info:', [
            'filePath' => $filePath,
        ]);

        // Mettre à jour la demande avec les nouvelles valeurs
        // NOTE: On enregistre SEULEMENT le chemin dans le champ 'file'
        $demande->update([
            'sujet' => $request->sujet,
            'sport_type' => $request->sport_type,
            'event_name' => $request->event_name,
            'objectif' => $request->objectif,
            'commentaire_motif' => $request->commentaire_motif,
            'commentaire_motif2' => $request->commentaire_motif2,
            'commentaire_motif3' => $request->commentaire_motif3,
            'type_formulaire_id' => $request->type_formulaire_id ?? $demande->type_formulaire_id,
            'user_id' => $demande->user_id,
            'id_type' => $request->id_type ?? $demande->id_type,
            'statut_id' => $request->statut_id ?? $demande->statut_id,
            'statut2_id' => $request->statut2_id ?? $demande->statut2_id,
            'role' => $request->role,
            'periode_min' => $request->periode_min,
            'periode_max' => $request->periode_max,
            'lieu' => $request->lieu,
            'organisme_non_sportif_id' => $demande->organisme_non_sportif_id,
            'file' => $filePath, // SEULEMENT CETTE LIGNE EST IMPORTANTE
        ]);

        // DÉBUGAGE : Vérifier après update
        $demande->refresh();
        \Log::info('After update - file value:', [
            'file' => $demande->file,
        ]);

        // Normalize integers helper
        $normalize = function ($v) {
            if (is_null($v) || $v === '') return null;
            $i = (int)$v;
            return ($i > 0) ? $i : null;
        };

        // compute old and new values for comparison
        $old_id_type = $normalize($ancienStatut);
        $old_statut_id = $normalize($ancienStatut1);
        $old_statut2_id = $normalize($ancienStatut2);

        // reload fresh model to get saved values
        $demande->refresh();

        $new_id_type = $normalize($demande->id_type);
        $new_statut_id = $normalize($demande->statut_id);
        $new_statut2_id = $normalize($demande->statut2_id);

        // helper to check tp_statut existence
        $statExists = function ($id) {
            if (is_null($id)) return false;
            return DB::table('tp_statut')->where('id_statut', $id)->exists();
        };

        // create a history row for each changed slot (use demande_manifestation_id)
        if ($old_id_type !== $new_id_type) {
            ImmobilierStatutHistory::create([
                'demande_manifestation_id' => $demande->id_demande,
                'from_statut_id'           => $old_id_type,
                'to_statut_id'             => $statExists($new_id_type) ? $new_id_type : null,
                'changed_by'               => Auth::id() ?? null,
                'commentaire'              => $commentForHistory,
            ]);
        }

        if ($old_statut_id !== $new_statut_id) {
            ImmobilierStatutHistory::create([
                'demande_manifestation_id' => $demande->id_demande,
                'from_statut_id'           => $old_statut_id,
                'to_statut_id'             => $statExists($new_statut_id) ? $new_statut_id : null,
                'changed_by'               => Auth::id() ?? null,
                'commentaire'              => $commentForHistory,
            ]);
        }

        if ($old_statut2_id !== $new_statut2_id) {
            ImmobilierStatutHistory::create([
                'demande_manifestation_id' => $demande->id_demande,
                'from_statut_id'           => $old_statut2_id,
                'to_statut_id'             => $statExists($new_statut2_id) ? $new_statut2_id : null,
                'changed_by'               => Auth::id() ?? null,
                'commentaire'              => $commentForHistory,
            ]);
        }

        // ============================================
        // GESTION DES AUTRES FICHIERS (attachments)
        // ============================================
        // ============================================
// GESTION DES AUTRES FICHIERS (attachments) - VERSION CORRIGÉE
// ============================================
$attachmentsMap = [
    'assurance_file' => 2,
    'accord_file'    => 7,
    'technique_file' => 8,
    'demande_file'   => 9,
];

foreach ($attachmentsMap as $inputName => $typeId) {
    if ($request->hasFile($inputName) && $request->file($inputName)->isValid()) {
        $f = $request->file($inputName);
        
        $oldAttachment = Attachement::where('id_demande', $demande->id_demande)
            ->where('id_type_attach', $typeId)
            ->first();

        // Supprimer l'ancien fichier
        if ($oldAttachment && !empty($oldAttachment->file)) {
            $oldPath = $oldAttachment->file;
            
            // Si le chemin est relatif, essayer avec le dossier d'upload
            if (strpos($oldPath, '/') !== 0) {
                $oldPath = $uploadDir . '/' . basename($oldPath);
            }
            
            if (file_exists($oldPath)) {
                try {
                    unlink($oldPath);
                    \Log::info("Ancien fichier d'attachment supprimé: " . basename($oldPath));
                } catch (\Exception $e) {
                    \Log::warning("Cannot delete old attachment: " . $e->getMessage());
                }
            }
        }

        // Générer un nom de fichier
        $original = pathinfo($f->getClientOriginalName(), PATHINFO_FILENAME);
        $ext = $f->getClientOriginalExtension();
        $safe = \Illuminate\Support\Str::slug(mb_substr($original, 0, 40)) ?: 'document';
        
        // Nom de fichier
        $filename = 'attachment_' . $demande->id_demande . '_' . $typeId . '_' . $safe . '_' . time() . '.' . $ext;
        $targetPath = $uploadDir . DIRECTORY_SEPARATOR . $filename;
        
        try {
            // Déplacer le fichier
            $f->move($uploadDir, $filename);
            
            // Vérifier si le fichier existe
            if (!file_exists($targetPath)) {
                throw new \Exception('لم يتم إنشاء ملف المرفق: ' . $targetPath);
            }
            
            // NE PAS utiliser realpath() - utiliser le chemin ABSOLU directement
            $filePathToStore = $targetPath; // Chemin absolu
            
            \Log::info("Fichier attachment téléchargé", [
                'demande_id' => $demande->id_demande,
                'type' => $typeId,
                'filename' => $filename,
                'path' => $filePathToStore
            ]);
            
            if ($oldAttachment) {
                $oldAttachment->update([
                    'file' => $filePathToStore, // Chemin absolu
                    'mime_type' => $f->getMimeType(),
                ]);
            } else {
                Attachement::create([
                    'id_demande'     => $demande->id_demande,
                    'id_type_attach' => $typeId,
                    'file'           => $filePathToStore, // Chemin absolu
                    'mime_type'      => $f->getMimeType(),
                ]);
            }
            
        } catch (\Exception $e) {
            \Log::error("Erreur attachment {$inputName}: " . $e->getMessage());
        }
    }
}

        // Evenement sportif logic (unchanged)
        $evenement = EvenementSportif::where('demande_id', $demande->id_demande)->first();

        if ($request->has('nom_even')) {
            $evenData = [
                'nom_even' => $request->nom_even,
                'objectifs' => $request->objectifs,
                'contenus' => $request->contenus,
                'date' => $request->date,
                'heure' => $request->heure,
                'instalation_sportives' => $request->instalation_sportives,
                'reference' => $request->reference,
                'estimation_evenement' => $request->estimation_evenement,
                'cout_total_evenement' => $request->cout_total_evenement,
                'difference_estimee' => $request->difference_estimee,
                'recompense' => $request->recompense,
                'moyen_transport' => $request->moyen_transport,
                'user_id' => Auth::id(),
                'id_type' => $request->id_type ?? 2,
                'id_formulaire' => $request->id_formulaire ?? 15,
                'gouvernorat_id' => $request->gouvernorat_id,
                'delegation_id' => $request->delegation_id,
                'demande_id' => $demande->id_demande,
            ];

            if ($evenement) {
                $evenement->update($evenData);
            } else {
                $evenement = EvenementSportif::create($evenData);
            }

            MembreOrgInvite::where('event_id', $evenement->id_even)->delete();
            AssocParticipante::where('event_id', $evenement->id_even)->delete();
            AthleteIndividuel::where('event_id', $evenement->id_even)->delete();
            SportifParticipant::where('even_id', $evenement->id_even)->delete();

            if ($request->has('membres')) {
                foreach ($request->membres as $membre) {
                    MembreOrgInvite::create([
                        'nom_prenom' => $membre['nom_prenom'],
                        'role' => $membre['role'] ?? null,
                        'nationalite' => $membre['nationalite'] ?? null,
                        'event_id' => $evenement->id_even,
                        'type_membre_id' => 1,
                    ]);
                }
            }

            if ($request->has('invites')) {
                foreach ($request->invites as $invite) {
                    MembreOrgInvite::create([
                        'nom_prenom' => $invite['nom_prenom'],
                        'role' => $invite['role'] ?? null,
                        'nationalite' => $invite['nationalite'] ?? null,
                        'event_id' => $evenement->id_even,
                        'type_membre_id' => 2,
                    ]);
                }
            }

            if ($request->has('assoc_participantes')) {
                foreach ($request->assoc_participantes as $assoc) {
                    $num_h = isset($assoc['num_athlete_h']) ? (int)$assoc['num_athlete_h'] : 0;
                    $num_f = isset($assoc['num_athlete_f']) ? (int)$assoc['num_athlete_f'] : 0;
                    $accomp = isset($assoc['accompagnants']) ? (int)$assoc['accompagnants'] : 0;
                    $total = $num_h + $num_f + $accomp;

                    AssocParticipante::create([
                        'nom' => $assoc['nom'] ?? null,
                        'num_athlete_h' => $num_h,
                        'num_athlete_f' => $num_f,
                        'accompagnants' => $accomp,
                        'total' => $total,
                        'event_id' => $evenement->id_even,
                    ]);
                }
            }

            if ($request->has('athletes')) {
                foreach ($request->athletes as $athlete) {
                    AthleteIndividuel::create([
                        'nom_prenom_sportif' => $athlete['nom_prenom_sportif'],
                        'num_athlete_h' => $athlete['num_athlete_h'] ?? 0,
                        'num_athlete_f' => $athlete['num_athlete_f'] ?? 0,
                        'accompagnants' => $athlete['accompagnants'] ?? 0,
                        'event_id' => $evenement->id_even,
                    ]);
                }
            }

            if ($request->has('sportifs')) {
                foreach ($request->sportifs as $sportif) {
                    SportifParticipant::create([
                        'nom_prenom' => $sportif['nom_prenom'],
                        'tranch_age' => $sportif['tranch_age'] ?? null,
                        'date_naissance' => $sportif['date_naissance'] ?? null,
                        'num_cin' => $sportif['num_cin'] ?? null,
                        'invitee' => $sportif['invitee'] ?? null,
                        'even_id' => $evenement->id_even,
                        'jeune_id' => $sportif['jeune_id'] ?? null,
                    ]);
                }
            }
        }

        DB::commit();

        // ============================================
        // NOTIFICATIONS
        // ============================================
        try {
            $d = $demande->fresh(['utilisateur']);

            $resolved = (int) ($request->id_type ?? $request->statut_id ?? $request->statut2_id ?? 0);
            $statusMap = [1 => 'accepted', 3 => 'rejected', 4 => 'incomplete', 5 => 'deferred'];
            $type = $statusMap[$resolved] ?? 'status_changed';

            // build the messageComment from the saved model or request
            $messageComment = trim(($d->commentaire_motif ?? '') . ' ' . ($d->commentaire_motif2 ?? '') . ' ' . ($d->commentaire_motif3 ?? ''));
            $messageComment = $messageComment !== '' ? $messageComment : null;

            // dossier number
            $idPad = str_pad($d->id_demande, 3, '0', STR_PAD_LEFT);
            $cin = $d->utilisateur->num_cin ?? $d->utilisateur->cin ?? '';
            $cinDigits = preg_replace('/\D+/', '', $cin);
            $cinPad = $cinDigits ? str_pad($cinDigits, 9, '0', STR_PAD_LEFT) : '000000000';
            $year = $d->created_at ? $d->created_at->format('Y') : now()->format('Y');
            $num_dossier = "{$idPad}-{$cinPad}-{$year}";

            $nomUtilisateur = $d->utilisateur->nom_prenom ?? 'المستخدم';
            $date = now()->format('Y-m-d');

            switch ($resolved) {
                case 1:
                    $messageStatus = "مرحباً <strong>{$nomUtilisateur}</strong>،<br>رقم الملف: <strong>{$num_dossier}</strong><br>تم قبول طلبكم بتاريخ: <strong>{$date}</strong>.<br>نهنئكم على هذا القبول.";
                    break;
                case 3:
                    $reason = e($messageComment ?? ($request->input('commentaire_motif') ?? 'سبب غير محدد'));
                    $messageStatus = "مرحباً <strong>{$nomUtilisateur}</strong>،<br>رقم الملف: <strong>{$num_dossier}</strong><br>تم رفض طلبكم.<br>التعليق: <b>{$reason}</b>.";
                    break;
                case 4:
                    $missing = e($request->input('commentaire_acceptation') ?? ($messageComment ?? 'يرجى استكمال الوثائق المطلوبة'));
                    $messageStatus = "مرحباً <strong>{$nomUtilisateur}</strong>،<br>رقم الملف: <strong>{$num_dossier}</strong><br>طلبكم <strong>غير مكتمل</strong>.<br>التوضيح: <b>{$missing}</b>.";
                    break;
                case 5:
                    $note = e($request->input('commentaire_avis') ?? ($messageComment ?? 'لا يوجد ملاحظات'));
                    $messageStatus = "مرحباً <strong>{$nomUtilisateur}</strong>،<br>رقم الملف: <strong>{$num_dossier}</strong><br>تم إرجاء طلبكم.<br>الملاحظة: <b>{$note}</b>.";
                    break;
                default:
                    $messageStatus = "مرحباً <strong>{$nomUtilisateur}</strong>،<br>رقم الملف: <strong>{$num_dossier}</strong><br>تم تحديث حالة ملفكم.";
            }

            // send mail
            if (!empty($d->utilisateur->email)) {
                try {
                    Mail::to($d->utilisateur->email)->send(
                        new \App\Mail\StatutDemandeMail($d->utilisateur, $num_dossier, $messageStatus, $type, $messageComment)
                    );
                } catch (\Exception $ex) {
                    Log::error("Mail error after update for DemandeManifestation {$d->id_demande}: " . $ex->getMessage());
                }
            }

            // DB notification
            if ($d->utilisateur) {
                $translationKey = "site.notifications.manifestation.{$type}";
                $d->utilisateur->notify(new \App\Notifications\StatutDemandeNotification(
                    $d, $type, [
                        'reason' => $messageComment ?? '',
                        'reference' => $num_dossier
                    ],
                    $d->id_demande, $translationKey
                ));
            }
        } catch (\Exception $ex) {
            Log::error("Notification error after update for DemandeManifestation {$demande->id_demande}: " . $ex->getMessage());
        }

        session()->forget('_old_input');

        return redirect()->route('admin.manifestation.index')
                         ->with('success', 'تم تحديث الطلب بنجاح');
    } catch (\Exception $e) {
        DB::rollBack();
        \Log::error('Update error: ' . $e->getMessage());
        \Log::error('Trace: ' . $e->getTraceAsString());
        
        return redirect()->back()->withInput()->with('error', 'حدث خطأ أثناء التحديث: ' . $e->getMessage());
    }
}

public function showDecision($id)
{
    $demande = DemandeManifestation::findOrFail($id);

    // إذا كان الحقل فارغاً
    if (empty($demande->file)) {
        abort(404, 'Aucun fichier disponible');
    }
    
    // تحديد المسار بناءً على نوع المسار المخزن
    $path = $demande->file;
    
    // إذا كان المسار ليس مطلقاً (لا يبدأ بـ /)
    if (strpos($path, '/') !== 0) {
        $uploadDir = '/home/preprov/www/mjs/shared_uploads/images';
        $path = $uploadDir . '/' . basename($path);
    }
    
    // التحقق من وجود الملف
    if (!file_exists($path)) {
        // محاولة أخرى للعثور على الملف
        $alternativePath = '/home/preprov/www/mjs/shared_uploads/images/' . basename($demande->file);
        if (file_exists($alternativePath)) {
            $path = $alternativePath;
        } else {
            \Log::error("Fichier introuvable pour la demande {$id}: " . $demande->file);
            abort(404, 'Fichier introuvable: ' . basename($demande->file));
        }
    }

    // تحديد Content-Type بناءً على امتداد الملف
    $extension = strtolower(pathinfo($path, PATHINFO_EXTENSION));
    $mimeTypes = [
        'pdf' => 'application/pdf',
        'jpg' => 'image/jpeg',
        'jpeg' => 'image/jpeg',
        'png' => 'image/png',
    ];

    $contentType = $mimeTypes[$extension] ?? mime_content_type($path);
    
    // التحقق مما إذا كان المستخدم يريد التحميل بدلاً من العرض
    if (request()->has('download')) {
        return response()->download($path, basename($path), [
            'Content-Type' => $contentType,
        ]);
    }

    // عرض الملف في المتصفح
    return response()->file($path, [
        'Content-Type' => $contentType,
        'Content-Disposition' => 'inline; filename="' . basename($path) . '"',
        'Cache-Control' => 'no-store, no-cache, must-revalidate, max-age=0',
        'Pragma' => 'no-cache',
        'Expires' => '0',
    ]);
}
*/



public function update(Request $request, $id)
{
    $request->validate([
        'sujet' => 'nullable|string|max:255',
        'sport_type' => 'nullable|string|max:255',
        'event_name' => 'nullable|string|max:255',
        'objectif' => 'nullable|string',
        'commentaire_motif' => 'nullable|string',
        'type_formulaire_id' => 'required|integer',
        'statut_id' => 'nullable|integer',
        'statut2_id' => 'nullable|integer',
        'commentaire_motif2' => 'nullable|string|max:255',
        'commentaire_motif3' => 'nullable|string|max:255',
        'file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:4096',
        'id_type' => 'nullable|integer',
        'organisme_non_sportif_id' => 'nullable|integer',
        'role' => 'nullable|string',
        'periode_min' => 'nullable|date',
        'periode_max' => 'nullable|date|after_or_equal:periode_min',
        'lieu' => 'nullable|string|max:255',
        // Organisme fields
        'org' => 'nullable|array',
        'org.nom' => 'nullable|string|max:255',
        'org.num_visa' => 'nullable|integer',
        'org.date_enregistrement' => 'nullable|date',
        'org.num_enregistrement' => 'nullable|string|max:50',
        'org.identifiant_fiscal' => 'nullable|numeric',
        'org.num_compte_bancaire' => 'nullable|string|max:50',
        'org.adresse' => 'nullable|string|max:255',
        'org.tel' => 'nullable|string|max:20',
        'org.fax' => 'nullable|string|max:20',
        'org.email' => 'nullable|email|max:255',
        'org.type_formulaire_id' => 'nullable|integer',
        // EvenementSportif
        'nom_even' => 'required_with:date|string|max:100',
        'objectifs' => 'nullable|string',
        'contenus' => 'nullable|string',
        'date' => 'required_with:nom_even|date',
        'heure' => 'nullable|date_format:H:i',
        'instalation_sportives' => 'nullable|string|max:200',
        'reference' => 'nullable|string|max:100',
        'estimation_evenement' => 'nullable|numeric',
        'cout_total_evenement' => 'nullable|numeric',
        'difference_estimee' => 'nullable|numeric',
        'recompense' => 'nullable|string',
        'moyen_transport' => 'nullable|string|max:200',
        'gouvernorat_id' => 'required_with:nom_even|integer',
        'delegation_id' => 'required_with:nom_even|integer',
        // collections
        'membres.*.nom_prenom' => 'required_with:membres|string|max:255',
        'invites.*.nom_prenom' => 'required_with:invites|string|max:255',
        'assoc_participantes.*.nom' => 'required_with:assoc_participantes|string|max:255',
        'athletes.*.nom_prenom_sportif' => 'nullable|string|max:255',
        'sportifs.*.nom_prenom' => 'required_with:sportifs|string|max:255',
        // attachements
        'assurance_file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
        'technique_file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
        'accord_file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
        'demande_file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
        // touristes
        'touristes.*.nom_prenom' => 'nullable|string|max:255',
        'touristes.*.pays_origine' => 'nullable|string|max:255',
        'touristes.*.num_passport' => 'nullable|string|max:50',
        'touristes.*.date_arrive' => 'nullable|date',
        'touristes.*.lieu_residence' => 'nullable|string|max:255',
        'touristes.*.date_depart' => 'nullable|date',
        'touriste2s.*.nom_prenom' => 'nullable|string|max:255',
        'touriste2s.*.pays_origine' => 'nullable|string|max:255',
        'touriste2s.*.num_passport' => 'nullable|string|max:50',
        'touriste2s.*.date_arrive' => 'nullable|date',
        'touriste2s.*.lieu_residence' => 'nullable|string|max:255',
        'touriste2s.*.date_depart' => 'nullable|date',
        'touriste3s.*.nom_prenom' => 'nullable|string|max:255',
        'touriste3s.*.pays_origine' => 'nullable|string|max:255',
        'touriste3s.*.num_passport' => 'nullable|string|max:50',
        'touriste3s.*.date_arrive' => 'nullable|date',
        'touriste3s.*.lieu_residence' => 'nullable|string|max:255',
        'touriste3s.*.date_depart' => 'nullable|date',
    ]);

    DB::beginTransaction();
    $committed = false;

    try {
        $demande = DemandeManifestation::findOrFail($id);
        
        // Définir le dossier d'upload
        $uploadDir = '/home/preprov/www/mjs/shared_uploads/images';
        
        // Vérifier et créer le dossier si nécessaire
        if (!is_dir($uploadDir)) {
            if (!mkdir($uploadDir, 0755, true)) {
                abort(500, 'لا يمكن إنشاء مجلد التحميل');
            }
        }

        // -------------------------
        // Organisme: update or create
        // -------------------------
        $organisme = null;
        $orgInput = $request->input('org', null);
        if (is_array($orgInput) && count(array_filter($orgInput)) > 0) {
            $orgInput['type_formulaire_id'] = $orgInput['type_formulaire_id'] ?? 14;

            if ($demande->organisme_non_sportif_id) {
                $organisme = OrganismeNonSportif::find($demande->organisme_non_sportif_id);
                if ($organisme) {
                    $organisme->fill($orgInput);
                    $organisme->save();
                } else {
                    $organisme = OrganismeNonSportif::create($orgInput);
                }
            } else {
                $organisme = OrganismeNonSportif::create($orgInput);
            }

            $pkName = $organisme->getKeyName();
            $demande->organisme_non_sportif_id = $organisme->{$pkName};
        }

        // -------------------------
        // Gestion du fichier principal (decision/file)
        // -------------------------
        $filePath = $demande->file; // Conserver l'ancien chemin par défaut

        if ($request->hasFile('file') && $request->file('file')->isValid()) {
            $file = $request->file('file');
            
            // Supprimer l'ancien fichier s'il existe
            if (!empty($demande->file)) {
                $oldFilePath = $demande->file;
                // Si c'est un chemin relatif, l'ajouter au dossier d'upload
                if (strpos($oldFilePath, '/') !== 0) {
                    $oldFilePath = $uploadDir . '/' . basename($oldFilePath);
                }
                if (file_exists($oldFilePath)) {
                    try {
                        unlink($oldFilePath);
                    } catch (\Exception $e) {
                        \Log::warning("Cannot delete old main file: " . $e->getMessage());
                    }
                }
            }
            
            // Générer un nom de fichier sécurisé
            $original = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
            $ext = $file->getClientOriginalExtension();
            $safe = \Illuminate\Support\Str::slug(mb_substr($original, 0, 40)) ?: 'document';
            
            // Nom de fichier
            $filename = 'decision_' . $demande->id_demande . '_' . $safe . '_' . time() . '.' . $ext;
            $targetPath = $uploadDir . DIRECTORY_SEPARATOR . $filename;
            
            // Déplacer le fichier
            try {
                $file->move($uploadDir, $filename);
                
                if (!file_exists($targetPath)) {
                    throw new \Exception('لم يتم إنشاء الملف بعد النقل');
                }
                
                $filePath = $targetPath;
                
                \Log::info("Fichier décision téléchargé avec succès", [
                    'demande_id' => $demande->id_demande,
                    'path' => $filePath,
                    'filename' => $filename
                ]);
                
            } catch (\Exception $e) {
                \Log::error("Erreur lors du téléchargement du fichier décision: " . $e->getMessage());
                DB::rollBack();
                return redirect()->back()
                    ->withInput()
                    ->with('error', 'خطأ في تحميل الملف: ' . $e->getMessage());
            }
        }

        // capture old status values
        $ancienStatut  = $demande->getOriginal('id_type');
        $ancienStatut1 = $demande->getOriginal('statut_id');
        $ancienStatut2 = $demande->getOriginal('statut2_id');

        // -------------------------
        // Mettre à jour la demande
        // -------------------------
        $updateData = [
            'sujet' => $request->input('sujet'),
            'sport_type' => $request->input('sport_type'),
            'event_name' => $request->input('event_name'),
            'objectif' => $request->input('objectif'),
            'commentaire_motif' => $request->input('commentaire_motif'),
            'commentaire_motif2' => $request->input('commentaire_motif2'),
            'commentaire_motif3' => $request->input('commentaire_motif3'),
            'type_formulaire_id' => $request->input('type_formulaire_id', $demande->type_formulaire_id),
            'user_id' => $demande->user_id,
            'id_type' => $request->input('id_type', $demande->id_type),
            'statut_id' => $request->input('statut_id', $demande->statut_id),
            'statut2_id' => $request->input('statut2_id', $demande->statut2_id),
            'role' => $request->input('role'),
            'periode_min' => $request->input('periode_min'),
            'periode_max' => $request->input('periode_max'),
            'lieu' => $request->input('lieu'),
            'organisme_non_sportif_id' => $demande->organisme_non_sportif_id,
            'file' => $filePath, // Chemin du fichier de décision
        ];

        // Perform update on DemandeManifestation
        $demande->update($updateData);

        // -------------------------
        // Gestion des attachements (fichiers supplémentaires)
        // -------------------------
        $attachmentsMap = [
            'assurance_file' => 2,
            'accord_file'    => 7,
            'technique_file' => 8,
            'demande_file'   => 9,
        ];

        foreach ($attachmentsMap as $inputName => $typeId) {
            if ($request->hasFile($inputName) && $request->file($inputName)->isValid()) {
                $f = $request->file($inputName);
                
                $oldAttachment = Attachement::where('id_demande', $demande->id_demande)
                    ->where('id_type_attach', $typeId)
                    ->first();

                // Supprimer l'ancien fichier s'il existe
                if ($oldAttachment && !empty($oldAttachment->file)) {
                    $oldPath = $oldAttachment->file;
                    
                    // Si le chemin est relatif, essayer avec le dossier d'upload
                    if (strpos($oldPath, '/') !== 0) {
                        $oldPath = $uploadDir . '/' . basename($oldPath);
                    }
                    
                    if (file_exists($oldPath)) {
                        try {
                            unlink($oldPath);
                            \Log::info("Ancien fichier d'attachment supprimé: " . basename($oldPath));
                        } catch (\Exception $e) {
                            \Log::warning("Cannot delete old attachment: " . $e->getMessage());
                        }
                    }
                }

                // Générer un nom de fichier
                $original = pathinfo($f->getClientOriginalName(), PATHINFO_FILENAME);
                $ext = $f->getClientOriginalExtension();
                $safe = \Illuminate\Support\Str::slug(mb_substr($original, 0, 40)) ?: 'document';
                
                // Nom de fichier
                $filename = 'attachment_' . $demande->id_demande . '_' . $typeId . '_' . $safe . '_' . time() . '.' . $ext;
                $targetPath = $uploadDir . DIRECTORY_SEPARATOR . $filename;
                
                try {
                    // Déplacer le fichier
                    $f->move($uploadDir, $filename);
                    
                    // Vérifier si le fichier existe
                    if (!file_exists($targetPath)) {
                        throw new \Exception('لم يتم إنشاء ملف المرفق: ' . $targetPath);
                    }
                    
                    // Stocker le chemin absolu
                    $filePathToStore = $targetPath;
                    
                    \Log::info("Fichier attachment téléchargé", [
                        'demande_id' => $demande->id_demande,
                        'type' => $typeId,
                        'filename' => $filename,
                        'path' => $filePathToStore
                    ]);
                    
                    if ($oldAttachment) {
                        $oldAttachment->update([
                            'file' => $filePathToStore, // Chemin absolu
                            'mime_type' => $f->getMimeType(),
                        ]);
                    } else {
                        Attachement::create([
                            'id_demande'     => $demande->id_demande,
                            'id_type_attach' => $typeId,
                            'file'           => $filePathToStore, // Chemin absolu
                            'mime_type'      => $f->getMimeType(),
                        ]);
                    }
                    
                } catch (\Exception $e) {
                    \Log::error("Erreur attachment {$inputName}: " . $e->getMessage());
                }
            }
        }

        // -------------------------
        // EvenementSportif and related collections
        // -------------------------
        $evenement = EvenementSportif::where('demande_id', $demande->id_demande)->first();

        if ($request->has('nom_even')) {
            $evenData = [
                'nom_even' => $request->input('nom_even'),
                'objectifs' => $request->input('objectifs'),
                'contenus' => $request->input('contenus'),
                'date' => $request->input('date'),
                'heure' => $request->input('heure'),
                'instalation_sportives' => $request->input('instalation_sportives'),
                'reference' => $request->input('reference'),
                'estimation_evenement' => $request->input('estimation_evenement'),
                'cout_total_evenement' => $request->input('cout_total_evenement'),
                'difference_estimee' => $request->input('difference_estimee'),
                'recompense' => $request->input('recompense'),
                'moyen_transport' => $request->input('moyen_transport'),
                'user_id' => Auth::id(),
                'id_type' => $request->input('id_type', 2),
                'id_formulaire' => $request->input('id_formulaire', 15),
                'gouvernorat_id' => $request->input('gouvernorat_id'),
                'delegation_id' => $request->input('delegation_id'),
                'demande_id' => $demande->id_demande,
            ];

            if ($evenement) {
                $evenement->update($evenData);
            } else {
                $evenement = EvenementSportif::create($evenData);
            }

            // delete old related rows then re-create
            MembreOrgInvite::where('event_id', $evenement->id_even)->delete();
            AssocParticipante::where('event_id', $evenement->id_even)->delete();
            AthleteIndividuel::where('event_id', $evenement->id_even)->delete();
            SportifParticipant::where('even_id', $evenement->id_even)->delete();
            Etrangere::where('even_id', $evenement->id_even)->delete();

            if ($request->has('membres')) {
                foreach ($request->input('membres', []) as $membre) {
                    MembreOrgInvite::create([
                        'nom_prenom' => $membre['nom_prenom'],
                        'role' => $membre['role'] ?? null,
                        'nationalite' => $membre['nationalite'] ?? null,
                        'event_id' => $evenement->id_even,
                        'type_membre_id' => 1,
                    ]);
                }
            }

            if ($request->has('invites')) {
                foreach ($request->input('invites', []) as $invite) {
                    MembreOrgInvite::create([
                        'nom_prenom' => $invite['nom_prenom'],
                        'role' => $invite['role'] ?? null,
                        'nationalite' => $invite['nationalite'] ?? null,
                        'event_id' => $evenement->id_even,
                        'type_membre_id' => 2,
                    ]);
                }
            }

            if ($request->has('assoc_participantes')) {
                foreach ($request->input('assoc_participantes', []) as $assoc) {
                    $num_h = isset($assoc['num_athlete_h']) ? (int)$assoc['num_athlete_h'] : 0;
                    $num_f = isset($assoc['num_athlete_f']) ? (int)$assoc['num_athlete_f'] : 0;
                    $accomp = isset($assoc['accompagnants']) ? (int)$assoc['accompagnants'] : 0;
                    $total = $num_h + $num_f + $accomp;

                    AssocParticipante::create([
                        'nom' => $assoc['nom'] ?? null,
                        'num_athlete_h' => $num_h,
                        'num_athlete_f' => $num_f,
                        'accompagnants' => $accomp,
                        'total' => $total,
                        'event_id' => $evenement->id_even,
                    ]);
                }
            }

            if ($request->has('athletes')) {
                foreach ($request->input('athletes', []) as $athlete) {
                    AthleteIndividuel::create([
                        'nom_prenom_sportif' => $athlete['nom_prenom_sportif'],
                        'num_athlete_h' => $athlete['num_athlete_h'] ?? 0,
                        'num_athlete_f' => $athlete['num_athlete_f'] ?? 0,
                        'accompagnants' => $athlete['accompagnants'] ?? 0,
                        'event_id' => $evenement->id_even,
                    ]);
                }
            }

            if ($request->has('sportifs')) {
                foreach ($request->input('sportifs', []) as $sportif) {
                    SportifParticipant::create([
                        'nom_prenom' => $sportif['nom_prenom'],
                        'tranch_age' => $sportif['tranch_age'] ?? null,
                        'date_naissance' => $sportif['date_naissance'] ?? null,
                        'num_cin' => $sportif['num_cin'] ?? null,
                        'invitee' => $sportif['invitee'] ?? null,
                        'even_id' => $evenement->id_even,
                        'jeune_id' => $sportif['jeune_id'] ?? null,
                    ]);
                }
            }

            // touristes (Etrangere)
            if ($request->has('touristes')) {
                foreach ($request->input('touristes', []) as $touriste) {
                    Etrangere::create([
                        'nom_prenom' => $touriste['nom_prenom'] ?? null,
                        'pays_origine' => $touriste['pays_origine'] ?? null,
                        'num_passport' => $touriste['num_passport'] ?? null,
                        'date_arrive' => $touriste['date_arrive'] ?? null,
                        'lieu_residence' => $touriste['lieu_residence'] ?? null,
                        'date_depart' => $touriste['date_depart'] ?? null,
                        'even_id' => $evenement->id_even,
                        'type_etranger_id' => 1,
                    ]);
                }
            }
            if ($request->has('touriste2s')) {
                foreach ($request->input('touriste2s', []) as $touriste2) {
                    Etrangere::create([
                        'nom_prenom' => $touriste2['nom_prenom'] ?? null,
                        'pays_origine' => $touriste2['pays_origine'] ?? null,
                        'num_passport' => $touriste2['num_passport'] ?? null,
                        'date_arrive' => $touriste2['date_arrive'] ?? null,
                        'lieu_residence' => $touriste2['lieu_residence'] ?? null,
                        'date_depart' => $touriste2['date_depart'] ?? null,
                        'even_id' => $evenement->id_even,
                        'type_etranger_id' => 2,
                    ]);
                }
            }
            if ($request->has('touriste3s')) {
                foreach ($request->input('touriste3s', []) as $touriste3) {
                    Etrangere::create([
                        'nom_prenom' => $touriste3['nom_prenom'] ?? null,
                        'pays_origine' => $touriste3['pays_origine'] ?? null,
                        'num_passport' => $touriste3['num_passport'] ?? null,
                        'date_arrive' => $touriste3['date_arrive'] ?? null,
                        'lieu_residence' => $touriste3['lieu_residence'] ?? null,
                        'date_depart' => $touriste3['date_depart'] ?? null,
                        'even_id' => $evenement->id_even,
                        'type_etranger_id' => 3,
                    ]);
                }
            }
        } // end evenement handling

        // -------------------------
        // Status history creation (after update)
        // -------------------------
        $normalize = function ($v) {
            if (is_null($v) || $v === '') return null;
            $i = (int) $v;
            return ($i > 0) ? $i : null;
        };

        $old_id_type    = $normalize($ancienStatut);
        $old_statut_id  = $normalize($ancienStatut1);
        $old_statut2_id = $normalize($ancienStatut2);

        // refresh model to ensure we compare current persisted values
        $demande->refresh();

        $new_id_type    = $normalize($demande->id_type);
        $new_statut_id  = $normalize($demande->statut_id);
        $new_statut2_id = $normalize($demande->statut2_id);

        $statExists = function ($id) {
            if (is_null($id)) return false;
            return DB::table('tp_statut')->where('id_statut', $id)->exists();
        };

        $commentForHistory = null;
        if ($request->filled('commentaire_motif')) {
            $commentForHistory = trim($request->input('commentaire_motif'));
        } elseif ($request->filled('commentaire_motif2')) {
            $commentForHistory = trim($request->input('commentaire_motif2'));
        } elseif ($request->filled('commentaire_motif3')) {
            $commentForHistory = trim($request->input('commentaire_motif3'));
        }

        if ($old_id_type !== $new_id_type) {
            ImmobilierStatutHistory::create([
                'demande_manifestation_id' => $demande->id_demande,
                'from_statut_id' => $old_id_type,
                'to_statut_id' => $statExists($new_id_type) ? $new_id_type : null,
                'changed_by' => Auth::id() ?? null,
                'commentaire' => $commentForHistory,
            ]);
        }

        if ($old_statut_id !== $new_statut_id) {
            ImmobilierStatutHistory::create([
                'demande_manifestation_id' => $demande->id_demande,
                'from_statut_id' => $old_statut_id,
                'to_statut_id' => $statExists($new_statut_id) ? $new_statut_id : null,
                'changed_by' => Auth::id() ?? null,
                'commentaire' => $commentForHistory,
            ]);
        }

        if ($old_statut2_id !== $new_statut2_id) {
            ImmobilierStatutHistory::create([
                'demande_manifestation_id' => $demande->id_demande,
                'from_statut_id' => $old_statut2_id,
                'to_statut_id' => $statExists($new_statut2_id) ? $new_statut2_id : null,
                'changed_by' => Auth::id() ?? null,
                'commentaire' => $commentForHistory,
            ]);
        }

        // -------------------------
        // Commit DB transaction before running non-critical operations (mail/notifications)
        // -------------------------
        DB::commit();
        $committed = true;

        session()->forget('_old_input');

        // -------------------------
        // Notifications & Mail (run after commit; failures here should NOT roll back DB)
        // -------------------------
        try {
            // Refresh the model and user
            $demande = DemandeManifestation::with('utilisateur')->findOrFail($id);
            $user = $demande->utilisateur;

            // Determine which status was changed (prefer request inputs)
            $resolved = (int) ($request->input('id_type') ?? $request->input('statut_id') ?? $request->input('statut2_id') ?? 0);
            if ($resolved <= 0) {
                // nothing to notify, just redirect success
                return redirect()->route('admin.manifestation2.index', $demande->id_demande)
                                 ->with('success', 'تم تحديث الطلب بنجاح');
            }

            // Status normalization
            $statusMap = [
                1 => 'accepted',
                3 => 'rejected',
                4 => 'incomplete',
                5 => 'deferred',
            ];
            $type = $statusMap[$resolved] ?? 'status_changed';

            // Build dossier reference
            $idPad = str_pad($demande->id_demande, 3, '0', STR_PAD_LEFT);
            $cin = $user->num_cin ?? $user->cin ?? '';
            $cinDigits = preg_replace('/\D+/', '', $cin);
            $cinPad = $cinDigits ? str_pad($cinDigits, 9, '0', STR_PAD_LEFT) : '000000000';
            $year = $demande->created_at ? $demande->created_at->format('Y') : now()->format('Y');
            $num_dossier = "{$idPad}-{$cinPad}-{$year}";

            // Prepare comment (prefer saved model fields, fallback to request)
            $messageComment = trim(($demande->commentaire_motif ?? '') . ' ' . ($demande->commentaire_motif2 ?? '') . ' ' . ($demande->commentaire_motif3 ?? ''));
            if ($messageComment === '') {
                $messageComment = trim($request->input('commentaire_motif') ?? $request->input('commentaire_motif2') ?? $request->input('commentaire_motif3') ?? '');
            }
            $messageComment = $messageComment !== '' ? $messageComment : null;

            // Build HTML messageStatus (comments are escaped with e())
            $nomUtilisateur = $user->nom_prenom ?? 'المستخدم';
            $date = now()->format('Y-m-d');

            switch ($resolved) {
                case 1: // accepted
                    $messageStatus = "مرحباً <strong>{$nomUtilisateur}</strong>،<br>رقم الملف: <strong>{$num_dossier}</strong><br>تم قبول طلبكم بتاريخ: <strong>{$date}</strong>.<br>نهنئكم على هذا القبول.";
                    break;

                case 3: // rejected
                    $reason = e($messageComment ?? 'سبب غير محدد');
                    $messageStatus = "مرحباً <strong>{$nomUtilisateur}</strong>،<br>رقم الملف: <strong>{$num_dossier}</strong><br>تم رفض طلبكم.<br>السبب: <b>{$reason}</b>.";
                    break;

                case 4: // incomplete
                    $missing = e($request->input('commentaire_acceptation') ?? $messageComment ?? 'يرجى استكمال الوثائق المطلوبة');
                    $messageStatus = "مرحباً <strong>{$nomUtilisateur}</strong>،<br>رقم الملف: <strong>{$num_dossier}</strong><br>طلبكم <strong>غير مكتمل</strong>.<br>التوضيح: <b>{$missing}</b>.";
                    break;

                case 5: // deferred
                    $note = e($request->input('commentaire_avis') ?? $messageComment ?? 'لا يوجد ملاحظات');
                    $messageStatus = "مرحباً <strong>{$nomUtilisateur}</strong>،<br>رقم الملف: <strong>{$num_dossier}</strong><br>تم إرجاء طلبكم.<br>الملاحظة: <b>{$note}</b>.";
                    break;

                default:
                    $messageStatus = "مرحباً <strong>{$nomUtilisateur}</strong>،<br>رقم الملف: <strong>{$num_dossier}</strong><br>تم تحديث حالة ملفكم.";
            }

            // Build translation params for DB notification
            $params = [];
            if ($resolved === 3) $params['reason'] = $messageComment ?? '';
            if ($resolved === 4) $params['missing'] = $request->input('commentaire_acceptation') ?? ($messageComment ?? '');
            if ($resolved === 5) $params['note'] = $request->input('commentaire_avis') ?? ($messageComment ?? '');
            if ($resolved === 1) $params['date'] = $date;
            $params['reference'] = $num_dossier;
            if (!empty($messageComment)) $params['comment'] = $messageComment;

            // Send mail (best-effort)
            if (!empty($user->email)) {
                try {
                    Mail::to($user->email)->send(new \App\Mail\StatutDemande2Mail($user, $num_dossier, $messageStatus, $type, $messageComment));
                } catch (\Exception $ex) {
                    Log::error("Mail error after update for DemandeManifestation {$demande->id_demande}: " . $ex->getMessage());
                }
            }

            // DB Notification (best-effort)
            try {
                if ($user) {
                    $translationKey = "site.notifications.manifestation.{$type}";
                    $user->notify(new \App\Notifications\StatutDemande2Notification($demande->fresh('utilisateur'), $type, $params, $demande->id_demande, $translationKey));
                }
            } catch (\Exception $ex) {
                Log::error("Notification error after update for DemandeManifestation {$demande->id_demande}: " . $ex->getMessage());
            }

            return redirect()->route('admin.manifestation2.index', $demande->id_demande)
                             ->with('success', 'تم تحديث الطلب بنجاح');
        } catch (\Exception $e) {
            // Notification failures should not break the user flow. Log and fallthrough to success redirect.
            Log::error("Notification processing error after update for DemandeManifestation {$id}: " . $e->getMessage());
            // still redirect success because DB changes are already committed
            return redirect()->route('admin.manifestation2.index', $demande->id_demande)
                             ->with('success', 'تم تحديث الطلب بنجاح');
        }
    } catch (\Exception $e) {
        // Only rollback if we haven't already committed
        if (!$committed) {
            DB::rollBack();
        }
        Log::error("Error updating DemandeManifestation {$id}: " . $e->getMessage());
        return redirect()->back()->withInput()->with('error', 'حدث خطأ أثناء التحديث: ' . $e->getMessage());
    }
}

public function showDecision($id)
{
    $demande = DemandeManifestation::findOrFail($id);

    // Vérifier si le fichier existe
    if (empty($demande->file)) {
        abort(404, 'Aucun fichier disponible');
    }
    
    $uploadDir = '/home/preprov/www/mjs/shared_uploads/images';
    $filePath = $demande->file;
    
    // Si le chemin est relatif, construire le chemin absolu
    if (strpos($filePath, '/') !== 0) {
        $filePath = $uploadDir . '/' . basename($filePath);
    }
    
    // Vérifier l'existence du fichier
    if (!file_exists($filePath)) {
        // Essayer de trouver le fichier avec différentes possibilités
        $alternativePaths = [
            $uploadDir . '/' . basename($demande->file),
            $demande->file,
            public_path($demande->file),
            storage_path('app/' . $demande->file),
        ];
        
        foreach ($alternativePaths as $path) {
            if (file_exists($path)) {
                $filePath = $path;
                break;
            }
        }
        
        if (!file_exists($filePath)) {
            \Log::error("Fichier introuvable pour la demande {$id}: " . $demande->file);
            abort(404, 'Fichier introuvable');
        }
    }

    // Déterminer le Content-Type
    $extension = strtolower(pathinfo($filePath, PATHINFO_EXTENSION));
    $mimeTypes = [
        'pdf' => 'application/pdf',
        'jpg' => 'image/jpeg',
        'jpeg' => 'image/jpeg',
        'png' => 'image/png',
    ];

    $contentType = $mimeTypes[$extension] ?? mime_content_type($filePath);
    
    // Vérifier si le téléchargement est demandé
    if (request()->has('download')) {
        return response()->download($filePath, basename($filePath), [
            'Content-Type' => $contentType,
        ]);
    }

    // Afficher le fichier dans le navigateur
    return response()->file($filePath, [
        'Content-Type' => $contentType,
        'Content-Disposition' => 'inline; filename="' . basename($filePath) . '"',
        'Cache-Control' => 'no-store, no-cache, must-revalidate, max-age=0',
        'Pragma' => 'no-cache',
        'Expires' => '0',
    ]);
}

public function showfile2($id)
{
    $att = Attachement::findOrFail($id);

    if (empty($att->file)) {
        abort(404, 'Fichier introuvable');
    }
    
    $uploadDir = '/home/preprov/www/mjs/shared_uploads/images';
    $filePath = $att->file;
    
    // Si le chemin est relatif, construire le chemin absolu
    if (strpos($filePath, '/') !== 0) {
        $filePath = $uploadDir . '/' . basename($filePath);
    }
    
    // Vérifier l'existence du fichier
    if (!file_exists($filePath)) {
        // Essayer de trouver le fichier avec différentes possibilités
        $alternativePaths = [
            $uploadDir . '/' . basename($att->file),
            $att->file,
            public_path($att->file),
            storage_path('app/' . $att->file),
        ];
        
        foreach ($alternativePaths as $path) {
            if (file_exists($path)) {
                $filePath = $path;
                break;
            }
        }
        
        if (!file_exists($filePath)) {
            \Log::error("Fichier attachment introuvable: " . $att->file);
            abort(404, 'Fichier introuvable');
        }
    }

    // Déterminer le Content-Type
    $extension = strtolower(pathinfo($filePath, PATHINFO_EXTENSION));
    $mimeTypes = [
        'pdf' => 'application/pdf',
        'jpg' => 'image/jpeg',
        'jpeg' => 'image/jpeg',
        'png' => 'image/png',
    ];

    $contentType = $mimeTypes[$extension] ?? mime_content_type($filePath);
    
    // Vérifier si le téléchargement est demandé
    if (request()->has('download')) {
        return response()->download($filePath, basename($filePath), [
            'Content-Type' => $contentType,
        ]);
    }

    // Afficher le fichier dans le navigateur
    return response()->file($filePath, [
        'Content-Type' => $contentType,
        'Content-Disposition' => 'inline; filename="' . basename($filePath) . '"',
        'Cache-Control' => 'no-store, no-cache, must-revalidate, max-age=0',
        'Pragma' => 'no-cache',
        'Expires' => '0',
    ]);
}










/*public function update(Request $request, $id)
{
    $request->validate([
        'sujet' => 'nullable|string|max:255',
        'sport_type' => 'nullable|string|max:255',
        'event_name' => 'nullable|string|max:255',
        'objectif' => 'nullable|string',
        'commentaire_motif' => 'nullable|string',
        'type_formulaire_id' => 'required|integer',
        'statut_id' => 'nullable|integer',
        'statut2_id' => 'nullable|integer',
        'commentaire_motif2' => 'nullable|string|max:255',
        'commentaire_motif3' => 'nullable|string|max:255',
        'file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:4096',
        'id_type' => 'nullable|integer',
        'organisme_non_sportif_id' => 'nullable|integer',
        'role' => 'nullable|string',
        'periode_min' => 'nullable|date',
        'periode_max' => 'nullable|date|after_or_equal:periode_min',
        'lieu' => 'nullable|string|max:255',

        // Organisme non sportif : valider org comme array
        'org' => 'nullable|array',
        'org.nom' => 'nullable|string|max:255',
        'org.num_visa' => 'nullable|integer',
        'org.date_enregistrement' => 'nullable|date',
        'org.num_enregistrement' => 'nullable|string|max:50',
        'org.identifiant_fiscal' => 'nullable|numeric',
        'org.num_compte_bancaire' => 'nullable|string|max:50',
        'org.adresse' => 'nullable|string|max:255',
        'org.tel' => 'nullable|string|max:20',
        'org.fax' => 'nullable|string|max:20',
        'org.email' => 'nullable|email|max:255',
        'org.type_formulaire_id' => 'nullable|integer',

        // EvenementSportif
        'nom_even' => 'required_with:date|string|max:100',
        'objectifs' => 'nullable|string',
        'contenus' => 'nullable|string',
        'date' => 'required_with:nom_even|date',
        'heure' => 'nullable|date_format:H:i',
        'instalation_sportives' => 'nullable|string|max:200',
        'reference' => 'nullable|string|max:100',
        'estimation_evenement' => 'nullable|numeric',
        'cout_total_evenement' => 'nullable|numeric',
        'difference_estimee' => 'nullable|numeric',
        'recompense' => 'nullable|string',
        'moyen_transport' => 'nullable|string|max:200',
        'gouvernorat_id' => 'required_with:nom_even|integer',
        'delegation_id' => 'required_with:nom_even|integer',

        // collections
        'membres.*.nom_prenom' => 'required_with:membres|string|max:255',
        'invites.*.nom_prenom' => 'required_with:invites|string|max:255',
        'assoc_participantes.*.nom' => 'required_with:assoc_participantes|string|max:255',
        'athletes.*.nom_prenom_sportif' => 'nullable|string|max:255',
        'sportifs.*.nom_prenom' => 'required_with:sportifs|string|max:255',

        // attachements
        'assurance_file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
        'technique_file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
        'accord_file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
        'demande_file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
    ]);

    DB::beginTransaction();

    try {
        $demande = DemandeManifestation::with(['organisme', 'evenementSportifs'])->findOrFail($id);

        $ancienStatut = $demande->getOriginal('id_type');
        $ancienStatut1 = $demande->getOriginal('statut_id');
        $ancienStatut2 = $demande->getOriginal('statut2_id');

        // Récupérer les valeurs de la requête
        $newIdType = $request->input('id_type');
        $newStatutId = $request->input('statut_id');
        $newStatut2Id = $request->input('statut2_id');

        // Validation de la progression logique des statuts
        // Basée sur les NOUVELLES valeurs sélectionnées, pas les anciennes
        if ($newStatutId && !in_array($newIdType, [1, 4, 5])) {
            return back()->withErrors(['id_type' => 'يجب قبول الطلب أو إرجاؤه أو إعلامه بعدم الاكتمال من قبل الإدارة الفرعية أولاً'])->withInput();
        }
        
        if ($newStatut2Id && !in_array($newStatutId, [1, 4, 5])) {
            return back()->withErrors(['statut_id' => 'يجب قبول الطلب أو إرجاؤه أو إعلامه بعدم الاكتمال من قبل الجامعة أولاً'])->withInput();
        }
        
        // Validation pour le fichier lorsque statut2_id = 1 (accepté)
        if ($newStatut2Id == 1) {
            $hasFileInRequest = $request->hasFile('file');
            $hasExistingFile = !empty($demande->file);
            
            if (!$hasFileInRequest && !$hasExistingFile) {
                return back()->withErrors(['file' => 'مطلوب رفع ملف الترخيص عند القبول النهائي من اللجنة الوطنية'])->withInput();
            }
        }

        $organisme = null;
        $orgInput = $request->input('org', null);

        if (is_array($orgInput) && count(array_filter($orgInput)) > 0) {
            $orgInput['type_formulaire_id'] = $orgInput['type_formulaire_id'] ?? 14;

            if ($demande->organisme_non_sportif_id) {
                $organisme = OrganismeNonSportif::find($demande->organisme_non_sportif_id);
                if ($organisme) {
                    $organisme->fill($orgInput);
                    $organisme->save();
                } else {
                    $organisme = OrganismeNonSportif::create($orgInput);
                }
            } else {
                $organisme = OrganismeNonSportif::create($orgInput);
            }

            $pkName = $organisme->getKeyName();
            $demande->organisme_non_sportif_id = $organisme->{$pkName};
        }
        
        $uploadDir = '/home/preprov/www/mjs/shared_uploads/images';
        if (!is_dir($uploadDir) || !is_writable($uploadDir)) {
            abort(500, 'Dossier upload non accessible');
        }

        $filePath = $demande->file; 
        $fileName = $demande->file_name ?? null;
        $fileType = $demande->file_type ?? null;

        if ($request->hasFile('file')) {
            $file = $request->file('file');
            
            if (!empty($demande->file) && file_exists($demande->file)) {
                @unlink($demande->file);
            }
            
            $original = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
            $ext = $file->getClientOriginalExtension();
            $safe = Str::slug(mb_substr($original, 0, 40)) ?: 'document';
            
            $filename = 'demande_' . $demande->id_demande . '_' . $safe . '-' . time() . '-' . Str::random(6) . '.' . $ext;
            $targetPath = $uploadDir . DIRECTORY_SEPARATOR . $filename;
            
            $file->move($uploadDir, $filename);
            
            $realPath = realpath($targetPath);
            if ($realPath === false) {
                abort(500, 'Erreur lors de la sauvegarde du fichier');
            }
            
            $filePath = $realPath;
            $fileName = $file->getClientOriginalName();
            $fileType = $file->getMimeType();
        }

        // Préparer les données de mise à jour
        $updateData = [
            'sujet' => $request->sujet,
            'sport_type' => $request->sport_type,
            'event_name' => $request->event_name,
            'objectif' => $request->objectif,
            'commentaire_motif' => $request->commentaire_motif,
            'commentaire_motif2' => $request->commentaire_motif2,
            'commentaire_motif3' => $request->commentaire_motif3,
            'type_formulaire_id' => $request->type_formulaire_id ?? $demande->type_formulaire_id,
            'user_id' => $demande->user_id,
            'role' => $request->role,
            'periode_min' => $request->periode_min,
            'periode_max' => $request->periode_max,
            'lieu' => $request->lieu,
            'organisme_non_sportif_id' => $demande->organisme_non_sportif_id,
            'file' => $filePath,
            'file_name' => $fileName,
            'file_type' => $fileType,
        ];

        // Mettre à jour les statuts avec les nouvelles valeurs
        // Si une valeur est fournie dans la requête, on l'utilise
        // Sinon, on garde l'ancienne valeur
        if ($request->has('id_type')) {
            $updateData['id_type'] = $newIdType;
            
            // Si on refuse au niveau 1, réinitialiser les niveaux suivants
            if ($newIdType == 3) {
                $updateData['statut_id'] = null;
                $updateData['statut2_id'] = null;
                $updateData['commentaire_motif2'] = null;
                $updateData['commentaire_motif3'] = null;
            }
        }
        
        if ($request->has('statut_id')) {
            $updateData['statut_id'] = $newStatutId;
            
            // Si on refuse au niveau 2, réinitialiser le niveau 3
            if ($newStatutId == 3) {
                $updateData['statut2_id'] = null;
                $updateData['commentaire_motif3'] = null;
            }
        }
        
        if ($request->has('statut2_id')) {
            $updateData['statut2_id'] = $newStatut2Id;
        }

        // Mettre à jour la demande
        $demande->update($updateData);

        // Normalize integers helper
        $normalize = function ($v) {
            if (is_null($v) || $v === '') return null;
            $i = (int)$v;
            return ($i > 0) ? $i : null;
        };

        // compute old and new values for comparison
        $old_id_type = $normalize($ancienStatut);
        $old_statut_id = $normalize($ancienStatut1);
        $old_statut2_id = $normalize($ancienStatut2);

        // reload fresh model to get saved values
        $demande->refresh();

        $new_id_type = $normalize($demande->id_type);
        $new_statut_id = $normalize($demande->statut_id);
        $new_statut2_id = $normalize($demande->statut2_id);

        // helper to check tp_statut existence
        $statExists = function ($id) {
            if (is_null($id)) return false;
            return DB::table('tp_statut')->where('id_statut', $id)->exists();
        };

        // Déterminer le commentaire pour l'historique
        $commentForHistory = null;
        if ($old_id_type !== $new_id_type && $request->filled('commentaire_motif')) {
            $commentForHistory = trim($request->input('commentaire_motif'));
        } elseif ($old_statut_id !== $new_statut_id && $request->filled('commentaire_motif2')) {
            $commentForHistory = trim($request->input('commentaire_motif2'));
        } elseif ($old_statut2_id !== $new_statut2_id && $request->filled('commentaire_motif3')) {
            $commentForHistory = trim($request->input('commentaire_motif3'));
        }

        // create a history row for each changed slot (use demande_manifestation_id)
        if ($old_id_type !== $new_id_type) {
            ImmobilierStatutHistory::create([
                'demande_manifestation_id' => $demande->id_demande,
                'from_statut_id'           => $old_id_type,
                'to_statut_id'             => $statExists($new_id_type) ? $new_id_type : null,
                'changed_by'               => Auth::id() ?? null,
                'commentaire'              => $commentForHistory,
            ]);
        }

        if ($old_statut_id !== $new_statut_id) {
            ImmobilierStatutHistory::create([
                'demande_manifestation_id' => $demande->id_demande,
                'from_statut_id'           => $old_statut_id,
                'to_statut_id'             => $statExists($new_statut_id) ? $new_statut_id : null,
                'changed_by'               => Auth::id() ?? null,
                'commentaire'              => $commentForHistory,
            ]);
        }

        if ($old_statut2_id !== $new_statut2_id) {
            ImmobilierStatutHistory::create([
                'demande_manifestation_id' => $demande->id_demande,
                'from_statut_id'           => $old_statut2_id,
                'to_statut_id'             => $statExists($new_statut2_id) ? $new_statut2_id : null,
                'changed_by'               => Auth::id() ?? null,
                'commentaire'              => $commentForHistory,
            ]);
        }

        $attachmentsMap = [
            'assurance_file' => 2,
            'accord_file'    => 7,
            'technique_file' => 8,
            'demande_file'   => 9,
        ];

        foreach ($attachmentsMap as $inputName => $typeId) {
            if ($request->hasFile($inputName)) {
                $f = $request->file($inputName);
                
                $oldAttachment = Attachement::where('id_demande', $demande->id_demande)
                    ->where('id_type_attach', $typeId)
                    ->first();

                if ($oldAttachment && !empty($oldAttachment->file) && file_exists($oldAttachment->file)) {
                    @unlink($oldAttachment->file);
                }

                $original = pathinfo($f->getClientOriginalName(), PATHINFO_FILENAME);
                $ext = $f->getClientOriginalExtension();
                $safe = Str::slug(mb_substr($original, 0, 40)) ?: 'document';
                
                $filename = 'attachement_' . $demande->id_demande . '_' . $typeId . '_' . $safe . '-' . time() . '-' . Str::random(6) . '.' . $ext;
                $targetPath = $uploadDir . DIRECTORY_SEPARATOR . $filename;
                
                $f->move($uploadDir, $filename);
                
                $realPath = realpath($targetPath);
                if ($realPath === false) {
                    abort(500, 'Erreur lors de la sauvegarde du fichier');
                }

                if ($oldAttachment) {
                    $oldAttachment->update([
                        'file' => $realPath, 
                        'mime_type' => $f->getMimeType(),
                    ]);
                } else {
                    Attachement::create([
                        'id_demande'     => $demande->id_demande,
                        'id_type_attach' => $typeId,
                        'file'           => $realPath, 
                        'mime_type'      => $f->getMimeType(),
                    ]);
                }
            }
        }

        // Evenement sportif logic
        $evenement = EvenementSportif::where('demande_id', $demande->id_demande)->first();

        if ($request->has('nom_even')) {
            $evenData = [
                'nom_even' => $request->nom_even,
                'objectifs' => $request->objectifs,
                'contenus' => $request->contenus,
                'date' => $request->date,
                'heure' => $request->heure,
                'instalation_sportives' => $request->instalation_sportives,
                'reference' => $request->reference,
                'estimation_evenement' => $request->estimation_evenement,
                'cout_total_evenement' => $request->cout_total_evenement,
                'difference_estimee' => $request->difference_estimee,
                'recompense' => $request->recompense,
                'moyen_transport' => $request->moyen_transport,
                'user_id' => Auth::id(),
                'id_type' => $request->id_type ?? 2,
                'id_formulaire' => $request->id_formulaire ?? 15,
                'gouvernorat_id' => $request->gouvernorat_id,
                'delegation_id' => $request->delegation_id,
                'demande_id' => $demande->id_demande,
            ];

            if ($evenement) {
                $evenement->update($evenData);
            } else {
                $evenement = EvenementSportif::create($evenData);
            }

            // Supprimer les anciennes données seulement si on a de nouvelles données
            if ($request->has('membres') || $request->has('invites') || 
                $request->has('assoc_participantes') || $request->has('athletes') || 
                $request->has('sportifs')) {
                
                MembreOrgInvite::where('event_id', $evenement->id_even)->delete();
                AssocParticipante::where('event_id', $evenement->id_even)->delete();
                AthleteIndividuel::where('event_id', $evenement->id_even)->delete();
                SportifParticipant::where('even_id', $evenement->id_even)->delete();

                if ($request->has('membres')) {
                    foreach ($request->membres as $membre) {
                        if (!empty($membre['nom_prenom'])) {
                            MembreOrgInvite::create([
                                'nom_prenom' => $membre['nom_prenom'],
                                'role' => $membre['role'] ?? null,
                                'nationalite' => $membre['nationalite'] ?? null,
                                'event_id' => $evenement->id_even,
                                'type_membre_id' => 1,
                            ]);
                        }
                    }
                }

                if ($request->has('invites')) {
                    foreach ($request->invites as $invite) {
                        if (!empty($invite['nom_prenom'])) {
                            MembreOrgInvite::create([
                                'nom_prenom' => $invite['nom_prenom'],
                                'role' => $invite['role'] ?? null,
                                'nationalite' => $invite['nationalite'] ?? null,
                                'event_id' => $evenement->id_even,
                                'type_membre_id' => 2,
                            ]);
                        }
                    }
                }

                if ($request->has('assoc_participantes')) {
                    foreach ($request->assoc_participantes as $assoc) {
                        if (!empty($assoc['nom'])) {
                            $num_h = isset($assoc['num_athlete_h']) ? (int)$assoc['num_athlete_h'] : 0;
                            $num_f = isset($assoc['num_athlete_f']) ? (int)$assoc['num_athlete_f'] : 0;
                            $accomp = isset($assoc['accompagnants']) ? (int)$assoc['accompagnants'] : 0;
                            $total = $num_h + $num_f + $accomp;

                            AssocParticipante::create([
                                'nom' => $assoc['nom'] ?? null,
                                'num_athlete_h' => $num_h,
                                'num_athlete_f' => $num_f,
                                'accompagnants' => $accomp,
                                'total' => $total,
                                'event_id' => $evenement->id_even,
                            ]);
                        }
                    }
                }

                if ($request->has('athletes')) {
                    foreach ($request->athletes as $athlete) {
                        if (!empty($athlete['nom_prenom_sportif'])) {
                            AthleteIndividuel::create([
                                'nom_prenom_sportif' => $athlete['nom_prenom_sportif'],
                                'num_athlete_h' => $athlete['num_athlete_h'] ?? 0,
                                'num_athlete_f' => $athlete['num_athlete_f'] ?? 0,
                                'accompagnants' => $athlete['accompagnants'] ?? 0,
                                'event_id' => $evenement->id_even,
                            ]);
                        }
                    }
                }

                if ($request->has('sportifs')) {
                    foreach ($request->sportifs as $sportif) {
                        if (!empty($sportif['nom_prenom'])) {
                            SportifParticipant::create([
                                'nom_prenom' => $sportif['nom_prenom'],
                                'tranch_age' => $sportif['tranch_age'] ?? null,
                                'date_naissance' => $sportif['date_naissance'] ?? null,
                                'num_cin' => $sportif['num_cin'] ?? null,
                                'invitee' => $sportif['invitee'] ?? null,
                                'even_id' => $evenement->id_even,
                                'jeune_id' => $sportif['jeune_id'] ?? null,
                            ]);
                        }
                    }
                }
            }
        }

        DB::commit();

        // -------------------------
        // Notifications after commit
        // -------------------------

        try {
            $d = $demande->fresh(['utilisateur']);

            // Déterminer quel statut a été modifié pour la notification
            $resolved = null;
            if ($old_id_type !== $new_id_type) {
                $resolved = $new_id_type;
            } elseif ($old_statut_id !== $new_statut_id) {
                $resolved = $new_statut_id;
            } elseif ($old_statut2_id !== $new_statut2_id) {
                $resolved = $new_statut2_id;
            } else {
                $resolved = $new_id_type;
            }

            $statusMap = [1 => 'accepted', 3 => 'rejected', 4 => 'incomplete', 5 => 'deferred'];
            $type = isset($statusMap[$resolved]) ? $statusMap[$resolved] : 'status_changed';

            // build the messageComment from the saved model or request
            $messageComment = trim(($d->commentaire_motif ?? '') . ' ' . ($d->commentaire_motif2 ?? '') . ' ' . ($d->commentaire_motif3 ?? ''));
            $messageComment = $messageComment !== '' ? $messageComment : null;

            // dossier number
            $idPad = str_pad($d->id_demande, 3, '0', STR_PAD_LEFT);
            $cin = $d->utilisateur->num_cin ?? $d->utilisateur->cin ?? '';
            $cinDigits = preg_replace('/\D+/', '', $cin);
            $cinPad = $cinDigits ? str_pad($cinDigits, 9, '0', STR_PAD_LEFT) : '000000000';
            $year = $d->created_at ? $d->created_at->format('Y') : now()->format('Y');
            $num_dossier = "{$idPad}-{$cinPad}-{$year}";

            $nomUtilisateur = $d->utilisateur->nom_prenom ?? 'المستخدم';
            $date = now()->format('Y-m-d');

            switch ($resolved) {
                case 1:
                    $messageStatus = "مرحباً <strong>{$nomUtilisateur}</strong>،<br>رقم الملف: <strong>{$num_dossier}</strong><br>تم قبول طلبكم بتاريخ: <strong>{$date}</strong>.<br>نهنئكم على هذا القبول.";
                    break;
                case 3:
                    $reason = e($messageComment ?? ($request->input('commentaire_motif') ?? 'سبب غير محدد'));
                    $messageStatus = "مرحباً <strong>{$nomUtilisateur}</strong>،<br>رقم الملف: <strong>{$num_dossier}</strong><br>تم رفض طلبكم.<br>التعليق: <b>{$reason}</b>.";
                    break;
                case 4:
                    $missing = e($request->input('commentaire_acceptation') ?? ($messageComment ?? 'يرجى استكمال الوثائق المطلوبة'));
                    $messageStatus = "مرحباً <strong>{$nomUtilisateur}</strong>،<br>رقم الملف: <strong>{$num_dossier}</strong><br>طلبكم <strong>غير مكتمل</strong>.<br>التوضيح: <b>{$missing}</b>.";
                    break;
                case 5:
                    $note = e($request->input('commentaire_avis') ?? ($messageComment ?? 'لا يوجد ملاحظات'));
                    $messageStatus = "مرحباً <strong>{$nomUtilisateur}</strong>،<br>رقم الملف: <strong>{$num_dossier}</strong><br>تم إرجاء طلبكم.<br>الملاحظة: <b>{$note}</b>.";
                    break;
                default:
                    $messageStatus = "مرحباً <strong>{$nomUtilisateur}</strong>،<br>رقم الملف: <strong>{$num_dossier}</strong><br>تم تحديث حالة ملفكم.";
            }

            // send mail
            if (!empty($d->utilisateur->email)) {
                try {
                    Mail::to($d->utilisateur->email)->send(
                        new \App\Mail\StatutDemandeMail($d->utilisateur, $num_dossier, $messageStatus, $type, $messageComment)
                    );
                } catch (\Exception $ex) {
                    Log::error("Mail error after update for DemandeManifestation {$d->id_demande}: " . $ex->getMessage());
                }
            }

            // DB notification
            if ($d->utilisateur) {
                $translationKey = "site.notifications.manifestation.{$type}";
                $d->utilisateur->notify(new \App\Notifications\StatutDemandeNotification(
                    $d, $type, [
                        'reason' => $messageComment ?? '',
                        'reference' => $num_dossier
                    ],
                    $d->id_demande, $translationKey
                ));
            }
        } catch (\Exception $ex) {
            Log::error("Notification error after update for DemandeManifestation {$demande->id_demande}: " . $ex->getMessage());
        }

        session()->forget('_old_input');

        return redirect()->route('admin.manifestation.index')
                         ->with('success', 'تم تحديث الطلب بنجاح');
    } catch (\Exception $e) {
        DB::rollBack();
        Log::error('Update error: ' . $e->getMessage());
        Log::error('Trace: ' . $e->getTraceAsString());
        return redirect()->back()->withInput()->with('error', 'حدث خطأ أثناء التحديث: ' . $e->getMessage());
    }
}

*/


/*
 public function update(Request $request, $id)
    {
        $request->validate([
            'sujet' => 'nullable|string|max:255',
            'sport_type' => 'nullable|string|max:255',
            'event_name' => 'nullable|string|max:255',
            'objectif' => 'nullable|string',
            'commentaire_motif' => 'nullable|string',
            'type_formulaire_id' => 'required|integer',
            'statut_id' => 'nullable|integer',
            'statut2_id' => 'nullable|integer',
            'commentaire_motif2' => 'nullable|string|max:255',
            'commentaire_motif3' => 'nullable|string|max:255',
            'file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:4096',
            'id_type' => 'nullable|integer',
            'organisme_non_sportif_id' => 'nullable|integer',
            'role' => 'nullable|string',
            'periode_min' => 'nullable|date',
            'periode_max' => 'nullable|date|after_or_equal:periode_min',
            'lieu' => 'nullable|string|max:255',

            // Organisme non sportif : valider org comme array
            'org' => 'nullable|array',
            'org.nom' => 'nullable|string|max:255',
            'org.num_visa' => 'nullable|integer',
            'org.date_enregistrement' => 'nullable|date',
            'org.num_enregistrement' => 'nullable|string|max:50',
            'org.identifiant_fiscal' => 'nullable|numeric',
            'org.num_compte_bancaire' => 'nullable|string|max:50',
            'org.adresse' => 'nullable|string|max:255',
            'org.tel' => 'nullable|string|max:20',
            'org.fax' => 'nullable|string|max:20',
            'org.email' => 'nullable|email|max:255',
            'org.type_formulaire_id' => 'nullable|integer',

            // EvenementSportif
            'nom_even' => 'required_with:date|string|max:100',
            'objectifs' => 'nullable|string',
            'contenus' => 'nullable|string',
            'date' => 'required_with:nom_even|date',
            'heure' => 'nullable|date_format:H:i',
            'instalation_sportives' => 'nullable|string|max:200',
            'reference' => 'nullable|string|max:100',
            'estimation_evenement' => 'nullable|numeric',
            'cout_total_evenement' => 'nullable|numeric',
            'difference_estimee' => 'nullable|numeric',
            'recompense' => 'nullable|string',
            'moyen_transport' => 'nullable|string|max:200',
            'gouvernorat_id' => 'required_with:nom_even|integer',
            'delegation_id' => 'required_with:nom_even|integer',

            // collections
            'membres.*.nom_prenom' => 'required_with:membres|string|max:255',
            'invites.*.nom_prenom' => 'required_with:invites|string|max:255',
            'assoc_participantes.*.nom' => 'required_with:assoc_participantes|string|max:255',
            'athletes.*.nom_prenom_sportif' => 'nullable|string|max:255',
            'sportifs.*.nom_prenom' => 'required_with:sportifs|string|max:255',

            // attachements
            'assurance_file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
            'technique_file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
            'accord_file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
            'demande_file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
        ]);

        DB::beginTransaction();

        try {
            $demande = DemandeManifestation::findOrFail($id);

            $organisme = null;
            $orgInput = $request->input('org', null);

            if (is_array($orgInput) && count(array_filter($orgInput)) > 0) {
                $orgInput['type_formulaire_id'] = $orgInput['type_formulaire_id'] ?? 14;

                if ($demande->organisme_non_sportif_id) {
                    $organisme = OrganismeNonSportif::find($demande->organisme_non_sportif_id);
                    if ($organisme) {
                        $organisme->fill($orgInput);
                        $organisme->save();
                    } else {
                        $organisme = OrganismeNonSportif::create($orgInput);
                    }
                } else {
                    $organisme = OrganismeNonSportif::create($orgInput);
                }

                $pkName = $organisme->getKeyName(); 
                $demande->organisme_non_sportif_id = $organisme->{$pkName};
            } else {
                
            }

          //  /*if ($request->hasFile('file')) {
           //     if (!empty($demande->file) && Storage::disk('public')->exists($demande->file)) {
           //         Storage::disk('public')->delete($demande->file);
            //    }
           //     $filePath = $request->file('file')->store('manifestations/files', 'public');
         //   } else {
            //    $filePath = $demande->file;
         //   }
			
			$filePath = $demande->file; // garder l'ancien par défaut

			if ($request->hasFile('file')) {

                // Lire le contenu du fichier
                $fileContent = file_get_contents($request->file('file')->getRealPath());

                // Sauvegarder directement dans la base (LONGBLOB)
                $demande->file = $fileContent;

                // (optionnel) sauvegarder le nom et le type
                $demande->file_name = $request->file('file')->getClientOriginalName();
                $demande->file_type = $request->file('file')->getMimeType();
            }

            $demande->save();



            $ancienStatut = $demande->getOriginal('id_type');
            $ancienStatut1 = $demande->getOriginal('statut_id');
            $ancienStatut2 = $demande->getOriginal('statut2_id');

            $demande->update([
                'sujet' => $request->sujet,
                'sport_type' => $request->sport_type,
                'event_name' => $request->event_name,
                'objectif' => $request->objectif,
                'commentaire_motif' => $request->commentaire_motif,
                'commentaire_motif2' => $request->commentaire_motif2,
                'commentaire_motif3' => $request->commentaire_motif3,
                'type_formulaire_id' => $request->type_formulaire_id ?? $demande->type_formulaire_id,
                //'user_id' => Auth::id(),
                'user_id' => $demande->user_id,
                'id_type' => $request->id_type ?? $demande->id_type,
                //'id_type' => $idType,

                'statut_id' => $request->statut_id ?? $demande->statut_id,
                'statut2_id' => $request->statut2_id ?? $demande->statut2_id,
                'role' => $request->role,
                'periode_min' => $request->periode_min,
                'periode_max' => $request->periode_max,
                'lieu' => $request->lieu,
                'organisme_non_sportif_id' => $demande->organisme_non_sportif_id,
                'file' => $filePath,
            ]);

            
        if (
            $demande->id_type != $ancienStatut ||
            $demande->statut_id != $ancienStatut1 ||
            $demande->statut2_id != $ancienStatut2
        ) {
            // Trouver le bon statut à notifier
        $nouveauStatut = $request->id_type ?? $request->statut_id ?? $request->statut2_id;
        $old_id_type    = $demande->getOriginal('id_type');     // id_type (ادارة فرعية)
        $old_statut_id  = $demande->getOriginal('statut_id');   // الجامعة
        $old_statut2_id = $demande->getOriginal('statut2_id');  // اللجنة

        $demande->refresh();

        // after you have done $demande->update([...]) and $demande->refresh();
      $normalize = function ($v) {
            if (is_null($v) || $v === '') return null;
            $i = (int)$v;
            return ($i > 0) ? $i : null;
        };

        $old_id_type    = $normalize($ancienStatut);   // original values you captured earlier
        $old_statut_id  = $normalize($ancienStatut1);
        $old_statut2_id = $normalize($ancienStatut2);

        $new_id_type    = $normalize($demande->id_type);
        $new_statut_id  = $normalize($demande->statut_id);
        $new_statut2_id = $normalize($demande->statut2_id);

        // helper to check tp_statut existence
        $statExists = function($id) {
            if (is_null($id)) return false;
            return DB::table('tp_statut')->where('id_statut', $id)->exists();
        };

        $commentForHistory = null;
        if ($request->filled('commentaire_motif')) {
            $commentForHistory = trim($request->input('commentaire_motif'));
        } elseif ($request->filled('commentaire_motif2')) {
            $commentForHistory = trim($request->input('commentaire_motif2'));
        } elseif ($request->filled('commentaire_motif3')) {
            $commentForHistory = trim($request->input('commentaire_motif3'));
        }

        // create a history row for each changed slot (use demande_manifestation_id)
        if ($old_id_type !== $new_id_type) {
            ImmobilierStatutHistory::create([
                'demande_manifestation_id' => $demande->id_demande,
                'from_statut_id'           => $old_id_type,
                'to_statut_id'             => $statExists($new_id_type) ? $new_id_type : null,
                'changed_by'               => Auth::id() ?? null,
                'commentaire'              => $commentForHistory,
            ]);
        }

        if ($old_statut_id !== $new_statut_id) {
            ImmobilierStatutHistory::create([
                'demande_manifestation_id' => $demande->id_demande,
                'from_statut_id'           => $old_statut_id,
                'to_statut_id'             => $statExists($new_statut_id) ? $new_statut_id : null,
                'changed_by'               => Auth::id() ?? null,
                'commentaire'              => $commentForHistory,
            ]);
        }

        if ($old_statut2_id !== $new_statut2_id) {
            ImmobilierStatutHistory::create([
                'demande_manifestation_id' => $demande->id_demande,
                'from_statut_id'           => $old_statut2_id,
                'to_statut_id'             => $statExists($new_statut2_id) ? $new_statut2_id : null,
                'changed_by'               => Auth::id() ?? null,
                'commentaire'              => $commentForHistory,
            ]);
        }


     $this->envoyerNotificationEtMail($demande->fresh(['utilisateur']), $nouveauStatut);
     }
 
            $attachmentsMap = [
                'assurance_file' => 2, 
                'accord_file' => 7,    
                'technique_file' => 8,
                'demande_file' => 9,
            ];

            $attachmentsMap = [
                    'assurance_file' => 2,
                    'accord_file'    => 7,
                    'technique_file' => 8,
                    'demande_file'   => 9,
                ];

                foreach ($attachmentsMap as $inputName => $typeId) {

                    if ($request->hasFile($inputName)) {

                        $file = $request->file($inputName);

                        // Chercher l'ancien attachement
                        $oldAttachment = Attachement::where('id_demande', $demande->id_demande)
                            ->where('id_type_attach', $typeId)
                            ->first();

                        $data = [
                            'file'       => file_get_contents($file->getRealPath()), // ✅ LONGBLOB
                            'mime_type' => $file->getMimeType(),                     // ⭐ recommandé
                        ];

                        if ($oldAttachment) {
                            // UPDATE
                            $oldAttachment->update($data);
                        } else {
                            // CREATE
                            Attachement::create([
                                'id_demande'      => $demande->id_demande,
                                'id_type_attach'  => $typeId,
                                'file'            => $data['file'],
                                'mime_type'       => $data['mime_type'],
                            ]);
                        }
                    }
                }


            $evenement = EvenementSportif::where('demande_id', $demande->id_demande)->first();

            if ($request->has('nom_even')) {
                $evenData = [
                    'nom_even' => $request->nom_even,
                    'objectifs' => $request->objectifs,
                    'contenus' => $request->contenus,
                    'date' => $request->date,
                    'heure' => $request->heure,
                    'instalation_sportives' => $request->instalation_sportives,
                    'reference' => $request->reference,
                    'estimation_evenement' => $request->estimation_evenement,
                    'cout_total_evenement' => $request->cout_total_evenement,
                    'difference_estimee' => $request->difference_estimee,
                    'recompense' => $request->recompense,
                    'moyen_transport' => $request->moyen_transport,
                    'user_id' => Auth::id(),
                    'id_type' => $request->id_type ?? 2,
                    'id_formulaire' => $request->id_formulaire ?? 15,
                    'gouvernorat_id' => $request->gouvernorat_id,
                    'delegation_id' => $request->delegation_id,
                    'demande_id' => $demande->id_demande,
                ];

                if ($evenement) {
                    $evenement->update($evenData);
                } else {
                    $evenement = EvenementSportif::create($evenData);
                }

                MembreOrgInvite::where('event_id', $evenement->id_even)->delete();
                AssocParticipante::where('event_id', $evenement->id_even)->delete();
                AthleteIndividuel::where('event_id', $evenement->id_even)->delete();
                SportifParticipant::where('even_id', $evenement->id_even)->delete();

                if ($request->has('membres')) {
                    foreach ($request->membres as $membre) {
                        MembreOrgInvite::create([
                            'nom_prenom' => $membre['nom_prenom'],
                            'role' => $membre['role'] ?? null,
                            'nationalite' => $membre['nationalite'] ?? null,
                            'event_id' => $evenement->id_even,
                            'type_membre_id' => 1,
                        ]);
                    }
                }

                if ($request->has('invites')) {
                    foreach ($request->invites as $invite) {
                        MembreOrgInvite::create([
                            'nom_prenom' => $invite['nom_prenom'],
                            'role' => $invite['role'] ?? null,
                            'nationalite' => $invite['nationalite'] ?? null,
                            'event_id' => $evenement->id_even,
                            'type_membre_id' => 2,
                        ]);
                    }
                }

                if ($request->has('assoc_participantes')) {
                    foreach ($request->assoc_participantes as $assoc) {
                        $num_h = isset($assoc['num_athlete_h']) ? (int) $assoc['num_athlete_h'] : 0;
                        $num_f = isset($assoc['num_athlete_f']) ? (int) $assoc['num_athlete_f'] : 0;
                        $accomp = isset($assoc['accompagnants']) ? (int) $assoc['accompagnants'] : 0;
                        $total = $num_h + $num_f + $accomp;

                        AssocParticipante::create([
                            'nom' => $assoc['nom'] ?? null,
                            'num_athlete_h' => $num_h,
                            'num_athlete_f' => $num_f,
                            'accompagnants' => $accomp,
                            'total' => $total,
                            'event_id' => $evenement->id_even,
                        ]);
                    }
                }

                if ($request->has('athletes')) {
                    foreach ($request->athletes as $athlete) {
                        AthleteIndividuel::create([
                            'nom_prenom_sportif' => $athlete['nom_prenom_sportif'],
                            'num_athlete_h' => $athlete['num_athlete_h'] ?? 0,
                            'num_athlete_f' => $athlete['num_athlete_f'] ?? 0,
                            'accompagnants' => $athlete['accompagnants'] ?? 0,
                            'event_id' => $evenement->id_even,
                        ]);
                    }
                }

                if ($request->has('sportifs')) {
                    foreach ($request->sportifs as $sportif) {
                        SportifParticipant::create([
                            'nom_prenom' => $sportif['nom_prenom'],
                            'tranch_age' => $sportif['tranch_age'] ?? null,
                            'date_naissance' => $sportif['date_naissance'] ?? null,
                            'num_cin' => $sportif['num_cin'] ?? null,
                            'invitee' => $sportif['invitee'] ?? null,
                            'even_id' => $evenement->id_even,
                            'jeune_id' => $sportif['jeune_id'] ?? null,
                        ]);
                    }
                }
            }
            DB::commit();
   // -------------------------
        // Notifications after commit
        // -------------------------
        try {
            $d = $demande->fresh(['utilisateur']);
            // determine which statut was changed (prefer request values)
            $resolved = (int) ($request->id_type ?? $request->statut_id ?? $request->statut2_id ?? 0);

            $statusMap = [
                1 => 'accepted',
                3 => 'rejected',
                4 => 'incomplete',
                5 => 'deferred',
            ];
            $type = $statusMap[$resolved] ?? 'status_changed';

            // build params
            $params = [];
            if ($resolved === 3) {
                $params['reason'] = $request->input('commentaire_motif')
                                    ?? $request->input('commentaire_motif2')
                                    ?? $request->input('commentaire_motif3')
                                    ?? ($commentForHistory ?? 'سبب غير محدد');
            } elseif ($resolved === 4) {
                $params['missing'] = $request->input('commentaire_acceptation') ?? ($commentForHistory ?? '');
            } elseif ($resolved === 5) {
                $params['note'] = $request->input('commentaire_avis') ?? ($commentForHistory ?? '');
            } elseif ($resolved === 1) {
                $params['date'] = now()->format('Y-m-d');
            }

            // Send mail (if email exists) — keep existing mail behaviour
            if (!empty($d->utilisateur->email)) {
                try {
                    $mailMessage = ($resolved == 1)
                        ? "مرحباً {$d->utilisateur->nom_prenom}،\nرقم الملف: {$d->id_demande}\nتم قبول طلبكم بتاريخ: " . ($params['date'] ?? now()->format('Y-m-d')) . "."
                        : "مرحباً {$d->utilisateur->nom_prenom}،\nرقم الملف: {$d->id_demande}\nتم رفض طلبكم. السبب: " . ($params['reason'] ?? '');
                    Mail::to($d->utilisateur->email)->send(
                        new StatutDemandeMail($d->utilisateur, $d->id_demande, $mailMessage)
                    );
                } catch (\Exception $ex) {
                    Log::error("Mail error after update for DemandeManifestation {$d->id_demande}: " . $ex->getMessage());
                }
            }

            // Store DB notification
            if ($d->utilisateur) {
                $translationKey = "site.notifications.manifestation.{$type}";
                $d->utilisateur->notify(new \App\Notifications\StatutDemandeNotification(
                    $d,
                    $type,
                    $params,
                    $d->id_demande,
                    $translationKey
                ));
            } else {
                Log::warning("No utilisateur to notify for DemandeManifestation {$d->id_demande} after update.");
            }
        } catch (\Exception $ex) {
            Log::error("Notification error after commit for DemandeManifestation {$demande->id_demande}: " . $ex->getMessage());
        }
        session()->forget('_old_input');
            return redirect()->route('admin.manifestation.index', $demande->id_demande)
                            ->with('success', 'تم تحديث الطلب بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->withInput()->with('error', 'حدث خطأ أثناء التحديث: ' . $e->getMessage());
        }
    }
*/
    public function destroy($id)
    {
        $demande = DemandeManifestation::findOrFail($id);

        foreach ($demande->attachements as $attachement) {
            if ($attachement->chemin && Storage::exists($attachement->chemin)) {
                Storage::delete($attachement->chemin);
            }
            $attachement->delete();
        }

        if ($demande->teamsAssociations) {
            foreach ($demande->teamsAssociations as $team) {
                $team->delete();
            }
        }

        if ($demande->evenementSportif) {
            $demande->evenementSportif->delete();
        }

        if ($demande->organismeNonSportif) {
            $demande->organismeNonSportif->delete();
        }

        $demande->delete();

        return redirect()->route('admin.manifestation.index')->with('success', 'تم حذف الطلب بنجاح.');
    }




public function toggleStatut(Request $request, $id)
{
    $request->validate([
        'id_statut' => 'nullable|integer|in:1,2,3,4,5',
        'statut_id' => 'nullable|integer|in:1,2,3,4,5',
        'statut2_id'=> 'nullable|integer|in:1,2,3,4,5',
        'commentaire_motif'  => 'nullable|string',
        'commentaire_motif2' => 'nullable|string',
        'commentaire_motif3' => 'nullable|string',
        'commentaire_acceptation' => 'nullable|string',
        'commentaire_avis'         => 'nullable|string',
        'file' => 'nullable|file',
    ]);

    $demande = DemandeManifestation::with(['utilisateur', 'evenementSportifs', 'organismeNonSportif'])->findOrFail($id);

    // capture original values for comparison
    $orig_id_type   = $demande->getOriginal('id_type');
    $orig_statut_id = $demande->getOriginal('statut_id');
    $orig_statut2_id= $demande->getOriginal('statut2_id');

    // Apply incoming fields (same behaviour as before)
    if ($request->has('id_statut')) {
        $demande->id_type = $request->input('id_statut');
        if ($request->filled('commentaire_motif')) {
            $demande->commentaire_motif = $request->input('commentaire_motif');
        }
    }

    if ($request->has('statut_id')) {
        $demande->statut_id = $request->input('statut_id');
        if ($request->filled('commentaire_motif2')) {
            $demande->commentaire_motif2 = $request->input('commentaire_motif2');
        }
    }

    if ($request->has('statut2_id')) {
        $demande->statut2_id = $request->input('statut2_id');
        if ($request->filled('commentaire_motif3')) {
            $demande->commentaire_motif3 = $request->input('commentaire_motif3');
        }
    }

    if ($request->hasFile('file')) {
        $file = $request->file('file');
        $filename = time() . '_' . $file->getClientOriginalName();
        $path = $file->storeAs('public/files', $filename);
        $demande->file = $filename;
    }

    // determine which field changed (priority: id_type, statut_id, statut2_id)
    $changedField = null;
    $newValue = null;

    if ($request->has('id_statut') && ((int)$orig_id_type !== (int)$demande->id_type)) {
        $changedField = 'id_type';
        $newValue = (int)$demande->id_type;
    } elseif ($request->has('statut_id') && ((int)$orig_statut_id !== (int)$demande->statut_id)) {
        $changedField = 'statut_id';
        $newValue = (int)$demande->statut_id;
    } elseif ($request->has('statut2_id') && ((int)$orig_statut2_id !== (int)$demande->statut2_id)) {
        $changedField = 'statut2_id';
        $newValue = (int)$demande->statut2_id;
    }

    // if nothing changed, just save and return
    if (!$changedField) {
        $demande->save();
        return redirect()->back()->with('info', 'لم يتم تغيير أي وضعية.');
    }

    DB::beginTransaction();
    try {
        $demande->save();

        // create history for the changed field
        ImmobilierStatutHistory::create([
            'demande_manifestation_id' => $demande->id_demande,
            'from_statut_id' => ($changedField === 'id_type' ? $orig_id_type : ($changedField === 'statut_id' ? $orig_statut_id : $orig_statut2_id)) ?: null,
            'to_statut_id'   => $newValue ?: null,
            'changed_by'     => Auth::id() ?? null,
            'commentaire'    => trim($request->input('commentaire_motif') ?? $request->input('commentaire_motif2') ?? $request->input('commentaire_motif3') ?? ''),
        ]);

        DB::commit();
    } catch (\Exception $e) {
        DB::rollBack();
        Log::error("toggleStatut error for DemandeManifestation #{$id}: " . $e->getMessage(), [
            'trace' => $e->getTraceAsString(), 'request' => $request->all()
        ]);
        return back()->withErrors(['error' => 'خطأ أثناء تغيير الحالة: ' . $e->getMessage()]);
    }

    // --- prepare dossier, message and notification params ---
    $idPad = str_pad($demande->id_demande, 3, '0', STR_PAD_LEFT);
    $cin = $demande->utilisateur->num_cin ?? $demande->utilisateur->cin ?? '';
    $cinDigits = preg_replace('/\D+/', '', $cin);
    $cinPad = $cinDigits ? str_pad($cinDigits, 9, '0', STR_PAD_LEFT) : '000000000';
    $year = $demande->created_at ? $demande->created_at->format('Y') : now()->format('Y');
    $num_dossier = "{$idPad}-{$cinPad}-{$year}";
    $nomUtilisateur = $demande->utilisateur->nom_prenom ?? 'المستخدم';
    $date = now()->format('Y-m-d');

    // choose the resolved status number we just stored
    $resolved = $newValue ?? 0;
    $statusMap = [1 => 'accepted', 3 => 'rejected', 4 => 'incomplete', 5 => 'deferred'];
    $type = $statusMap[$resolved] ?? 'status_changed';

    // gather comment from model (prefer stored model fields)
    $messageComment = trim(
        ($demande->commentaire_motif ?? '') . ' ' .
        ($demande->commentaire_motif2 ?? '') . ' ' .
        ($demande->commentaire_motif3 ?? '')
    );
    $messageComment = $messageComment !== '' ? $messageComment : null;

    // prepare the HTML message status (same wording as update)
    switch ($resolved) {
        case 1:
            $messageStatus = "مرحباً <strong>{$nomUtilisateur}</strong>،<br>رقم الملف: <strong>{$num_dossier}</strong><br>تم قبول طلبكم بتاريخ: <strong>{$date}</strong>.<br>نهنئكم على هذا القبول.";
            break;
        case 3:
            $reason = e($messageComment ?? ($request->input('commentaire_motif') ?? 'سبب غير محدد'));
            $messageStatus = "مرحباً <strong>{$nomUtilisateur}</strong>،<br>رقم الملف: <strong>{$num_dossier}</strong><br>تم رفض طلبكم.<br>التعليق: <b>{$reason}</b>.";
            break;
        case 4:
            $missing = e($request->input('commentaire_acceptation') ?? ($messageComment ?? 'يرجى استكمال الوثائق المطلوبة'));
            $messageStatus = "مرحباً <strong>{$nomUtilisateur}</strong>،<br>رقم الملف: <strong>{$num_dossier}</strong><br>طلبكم <strong>غير مكتمل</strong>.<br>التوضيح: <b>{$missing}</b>.";
            break;
        case 5:
            $note = e($request->input('commentaire_avis') ?? ($messageComment ?? 'لا يوجد ملاحظة'));
            $messageStatus = "مرحباً <strong>{$nomUtilisateur}</strong>،<br>رقم الملف: <strong>{$num_dossier}</strong><br>تم إرجاء طلبكم.<br>الملاحظة: <b>{$note}</b>.";
            break;
        default:
            $messageStatus = "مرحباً <strong>{$nomUtilisateur}</strong>،<br>رقم الملف: <strong>{$num_dossier}</strong><br>تم تحديث حالة ملفكم.";
    }

    // prepare params for DB notification
    $params = [];
    if ($resolved === 3) $params['reason'] = $messageComment ?? ($request->input('commentaire_motif') ?? '');
    if ($resolved === 4) $params['missing'] = $request->input('commentaire_acceptation') ?? ($messageComment ?? '');
    if ($resolved === 5) $params['note'] = $request->input('commentaire_avis') ?? ($messageComment ?? '');
    if ($resolved === 1) $params['date'] = $date;
    $params['reference'] = $num_dossier;
    if (!empty($messageComment)) $params['comment'] = $messageComment;

    // Send email (best-effort)
    if ($demande->utilisateur?->email) {
        try {
            Mail::to($demande->utilisateur->email)
                ->send(new \App\Mail\StatutDemandeMail($demande->utilisateur, $num_dossier, $messageStatus, $type, $messageComment));
        } catch (\Exception $ex) {
            Log::error("toggleStatut mail failed for DemandeManifestation {$demande->id_demande}: " . $ex->getMessage());
        }
    } else {
        Log::warning("toggleStatut: no email on utilisateur for DemandeManifestation {$demande->id_demande}");
    }

    // DB notification (best-effort)
    try {
        if ($demande->utilisateur) {
            $translationKey = "site.notifications.manifestation.{$type}";
            $demande->utilisateur->notify(new \App\Notifications\StatutDemandeNotification(
                $demande, $type, $params, $demande->id_demande, $translationKey
            ));
        }
    } catch (\Exception $ex) {
        Log::error("toggleStatut notification failed for DemandeManifestation {$demande->id_demande}: " . $ex->getMessage());
    }

    $statut_ar = $resolved == 1 ? 'قبول' : ($resolved == 3 ? 'رفض' : 'تم التغيير');
    return redirect()->back()->with('success', "تم تغيير الحالة إلى $statut_ar");
}

    
public function getDelegations_manifestation($gouvernoratId)    {        $delegations = Delegation::where('gouver_id', $gouvernoratId)->get();        return response()->json($delegations);    }



/*public function showDecision($id)
{
    $demande = DemandeManifestation::findOrFail($id);

    // Vérifier si le fichier existe
    if (empty($demande->file)) {
        abort(404, 'Aucun fichier disponible');
    }

    // Détection MIME depuis le BLOB
    $finfo = new \finfo(FILEINFO_MIME_TYPE);
    $mimeType = $finfo->buffer($demande->file);

    return response($demande->file, 200)
        ->header('Content-Type', $mimeType)
        ->header('Content-Disposition', 'inline');
}
*/

//last showDecision
/*public function showDecision($id)
{
    $demande = DemandeManifestation::findOrFail($id);

    // Vérifier si le fichier existe (maintenant c'est un chemin)
    if (empty($demande->file) || !file_exists($demande->file)) {
        abort(404, 'Aucun fichier disponible');
    }

    $path = $demande->file;
    
    return response()->file($path, [
        'Content-Type' => mime_content_type($path),
        'Content-Disposition' => 'inline; filename="' . basename($path) . '"',
        'Cache-Control' => 'no-store, no-cache, must-revalidate, max-age=0',
        'Pragma' => 'no-cache',
        'Expires' => '0',
    ]);
}*/









/*public function showfile($id)
{
    $att = Attachement::findOrFail($id);

    if (empty($att->file)) {
        abort(404, 'Fichier introuvable');
    }

    // Détecte automatiquement le type MIME à partir du contenu binaire
    $finfo = new \finfo(FILEINFO_MIME_TYPE);
    $mimeType = $finfo->buffer($att->file);

    return response($att->file, 200)
        ->header('Content-Type', $mimeType)
        ->header('Content-Disposition', 'inline'); // Affichage direct dans le navigateur
}*/

public function showfile($id)
{
    $att = Attachement::findOrFail($id);

    if (empty($att->file)) {
        abort(404, 'Fichier introuvable');
    }

    // Utiliser le chemin tel qu'il est enregistré
    $path = $att->file;
    
    // Si le chemin n'est pas absolu (ne commence pas par /)
    if (strpos($path, '/') !== 0) {
        // C'est probablement un chemin relatif, on le convertit en chemin absolu
        $uploadDir = '/home/preprov/www/mjs/shared_uploads/images';
        $path = $uploadDir . '/' . basename($path);
    }
    
    // Vérifier si le fichier existe
    if (!file_exists($path)) {
        \Log::error("Fichier d'attachment introuvable", [
            'attachment_id' => $id,
            'stored_path' => $att->file,
            'tried_path' => $path
        ]);
        abort(404, 'Fichier introuvable: ' . basename($att->file));
    }

    return response()->file($path, [
        'Content-Type' => mime_content_type($path),
        'Content-Disposition' => 'inline; filename="' . basename($path) . '"',
        'Cache-Control' => 'no-store, no-cache, must-revalidate, max-age=0',
        'Pragma' => 'no-cache',
        'Expires' => '0',
    ]);
}

public function downloadDemandeFile($id)
{
    $demande = DemandeManifestation::findOrFail($id);
    
    if (empty($demande->file) || !file_exists($demande->file)) {
        abort(404, 'Fichier introuvable');
    }
    
    $path = $demande->file;
    $fileName = basename($path); // Extraire le nom du fichier depuis le chemin
    
    return response()->download($path, $fileName, [
        'Content-Type' => mime_content_type($path),
    ]);
}


// Fonction pour télécharger le fichier principal d'une demande
/*public function downloadDemandeFile($id)
{
    $demande = DemandeManifestation::findOrFail($id);
    
    if (empty($demande->file) || !file_exists($demande->file)) {
        abort(404, 'Fichier introuvable');
    }
    
    $path = $demande->file;
    
    return response()->download($path, $demande->file_name ?? basename($path), [
        'Content-Type' => mime_content_type($path),
    ]);
}*/

// Fonction pour télécharger les attachements
public function downloadAttachment($id)
{
    $att = Attachement::findOrFail($id);
    
    if (empty($att->file) || !file_exists($att->file)) {
        abort(404, 'Fichier introuvable');
    }
    
    $path = $att->file;
    
    return response()->download($path, basename($path), [
        'Content-Type' => mime_content_type($path),
    ]);
}

}