<?php
namespace App\Http\Controllers\Investissement;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;
use App\Models\DocumentsFormulaire;
use App\Models\Utilisateur;
use App\Models\TypeFormulaire;
use App\Models\Identite_personnel;
use App\Models\TpStatut;

use App\Mail\DocumentStatusMail;
use App\Notifications\DocumentStatusNotification;
use Illuminate\Support\Facades\Notification;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use Barryvdh\DomPDF\Facade\Pdf;
use Mpdf\Mpdf;
use App\Models\ImmobilierStatutHistory;
use Illuminate\Support\Facades\Auth;
use Illuminate\Http\Response;
use Illuminate\Support\Str;


class ExonarationController extends Controller
{
    public function index(Request $request)
{
    $query = DocumentsFormulaire::with(['utilisateur', 'statut']);

    if ($request->filled('search_option') && $request->search_option === 'cahier' && $request->filled('id_type_formulaire')) {
        $query->where('id_type_formulaire', $request->id_type_formulaire);
    }

    if ($request->filled('search_option') && $request->search_option === 'date' && $request->filled('date')) {
        $query->whereDate('created_at', $request->date);
    }

    if ($request->filled('search_option') && $request->search_option === 'identite' && $request->filled('type_identite')) {
        $type = $request->type_identite;

        if ($type == 1 && $request->filled('cin')) {
            $query->whereHas('utilisateur', fn($q) => $q->where('cin', $request->cin));
        }
        if ($type == 2 && $request->filled('passport')) {
            $query->whereHas('utilisateur', fn($q) => $q->where('passport', $request->passport));
        }
        if ($type == 3 && $request->filled('num_fiscal')) {
            $query->whereHas('utilisateur', fn($q) => $q->where('num_fiscal', $request->num_fiscal));
        }
    }

    $documents = $query->latest()->paginate(10)->withQueryString();

    $typesFormulaires = TypeFormulaire::whereIn('id_type', [1,3,4,21,5])->get();

    return view('backoffice.investissement.gestion_exoneration.index', compact('documents', 'typesFormulaires'));
}


private function filteredQuery(Request $request)
{
    $query = DocumentsFormulaire::with(['utilisateur', 'statut']);

    if ($request->search_option === 'cahier' && $request->filled('id_type_formulaire')) {
        $query->where('id_type_formulaire', $request->id_type_formulaire);
    }

    if ($request->search_option === 'date' && $request->filled('date')) {
        $query->whereDate('created_at', $request->date);
    }

    if ($request->search_option === 'identite' && $request->filled('type_identite')) {
        if ($request->type_identite == 1 && $request->filled('cin')) {
            $query->whereHas('utilisateur', fn($q) => $q->where('cin', $request->cin));
        }

        if ($request->type_identite == 2 && $request->filled('passport')) {
            $query->whereHas('utilisateur', fn($q) => $q->where('passport', $request->passport));
        }

        if ($request->type_identite == 3 && $request->filled('num_fiscal')) {
            $query->whereHas('utilisateur', fn($q) => $q->where('num_fiscal', $request->num_fiscal));
        }
    }

    return $query;
}


public function exportPdf(Request $request)
{
    $documents = $this->filteredQuery($request)
        ->latest()
        ->get();

    $html = view(
        'backoffice.investissement.gestion_exoneration.exports.exoneration_pdf',
        compact('documents')
    )->render();

    $mpdf = new Mpdf([
        'mode' => 'utf-8',
        'format' => 'A4',
        'default_font' => 'dejavusans',
        'directionality' => 'rtl',
    ]);

    $mpdf->WriteHTML($html);

    return response($mpdf->Output('exoneration.pdf', 'S'))
        ->header('Content-Type', 'application/pdf');
}

public function exportExcel(Request $request)
{
    $documents = $this->filteredQuery($request)
        ->latest()
        ->get();

    $spreadsheet = new Spreadsheet();
    $sheet = $spreadsheet->getActiveSheet();
    $sheet->setRightToLeft(true);

    // Header
    $sheet->fromArray([
        'المعرف',
        'الاسم و اللقب',
        'اسم المشروع',
        'مكان المشروع',
        'وضعية المشروع',
    ], null, 'A1');

    $sheet->getStyle('A1:E1')->getFont()->setBold(true);

    // Data
    $row = 2;
    foreach ($documents as $doc) {
        $sheet->setCellValue('A'.$row, $doc->id_doc);
        $sheet->setCellValue('B'.$row, $doc->utilisateur?->nom_prenom ?? '---');
        $sheet->setCellValue('C'.$row, $doc->nom_projet);
        $sheet->setCellValue('D'.$row, $doc->adresse_projet);
        $sheet->setCellValue('E'.$row, $doc->statut?->statut_ar ?? 'في الانتظار');
        $row++;
    }

    foreach (range('A', 'E') as $col) {
        $sheet->getColumnDimension($col)->setAutoSize(true);
    }

    $writer = new Xlsx($spreadsheet);

    return response()->streamDownload(
        fn () => $writer->save('php://output'),
        'exoneration.xlsx'
    );
}


    public function store(Request $request)
    {
        DB::beginTransaction();
        try {
            $request->validate([
                // Utilisateur
                'nom_prenom'       => 'required|string|max:255',
                'id_type_identite' => 'required|integer|exists:tp_type_identite_pers,id_type',
                'cin'              => 'nullable|string|max:20',
                'passport'         => 'nullable|string|max:50',
                'num_fiscal'       => 'nullable|string|max:50',
                'tel'              => 'nullable|string|max:20',
                'adresse'          => 'nullable|string|max:255',
                'email'            => 'required|email|unique:utilisateur,email',
                'date_naissance'   => 'nullable|date',
                'lieu_naissance'   => 'nullable|string|max:100',                

                // Documents
                'nom_projet'       => 'required|string|max:255',
                'adresse_projet'   => 'required|string|max:255',
                'copie_cahier_des_charges'           => 'nullable|file|mimes:pdf,jpg,jpeg,png',
                'copie_originale'                    => 'nullable|file|mimes:pdf,jpg,jpeg,png',
                'facture'                            => 'nullable|file|mimes:pdf,jpg,jpeg,png',
                'contrat'                            => 'nullable|file|mimes:pdf,jpg,jpeg,png',
                'fichier_arrivage'                   => 'nullable|file|mimes:pdf,jpg,jpeg,png',
                'fichier_colisage'                   => 'nullable|file|mimes:pdf,jpg,jpeg,png',
                'copie_registre_fiscal'              => 'nullable|file|mimes:pdf,jpg,jpeg,png',
                'copie_registre_national_institutions' => 'nullable|file|mimes:pdf,jpg,jpeg,png',
                'copie_certificat_propriete'         => 'nullable|file|mimes:pdf,jpg,jpeg,png',
                'engagement'                         => 'nullable|file|mimes:pdf,jpg,jpeg,png',
                'exemple_projet'                     => 'nullable|file|mimes:pdf,jpg,jpeg,png',
            ]);

            $utilisateur = Utilisateur::create([
                'nom_prenom'          => $request->nom_prenom,
                'id_type_identite'    => $request->id_type_identite,
                'cin'                 => $request->cin,
                'passport'            => $request->passport,
                'num_fiscal'          => $request->num_fiscal,
                'adresse'             => $request->adresse,
                'tel'                 => $request->tel,
                'email'               => $request->email,
                'date_naissance'      => $request->date_naissance,
                'lieu_naissance'      => $request->lieu_naissance,
            ]);

            $files = [
                'copie_cahier_des_charges',
                'copie_originale',
                'facture',
                'contrat',
                'fichier_arrivage',
                'fichier_colisage',
                'copie_registre_fiscal',
                'copie_registre_national_institutions',
                'copie_certificat_propriete',
                'engagement',
                'exemple_projet'
            ];

            $fileData = [];
            foreach ($files as $fileField) {
                if ($request->hasFile($fileField)) {
                    $path = $request->file($fileField)->store('documents/exoneration', 'public');
                    $fileData[$fileField] = $path;
                }
            }

            DocumentsFormulaire::create(array_merge($fileData, [
                'id_user'      => $utilisateur->id,
                'id_formulaire'=> $request->id_formulaire,
                'nom_projet'   => $request->nom_projet,
                'adresse_projet'=> $request->adresse_projet,
                'id_statut'    => 2,
            ]));

           DB::commit();
            return redirect()->route('admin.exoneration.index')
                            ->with('success', 'تم تسجيل المشروع بنجاح');

            } catch (\Exception $e) {
                DB::rollBack();
                return back()->withErrors(['error' => 'حدث خطأ أثناء التسجيل: '.$e->getMessage()])
                            ->withInput();
        }
    }

    public function edit($id)
    {
        $document = DocumentsFormulaire::with('utilisateur')->findOrFail($id);
        $typesIdentite = Identite_personnel::all();
        $formulaire = TypeFormulaire::find($document->id_formulaire);
        $statuts = TpStatut::orderBy('id_statut')->get();
        return view('backoffice.investissement.gestion_exoneration.edit', compact(
            'document','typesIdentite','formulaire', 'statuts'
        ));
    }




    public function exonerationPdf($id)
{
    $document = DocumentsFormulaire::with([
        'immobilier.utilisateur'
    ])->findOrFail($id);

     $lang = 'ar';
    $direction = 'rtl';

    $html = view(
        'backoffice.investissement.gestion_exoneration.pdf',
        compact('document')
    )->render();

    $mpdf = new Mpdf([
        'mode' => 'utf-8',
        'format' => 'A4',
        'default_font' => 'dejavusans',
        'directionality' => 'rtl',
        'autoScriptToLang' => true,
        'autoLangToFont' => true,
    ]);

    $mpdf->SetFooter('{PAGENO} / {nbpg}');
    $mpdf->WriteHTML($html);

    return $mpdf->Output(
        'exoneration_'.$id.'.pdf',
        'I' 
    );
}


public function update(Request $request, $id)
{
    DB::beginTransaction();
    try {
        $document = DocumentsFormulaire::with('utilisateur')->findOrFail($id);
        $utilisateur = $document->utilisateur;
        $request->validate([
            // Utilisateur
            'nom_prenom'       => 'required|string|max:255',
            'id_type_identite' => 'required|integer|exists:tp_type_identite_pers,id_type',
            'cin'              => 'nullable|string|max:20',
            'passport'         => 'nullable|string|max:50',
            'num_fiscal'       => 'nullable|string|max:50',
            'tel'              => 'nullable|string|max:20',
            'adresse'          => 'nullable|string|max:255',
            'email'            => 'required|email|unique:utilisateur,email,' . $utilisateur->id,
            'date_naissance'   => 'nullable|date',
            'lieu_naissance'   => 'nullable|string|max:100',

            // Documents
            'nom_projet'       => 'required|string|max:255',
            'adresse_projet'   => 'required|string|max:255',
            'id_statut'        => 'required|integer',
            'commentaire_motif'        => 'nullable|string',
            'commentaire_acceptation'  => 'nullable|string',
            'commentaire_avis'         => 'nullable|string',
            'copie_cahier_des_charges'           => 'nullable|file|mimes:pdf,jpg,jpeg,png',
            'copie_originale'                    => 'nullable|file|mimes:pdf,jpg,jpeg,png',
            'facture'                            => 'nullable|file|mimes:pdf,jpg,jpeg,png',
            'contrat'                            => 'nullable|file|mimes:pdf,jpg,jpeg,png',
            'fichier_arrivage'                   => 'nullable|file|mimes:pdf,jpg,jpeg,png',
            'fichier_colisage'                   => 'nullable|file|mimes:pdf,jpg,jpeg,png',
            'copie_registre_fiscal'              => 'nullable|file|mimes:pdf,jpg,jpeg,png',
            'copie_registre_national_institutions' => 'nullable|file|mimes:pdf,jpg,jpeg,png',
            'copie_certificat_propriete'         => 'nullable|file|mimes:pdf,jpg,jpeg,png',
            'engagement'                         => 'nullable|file|mimes:pdf,jpg,jpeg,png',
            'exemple_projet'                     => 'nullable|file|mimes:pdf,jpg,jpeg,png',
        ]);

        $utilisateur->update([
            'nom_prenom'       => $request->nom_prenom,
            'id_type_identite' => $request->id_type_identite,
            'cin'              => $request->cin,
            'passport'         => $request->passport,
            'num_fiscal'       => $request->num_fiscal,
            'adresse'          => $request->adresse,
            'tel'              => $request->tel,
            'email'            => $request->email,
            'date_naissance'   => $request->date_naissance,
            'lieu_naissance'   => $request->lieu_naissance,
        ]);

        /*$files = [
            'copie_cahier_des_charges',
            'copie_originale',
            'facture',
            'contrat',
            'fichier_arrivage',
            'fichier_colisage',
            'copie_registre_fiscal',
            'copie_registre_national_institutions',
            'copie_certificat_propriete',
            'engagement',
            'exemple_projet'
        ];

        foreach ($files as $fileField) {
            if ($request->hasFile($fileField)) {
                $file = $request->file($fileField);
                // Lire le contenu binaire du fichier
                $document->$fileField = file_get_contents($file->getRealPath());
            }
        }*/
		
		$files = [
    'copie_cahier_des_charges',
    'copie_originale',
    'facture',
    'contrat',
    'fichier_arrivage',
    'fichier_colisage',
    'copie_registre_fiscal',
    'copie_registre_national_institutions',
    'copie_certificat_propriete',
    'engagement',
    'exemple_projet'
];

$uploadDir = '/home/preprov/www/mjs/shared_uploads/images';

if (!is_dir($uploadDir) || !is_writable($uploadDir)) {
    abort(500, 'Dossier upload non accessible');
}

foreach ($files as $field) {
    if ($request->hasFile($field)) {

        $file = $request->file($field);

        // 🧹 supprimer ancien fichier
        if (!empty($document->$field) && file_exists($document->$field)) {
            @unlink($document->$field);
        }

        // 🔐 nom sécurisé
        $original = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
        $ext = $file->getClientOriginalExtension();
        $safe = Str::slug(mb_substr($original, 0, 40)) ?: 'document';

        $filename = $safe . '-' . time() . '-' . Str::random(6) . '.' . $ext;
        $targetPath = $uploadDir . DIRECTORY_SEPARATOR . $filename;

        // 🚀 upload réel
        $file->move($uploadDir, $filename);

        $realPath = realpath($targetPath);
        if ($realPath === false) {
            abort(500, 'Erreur lors de la sauvegarde du fichier');
        }

        // 💾 stocker PATH STRING
        $document->$field = $realPath;
    }
}

$document->save();

        if ($request->id_statut != 2 && $utilisateur && !empty($utilisateur->email)) {
            $messageStatus = '';

            switch ($request->id_statut) {
                case 1:
                    $messageStatus = " تم <b>قبول</b> طلبكم بنجاح. نهنئكم على هذا القبول.";
                    break;

                case 3:
                    $messageStatus = " تم <b>رفض</b> طلبكم.<br>السبب: <b>" . e($request->commentaire_motif) . "</b>";
                    break;

                case 4:
                    $messageStatus = " طلبكم <b>غير مكتمل</b>.<br>يرجى استكمال النواقص التالية: <b>" . e($request->commentaire_acceptation) . "</b>";
                    break;

                case 5:
                    $note = trim((string) $request->commentaire_avis);
                    if ($note !== '') {
                        $messageStatus = "تم <b>إرجاء طلبكم لإبداء الرأي</b>.<br>الملاحظة: <b>" . e($note) . "</b>";
                    } else {
                        $messageStatus = "تم <b>إرجاء طلبكم لإبداء الرأي</b>.";
                    }
                    break;
            }

            $oldStatus = (int) ($document->id_statut ?? 0);
            $newStatus = (int) $request->input('id_statut', $oldStatus);

            // choisir le commentaire selon le type de statut (garder noms existants)
            $commentRefus   = $request->input('commentaire_motif');
            $commentAccept  = $request->input('commentaire_acceptation');
            $commentAvis    = $request->input('commentaire_avis');

            $comment = null;
            if ($request->filled('commentaire_refus')) {
                $comment = trim($request->input('commentaire_refus'));
            } elseif ($request->filled('commentaire_status')) {
                $comment = trim($request->input('commentaire_status'));
            } elseif ($request->filled('commentaire_motif')) {
                $comment = trim($request->input('commentaire_motif'));
            }
            if ($comment === '') $comment = null;

            $document->fill([
                'nom_projet'              => $request->nom_projet,
                'adresse_projet'          => $request->adresse_projet,
                'id_statut'               => $request->id_statut,
                'commentaire_motif'       => $request->commentaire_motif,
                'commentaire_acceptation' => $request->commentaire_acceptation,
                'commentaire_avis'        => $request->commentaire_avis,
                'etat_formulaire'         => $request->id_statut != 2 ? 'reponse' : 'remplir',
            ])->save();

            // si le statut a changé, créer l'historique AVANT (ou APRÈS) la mise à jour
            if ($oldStatus !== $newStatus) {
                ImmobilierStatutHistory::create([
                    'documents_formulaire_id' => $document->id_doc,
                    'from_statut_id'          => $oldStatus ?: null,
                    'to_statut_id'            => $newStatus,
                    'changed_by'              => Auth::id() ?? null,
                    'commentaire'             => $comment,
                ]);
            }

            // ensuite tu peux mettre à jour le document (ou faire la création d'historique après en utilisant getOriginal())
            $document->update([
                'nom_projet'    => $request->nom_projet,
                'adresse_projet'=> $request->adresse_projet,
                'id_statut'     => $request->id_statut,
                'commentaire_motif'      => $request->commentaire_motif,
                'commentaire_acceptation'=> $request->commentaire_acceptation,
                'commentaire_avis'       => $request->commentaire_avis,
                'etat_formulaire' => $request->id_statut != 2 ? 'reponse' : 'remplir',
            ]);

            // --- NOUVEAU : récupérer le dernier commentaire depuis la table d'historique ---
            // Si un commentaire d'historique existe pour ce document, on l'utilise pour le mail.
            $latestHistoryComment = ImmobilierStatutHistory::where('documents_formulaire_id', $document->id_doc)
                ->orderBy('created_at', 'desc')
                ->value('commentaire');

            // Si on a un commentaire en historique, reconstruire $messageStatus en le privilégiant.
            if (!empty($latestHistoryComment)) {
                switch ($request->id_statut) {
                    case 1:
                        $messageStatus = " تم <b>قبول</b> طلبكم بنجاح. نهنئكم على هذا القبول.";
                        break;

                    case 3:
                        $messageStatus = " تم <b>رفض</b> طلبكم.<br>السبب: <b>" . e($latestHistoryComment) . "</b>";
                        break;

                    case 4:
                        $messageStatus = " طلبكم <b>غير مكتمل</b>.<br>يرجى استكمال النواقص التالية: <b>" . e($latestHistoryComment) . "</b>";
                        break;

                    case 5:
                        $note = trim((string) $latestHistoryComment);
                        if ($note !== '') {
                            $messageStatus = "تم <b>إرجاء طلبكم لإبداء الرأي</b>.<br>الملاحظة: <b>" . e($note) . "</b>";
                        } else {
                            $messageStatus = "تم <b>إرجاء طلبكم لإبداء الرأي</b>.";
                        }
                        break;
                }
            }

            // normalization map for statuses
            $statusMap = [
                1 => 'accepted',
                3 => 'rejected',
                4 => 'incomplete',
                5 => 'deferred',
            ];

            // choose normalized type
            $normType = $statusMap[$newStatus] ?? 'status_changed';

            // prepare translation key and params
            $translationKey = "site.notifications.documents_formulaire.{$normType}";
            $translationParams = [];

            // fill params depending on status (keep existing logic — do not change)
            if ($normType === 'rejected') {
                $translationParams['reason'] = $commentRefus ?? ($comment ?? '');
            } elseif ($normType === 'incomplete') {
                $translationParams['missing'] = $commentAccept ?? ($comment ?? '');
            } elseif ($normType === 'deferred') {
                $translationParams['note'] = $commentAvis ?? ($comment ?? '');
            }

            $numeroDossier = $document->numero_dossier
                ?? $document->num_dossier
                ?? '—';

            if (empty($messageStatus)) {
                $messageStatus = 'لا توجد ملاحظات إضافية.';
            }

            // Send the mail — $messageStatus now preferentially contains the historique comment if present
            Mail::to($utilisateur->email)->send(
                 new DocumentStatusMail(
                    $utilisateur,
                    $normType,
                    $messageStatus,
                    $numeroDossier
                )
            );

            // notification DB (keeps using translation params built above — unchanged)
            $utilisateur->notify(new \App\Notifications\DocumentStatusNotification(
                $normType,
                $translationParams,
                $document->num_dossier
            ));
        }

        DB::commit();

        return redirect()->route('admin.exoneration.index')
                        ->with('success', 'تم تحديث المشروع بنجاح ');

    } catch (\Exception $e) {
        DB::rollBack();
        return back()->withErrors(['error' => 'حدث خطأ أثناء التحديث: ' . $e->getMessage()])
                    ->withInput();
    }
}

/*
public function showFile2($id, $field)
{
    $document = DocumentsFormulaire::findOrFail($id);

    $allowedFields = [
        'copie_cahier_des_charges',
        'copie_originale',
        'facture',
        'contrat',
        'fichier_arrivage',
        'fichier_colisage',
        'copie_registre_fiscal',
        'copie_registre_national_institutions',
        'copie_certificat_propriete',
        'engagement',
        'exemple_projet'
    ];

    if (!in_array($field, $allowedFields)) {
        abort(404);
    }

    $fileContent = $document->$field;

    if (empty($fileContent)) {
        abort(404);
    }

    // Détection MIME (PDF, JPG, PNG…)
    $finfo = finfo_open(FILEINFO_MIME_TYPE);
    $mimeType = finfo_buffer($finfo, $fileContent);
    finfo_close($finfo);

    return response($fileContent, 200, [
        'Content-Type'        => $mimeType,
        'Content-Disposition'=> 'inline',
        'Content-Length'     => strlen($fileContent),
    ]);
}
*/

public function showFile2($id, $field)
{
    $document = DocumentsFormulaire::findOrFail($id);

    $allowed = [
        'copie_cahier_des_charges',
        'copie_originale',
        'facture',
        'contrat',
        'fichier_arrivage',
        'fichier_colisage',
        'copie_registre_fiscal',
        'copie_registre_national_institutions',
        'copie_certificat_propriete',
        'engagement',
        'exemple_projet'
    ];

    if (!in_array($field, $allowed, true)) {
        abort(404);
    }

    $path = $document->$field;

    if (empty($path) || !file_exists($path)) {
        abort(404, 'Fichier introuvable');
    }

    return response()->file($path, [
        'Content-Type'        => mime_content_type($path),
        'Content-Disposition'=> 'inline; filename="'.basename($path).'"',
        'Cache-Control'      => 'no-store, no-cache, must-revalidate, max-age=0',
        'Pragma'             => 'no-cache',
        'Expires'            => '0',
    ]);
}


/*public function destroy($id)
    {
        DB::beginTransaction();
        try {
            $document = DocumentsFormulaire::findOrFail($id);

            $files = [
                'copie_cahier_des_charges','copie_originale','facture','contrat','fichier_arrivage','fichier_colisage',
                'copie_registre_fiscal','copie_registre_national_institutions','copie_certificat_propriete','engagement','exemple_projet'
            ];

            foreach ($files as $file) {
                if (!empty($document->$file) && \Storage::disk('public')->exists($document->$file)) {
                    \Storage::disk('public')->delete($document->$file);
                }
            }

            $document->utilisateur()->delete();

            $document->delete();

            DB::commit();
            return redirect()->route('admin.exoneration.index')
                            ->with('success', 'تم حذف المشروع بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->withErrors(['error' => 'حدث خطأ أثناء الحذف: '.$e->getMessage()]);
        }
    }*/
	
public function destroy($id)
{
    try {
        $document = DocumentsFormulaire::findOrFail($id);

        // sauvegarder les chemins AVANT delete
        $files = [
            $document->copie_cahier_des_charges,
            $document->copie_originale,
            $document->facture,
            $document->contrat,
            $document->fichier_arrivage,
            $document->fichier_colisage,
            $document->copie_registre_fiscal,
            $document->copie_registre_national_institutions,
            $document->copie_certificat_propriete,
            $document->engagement,
            $document->exemple_projet,
        ];

        DB::beginTransaction();

        // ✅ supprimer le document EN PREMIER
        $document->delete();

        DB::commit();

        // 🧹 supprimer les fichiers APRÈS le commit
        foreach ($files as $path) {
            if ($path && Storage::disk('shared_uploads')->exists($path)) {
                Storage::disk('shared_uploads')->delete($path);
            }
        }

        
			 return redirect()->route('admin.exoneration.index')
                        ->with('success', 'تم حذف المشروع بنجاح ');

    } catch (\Throwable $e) {

        DB::rollBack();

        \Log::error('Exoneration destroy error', [
            'id_doc' => $id,
            'error'  => $e->getMessage()
        ]);

        return back()->withErrors('حدث خطأ أثناء الحذف');
    }
}




	


	
	

  /*  public function update(Request $request, $id)
    {
        DB::beginTransaction();
        try {
            $document = DocumentsFormulaire::with('utilisateur')->findOrFail($id);
            $utilisateur = $document->utilisateur;

            $request->validate([
                // Utilisateur
                'nom_prenom'       => 'required|string|max:255',
                'id_type_identite' => 'required|integer|exists:tp_type_identite_pers,id_type',
                'cin'              => 'nullable|string|max:20',
                'passport'         => 'nullable|string|max:50',
                'num_fiscal'       => 'nullable|string|max:50',
                'tel'              => 'nullable|string|max:20',
                'adresse'          => 'nullable|string|max:255',
                'email'            => 'required|email|unique:utilisateur,email,' . $utilisateur->id,
                'date_naissance'   => 'nullable|date',
                'lieu_naissance'   => 'nullable|string|max:100',

                // Documents
                'nom_projet'       => 'required|string|max:255',
                'adresse_projet'   => 'required|string|max:255',
                'id_statut'        => 'required|integer',
                'commentaire_motif'        => 'nullable|string',
                'commentaire_acceptation'  => 'nullable|string',
                'commentaire_avis'         => 'nullable|string',
                'copie_cahier_des_charges'           => 'nullable|file|mimes:pdf,jpg,jpeg,png',
                'copie_originale'                    => 'nullable|file|mimes:pdf,jpg,jpeg,png',
                'facture'                            => 'nullable|file|mimes:pdf,jpg,jpeg,png',
                'contrat'                            => 'nullable|file|mimes:pdf,jpg,jpeg,png',
                'fichier_arrivage'                   => 'nullable|file|mimes:pdf,jpg,jpeg,png',
                'fichier_colisage'                   => 'nullable|file|mimes:pdf,jpg,jpeg,png',
                'copie_registre_fiscal'              => 'nullable|file|mimes:pdf,jpg,jpeg,png',
                'copie_registre_national_institutions' => 'nullable|file|mimes:pdf,jpg,jpeg,png',
                'copie_certificat_propriete'         => 'nullable|file|mimes:pdf,jpg,jpeg,png',
                'engagement'                         => 'nullable|file|mimes:pdf,jpg,jpeg,png',
                'exemple_projet'                     => 'nullable|file|mimes:pdf,jpg,jpeg,png',
            ]);

            $utilisateur->update([
                'nom_prenom'       => $request->nom_prenom,
                'id_type_identite' => $request->id_type_identite,
                'cin'              => $request->cin,
                'passport'         => $request->passport,
                'num_fiscal'       => $request->num_fiscal,
                'adresse'          => $request->adresse,
                'tel'              => $request->tel,
                'email'            => $request->email,
                'date_naissance'   => $request->date_naissance,
                'lieu_naissance'   => $request->lieu_naissance,
            ]);

            $files = [
                'copie_cahier_des_charges',
                'copie_originale',
                'facture',
                'contrat',
                'fichier_arrivage',
                'fichier_colisage',
                'copie_registre_fiscal',
                'copie_registre_national_institutions',
                'copie_certificat_propriete',
                'engagement',
                'exemple_projet'
            ];

           
			foreach ($files as $fileField) {
				if ($request->hasFile($fileField)) {

					$file = $request->file($fileField);

					// Lire le contenu binaire du fichier
					$document->$fileField = file_get_contents($file->getRealPath());
				}
			}

          

            if ($request->id_statut != 2 && $utilisateur && !empty($utilisateur->email)) {
                $messageStatus = '';

                switch ($request->id_statut) {
                    case 1:
                        $messageStatus = " تم <b>قبول</b> طلبكم بنجاح. نهنئكم على هذا القبول.";
                        break;

                    case 3:
                        $messageStatus = " تم <b>رفض</b> طلبكم.<br>السبب: <b>" . e($request->commentaire_motif) . "</b>";
                        break;

                    case 4:
                        $messageStatus = " طلبكم <b>غير مكتمل</b>.<br>يرجى استكمال النواقص التالية: <b>" . e($request->commentaire_acceptation) . "</b>";
                        break;

                    case 5:
						$note = trim((string) $request->commentaire_avis);

						if ($note !== '') {
							$messageStatus = "تم <b>إرجاء طلبكم لإبداء الرأي</b>.<br>الملاحظة: <b>" . e($note) . "</b>";
						} else {
							$messageStatus = "تم <b>إرجاء طلبكم لإبداء الرأي</b>.";
						}
						break;
                }

            //Mail::to($utilisateur->email)->send(new DocumentStatusMail($utilisateur, $request->id_statut, $messageStatus));
			$oldStatus = (int) ($document->id_statut ?? 0);
			$newStatus = (int) $request->input('id_statut', $oldStatus);

			// choisir le commentaire selon le type de statut
			$commentRefus   = $request->input('commentaire_motif');
			$commentAccept  = $request->input('commentaire_acceptation');
			$commentAvis    = $request->input('commentaire_avis');

     $comment = null;
        if ($request->filled('commentaire_refus')) {
            $comment = trim($request->input('commentaire_refus'));
        } elseif ($request->filled('commentaire_status')) {
            $comment = trim($request->input('commentaire_status'));
        } elseif ($request->filled('commentaire_motif')) {
            $comment = trim($request->input('commentaire_motif'));
        }
        if ($comment === '') $comment = null;
		
		$document->fill([
			'nom_projet'              => $request->nom_projet,
			'adresse_projet'          => $request->adresse_projet,
			'id_statut'               => $request->id_statut,
			'commentaire_motif'       => $request->commentaire_motif,
			'commentaire_acceptation' => $request->commentaire_acceptation,
			'commentaire_avis'        => $request->commentaire_avis,
			'etat_formulaire'         => $request->id_statut != 2 ? 'reponse' : 'remplir',
		])->save(); 	


		// si le statut a changé, créer l'historique AVANT (ou APRÈS) la mise à jour
		if ($oldStatus !== $newStatus) {
			ImmobilierStatutHistory::create([
				'documents_formulaire_id' => $document->id_doc,
				'from_statut_id'          => $oldStatus ?: null,
				'to_statut_id'            => $newStatus,
				'changed_by'              => Auth::id() ?? null,
				'commentaire'             => $comment,
			]);
		}
		// ensuite tu peux mettre à jour le document (ou faire la création d'historique après en utilisant getOriginal())
		$document->update([
			'nom_projet'    => $request->nom_projet,
			'adresse_projet'=> $request->adresse_projet,
			'id_statut'     => $request->id_statut,
			'commentaire_motif'      => $request->commentaire_motif,
			'commentaire_acceptation'=> $request->commentaire_acceptation,
			'commentaire_avis'       => $request->commentaire_avis,
			'etat_formulaire' => $request->id_statut != 2 ? 'reponse' : 'remplir',
		]);
		// normalization map for statuses
		$statusMap = [
			1 => 'accepted',
			3 => 'rejected',
			4 => 'incomplete',
			5 => 'deferred',
		];

			// choose normalized type
			$normType = $statusMap[$newStatus] ?? 'status_changed';

			// prepare translation key and params
			$translationKey = "site.notifications.documents_formulaire.{$normType}";
			$translationParams = [];

			// fill params depending on status
			if ($normType === 'rejected') {
				$translationParams['reason'] = $commentRefus ?? ($comment ?? '');
			} elseif ($normType === 'incomplete') {
				$translationParams['missing'] = $commentAccept ?? ($comment ?? '');
			} elseif ($normType === 'deferred') {
				$translationParams['note'] = $commentAvis ?? ($comment ?? '');
			}


			$numeroDossier = $document->numero_dossier
				?? $document->num_dossier
				?? '—';

			if (empty($messageStatus)) {
				$messageStatus = 'لا توجد ملاحظات إضافية.';
			}



            Mail::to($utilisateur->email)->send(
                //new DocumentStatusMail($utilisateur, $request->id_statut, $messageStatus, $document->num_dossier)
				 new DocumentStatusMail(
        $utilisateur,
        //$request->id_statut,
		$normType,  
        $messageStatus,
        $numeroDossier
    )
            );

						//$utilisateur->notify(new DocumentStatusNotification(strip_tags($messageStatus)));
				$utilisateur->notify(new \App\Notifications\DocumentStatusNotification(
				$normType,
				$translationParams,
				$document->num_dossier
			));
        }

            DB::commit();

            return redirect()->route('admin.exoneration.index')
                            ->with('success', 'تم تحديث المشروع بنجاح ');

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->withErrors(['error' => 'حدث خطأ أثناء التحديث: ' . $e->getMessage()])
                        ->withInput();
        }
    } */

    public function show($id)
    {
        $document = DocumentsFormulaire::with('utilisateur')->findOrFail($id);
        $formulaire = TypeFormulaire::find($document->id_formulaire);
        $statuts = TpStatut::orderBy('id_statut')->get();
        return view('backoffice.investissement.gestion_exoneration.show', compact('document','formulaire','statuts'));
    }


    

public function toggleStatut(Request $request, $id)
{
    $request->validate([
        'id_statut' => 'required|integer|in:1,3,4,5',
        'commentaire_motif' => 'nullable|string',
        'commentaire_acceptation' => 'nullable|string',
        'commentaire_avis' => 'nullable|string',
        // legacy / alternative names
        'commentaire_refus' => 'nullable|string',
        'commentaire_status' => 'nullable|string',
    ]);

    DB::beginTransaction();
    try {
        // Load document + user
        $document = DocumentsFormulaire::with('utilisateur')->findOrFail($id);
        $utilisateur = $document->utilisateur;

        $oldStatus = (int) ($document->id_statut ?? 0);
        $newStatus = (int) $request->input('id_statut');

        // no-op
        if ($oldStatus === $newStatus) {
            DB::rollBack();
            return redirect()->back()->with('info', 'لا تغيير في الوضعية.');
        }

        // Select comment using same priority as update()
        $comment = null;
        if ($request->filled('commentaire_refus')) {
            $comment = trim($request->input('commentaire_refus'));
        } elseif ($request->filled('commentaire_status')) {
            $comment = trim($request->input('commentaire_status'));
        } elseif ($request->filled('commentaire_motif')) {
            $comment = trim($request->input('commentaire_motif'));
        }

        // fallback to acceptation / avis
        if (empty($comment)) {
            if ($request->filled('commentaire_acceptation')) {
                $comment = trim($request->input('commentaire_acceptation'));
            } elseif ($request->filled('commentaire_avis')) {
                $comment = trim($request->input('commentaire_avis'));
            }
        }
        if ($comment === '') $comment = null;

        // Apply new status and set comment fields consistently
        $document->id_statut = $newStatus;

        if ($newStatus == 3) { // refused
            $document->commentaire_motif = $request->input('commentaire_motif') 
                ?? $request->input('commentaire_refus')
                ?? $comment;
            $document->commentaire_acceptation = null;
            $document->commentaire_avis = null;
        } elseif ($newStatus == 4) { // incomplete
            $document->commentaire_acceptation = $request->input('commentaire_acceptation') ?? $comment;
            $document->commentaire_motif = null;
            $document->commentaire_avis = null;
        } elseif ($newStatus == 5) { // deferred
            $document->commentaire_avis = $request->input('commentaire_avis') ?? $comment;
            $document->commentaire_motif = null;
            $document->commentaire_acceptation = null;
        } else { // 1 or other
            $document->commentaire_motif = $request->input('commentaire_motif') ?? $comment;
            $document->commentaire_acceptation = null;
            $document->commentaire_avis = null;
        }

        if ($document->id_statut != 2) {
            $document->etat_formulaire = 'reponse';
        }

        // Save document first
        $document->save();

        // Create history row and keep it for later (so we can use its commentaire)
        $history = ImmobilierStatutHistory::create([
            'documents_formulaire_id' => $document->id_doc,
            'from_statut_id'          => $oldStatus ?: null,
            'to_statut_id'            => $newStatus ?: null,
            'changed_by'              => Auth::id() ?? null,
            'commentaire'             => $comment,
        ]);

        DB::commit();
    } catch (\Exception $e) {
        DB::rollBack();
        \Log::error('toggleStatut: failed to change status for document ID '.$id.': '.$e->getMessage(), [
            'trace' => $e->getTraceAsString(),
            'request' => $request->all()
        ]);
        return back()->withErrors(['error' => 'حدث خطأ أثناء تغيير الوضعية: '.$e->getMessage()]);
    }

    // Post-commit: build mail text and send (best-effort)
    try {
        // fresh document
        $document->refresh();
        $numeroDossier = $document->numero_dossier ?? $document->num_dossier ?? $document->id_doc;

        // map numeric -> normalized type strings expected by the mail/template/notifications
        $statusMap = [
            1 => 'accepted',
            3 => 'rejected',
            4 => 'incomplete',
            5 => 'deferred',
        ];
        $normType = $statusMap[$newStatus] ?? 'status_changed';

        // Determine motif: prefer history comment -> request fields -> document fields
        $motif = $history->commentaire ?? null;
        if (empty($motif)) {
            $motif = $request->input('commentaire_motif')
                  ?? $request->input('commentaire_refus')
                  ?? $request->input('commentaire_acceptation')
                  ?? $request->input('commentaire_avis')
                  ?? $document->commentaire_motif
                  ?? $document->commentaire_acceptation
                  ?? $document->commentaire_avis
                  ?? null;
        }
        $motif = is_string($motif) ? trim($motif) : null;

        // Build a user-facing message (used in old mail structure if needed)
        $messageStatus = '';
        switch ($newStatus) {
            case 1:
                $messageStatus = " تم <b>قبول</b> طلبكم بنجاح. نهنئكم على هذا القبول.";
                break;
            case 3:
                $messageStatus = " تم <b>رفض</b> طلبكم.";
                if (!empty($motif)) $messageStatus .= "<br>السبب: <b>" . e($motif) . "</b>";
                break;
            case 4:
                $messageStatus = " طلبكم <b>غير مكتمل</b>.";
                if (!empty($motif)) $messageStatus .= "<br>يرجى استكمال النواقص التالية: <b>" . e($motif) . "</b>";
                break;
            case 5:
                $messageStatus = " تم <b>إرجاء طلبكم لإبداء الرأي</b>.";
                if (!empty($motif)) $messageStatus .= "<br>الملاحظة: <b>" . e($motif) . "</b>";
                break;
            default:
                $messageStatus = 'تم تحديث حالة طلبكم.';
        }

        // Build translation params for DB notification
        $translationParams = [];
        if ($normType === 'rejected') {
            $translationParams['reason'] = $motif ?? '';
        } elseif ($normType === 'incomplete') {
            $translationParams['missing'] = $motif ?? '';
        } elseif ($normType === 'deferred') {
            $translationParams['note'] = $motif ?? '';
        }

        if ($utilisateur && !empty($utilisateur->email)) {
            // send mail (DocumentStatusMail expects: ($utilisateur, string $type, $messageStatus, $numeroDossier))
            try {
                Mail::to($utilisateur->email)->send(
                    new \App\Mail\DocumentStatusMail($utilisateur, $normType, $messageStatus, $numeroDossier)
                );
                \Log::info("toggleStatut: mail sent for document {$document->id_doc}", [
                    'to' => $utilisateur->email, 'type' => $normType, 'dossier' => $numeroDossier
                ]);
            } catch (\Throwable $mailEx) {
                \Log::error("toggleStatut: mail failed for document {$document->id_doc}: " . $mailEx->getMessage());
            }

            // DB notification (translation-aware)
            try {
                $utilisateur->notify(new \App\Notifications\DocumentStatusNotification(
                    $normType,
                    $translationParams,
                    $numeroDossier
                ));
                \Log::info("toggleStatut: DB notification created", ['document' => $document->id_doc, 'user' => $utilisateur->id, 'type' => $normType]);
            } catch (\Throwable $notifEx) {
                \Log::error("toggleStatut: DB notification failed for document {$document->id_doc}: " . $notifEx->getMessage());
            }
        } else {
            \Log::warning("toggleStatut: no user or email for document {$document->id_doc}");
        }
    } catch (\Throwable $ex) {
        \Log::error("toggleStatut post-commit error for document {$id}: " . $ex->getMessage());
    }

    return redirect()->back()->with('success', 'تم تغيير الوضعية بنجاح');
}


}
