<?php

namespace App\Http\Controllers\Financement;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Projet;
use App\Models\TypeFormulaire;
use App\Models\TpTypeDocument;
use App\Models\DocumentsFinancement;
use App\Models\TypeFinancement;
use App\Models\TpStatut;
use App\Models\DocumentsFinancementList;
use App\Mail\Statut2DocumentChanged;
use App\Notifications\Statut2DocumentNotification;
use App\Models\Utilisateur;
use App\Models\DemandeFinancement;
use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;
use App\Models\ImmobilierStatutHistory;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use Illuminate\Support\Facades\DB;
use App\Models\ActivitePhase;
use App\Models\EstimationBudget;
use App\Models\ContributeurProjet;
use App\Models\PosteDepense;
use App\Models\TypeDocument;
use Illuminate\Support\Collection;
use Mpdf\Mpdf;
use Illuminate\Support\Facades\Schema;
use Carbon\Carbon;
use Illuminate\Support\Str;


class Document2FinancementController extends Controller
{
    public function index(Request $request)
{
    $query = DocumentsFinancementList::with([
        'statut2',
        'projet.superviseur',
        'projet.demande'
    ])
    ->where('id_statut1', 1)
    ->orderBy('id_doc_list', 'desc');

    // 🔍 Recherche par type de financement
    if ($request->filled('type_demande')) {
        $query->whereHas('projet.demande', function ($q) use ($request) {
            $q->where('type_financement_id', $request->type_demande);
        });
    }

    // 🔍 Recherche par nom du projet
    if ($request->filled('nom_projet')) {
        $query->whereHas('projet', function ($q) use ($request) {
            $q->where('nom_projet', 'like', '%' . $request->nom_projet . '%');
        });
    }

    // 🔍 Recherche par superviseur
    if ($request->filled('nom_superviseur')) {
        $query->whereHas('projet.superviseur', function ($q) use ($request) {
            $q->where('nom_prenom', 'like', '%' . $request->nom_superviseur . '%');
        });
    }

    $documents = $query->paginate(10)->appends($request->query());

    $type_financement = TypeFinancement::all();

    return view(
        'backoffice.financement.documents_financement.index2',
        compact('documents', 'type_financement')
    );
}
		
   
private function baseQueryIndex(Request $request)
{
    $query = DocumentsFinancementList::with([
        'statut2',
        'projet.association',
        'projet.superviseur',
        'projet.demande'
    ])
    ->where('id_statut1', 1)
    ->orderBy('id_doc_list', 'desc');

    if ($request->filled('nom_association')) {
        $query->whereHas('projet.association', function ($q) use ($request) {
            $q->where('nom_association', 'like', '%' . $request->nom_association . '%');
        });
    }

    if ($request->filled('type_demande')) {
        $query->whereHas('projet.demande', function ($q) use ($request) {
            $q->where('type_financement_id', $request->type_demande);
        });
    }

    if ($request->filled('nom_projet')) {
        $query->whereHas('projet', function ($q) use ($request) {
            $q->where('nom_projet', 'like', '%' . $request->nom_projet . '%');
        });
    }

    if ($request->filled('nom_superviseur')) {
        $query->whereHas('projet.superviseur', function ($q) use ($request) {
            $q->where('nom_prenom', 'like', '%' . $request->nom_superviseur . '%');
        });
    }

    return $query;
}



public function exportPdf(Request $request)
{
    $documents = $this->baseQueryIndex($request)->get();

    $html = view(
        'backoffice.financement.documents_financement.exports.document_pdf2',
        compact('documents')
    )->render();

    $mpdf = new \Mpdf\Mpdf([
        'mode' => 'utf-8',
        'format' => 'A4',
        'default_font' => 'dejavusans',
        'directionality' => 'rtl',
    ]);

    $mpdf->WriteHTML($html);

    return response(
        $mpdf->Output('documents_financement_index2.pdf', 'S')
    )->header('Content-Type', 'application/pdf');
}

public function exportExcel(Request $request)
{
    $documents = $this->baseQueryIndex($request)->get();

    $spreadsheet = new Spreadsheet();
    $sheet = $spreadsheet->getActiveSheet();
    $sheet->setRightToLeft(true);

    $sheet->fromArray([
        '',
        'اسم المشروع',
        'الحالة',
    ], null, 'A1');

    $sheet->getStyle('A1:C1')->getFont()->setBold(true);

    $row = 2;
    foreach ($documents as $i => $doc) {
        $sheet->setCellValue('A'.$row, $i + 1);
        $sheet->setCellValue('B'.$row, $doc->projet?->nom_projet);
       
        $sheet->setCellValue('C'.$row, $doc->statut2?->statut_ar);
        $row++;
    }

    foreach (range('A', 'C') as $col) {
        $sheet->getColumnDimension($col)->setAutoSize(true);
    }

    $writer = new Xlsx($spreadsheet);

    return response()->streamDownload(function () use ($writer) {
        $writer->save('php://output');
    }, 'documents_financement_index2.xlsx');
}

    public function edit($id_doc_list)
    {
        $docList = DocumentsFinancementList::with(['projet',
            'projet.superviseur',
            'projet.activites',
            'projet.estimationBudget',
            'projet.estimationBudget.contributeurs',
            'projet.estimationBudget.postesDepense','documents',
            'documents.typeDocument',
            'statut1', 'statut2'
        ])->findOrFail($id_doc_list);

        $formulaire = TypeFormulaire::find(6);
        $formulaire2 = TypeFormulaire::find(7);
        $formulaire3 = TypeFormulaire::find(8);
        $type_financement = TypeFinancement::all();
        $demande_id = $docList->projet->id_demande;
        $demande = DemandeFinancement::with('association')->find($demande_id);

        $selectedAssociationId = $demande && $demande->association 
            ? $demande->association->id_association 
            : null;

        $selectedAssociationName = $demande && $demande->association 
            ? $demande->association->nom_association 
            : null;

        $id_projet = $docList->projet->id_projet; 
        $documents = DocumentsFinancementList::where('id_projet', $id_projet)->get();
        $typedocum = TpTypeDocument::all();

        $selectedTypes = DemandeFinancement::where('association_id', $demande->association_id)
                                            ->pluck('type_financement_id')
                                            ->toArray();

        $documents = $docList->documents;
        $statuts = TpStatut::orderBy('id_statut')->get();
        return view('backoffice.financement.documents_financement.edit2', compact(
            'docList','formulaire','formulaire2','formulaire3','type_financement',
            'demande_id','demande','selectedAssociationId','selectedAssociationName','documents',
            'id_projet','typedocum','selectedTypes','docList','statuts'
        ));
    }

    
public function update(Request $request, $id)
{
    DB::beginTransaction();

    try {
        $rules = [
            // Association
            'nom_association' => 'required|string|max:255',
            'adresse' => 'required|string|max:255',
            'code_postal' => 'required|string|max:20',
            'tel' => 'required|string|max:50',
            'fax' => 'required|string|max:50',
            'date_creation' => 'required|date',
            'num_visa' => 'required|string|max:100',
            'date_visa' => 'required|date',
            'num_identification_fiscale' => 'required|string|max:100',
            'num_compte_courant' => 'required|string|max:100',
            'banque' => 'required|string|max:255',
            'adresse_mail' => 'required|email|max:255',
            'adresse_site_internet' => 'required|string|max:255',
            'nom_president_association' => 'required|string|max:255',
            'tel_personnel' => 'required|string|max:50',
            'num_membre_bureau_national_a' => 'required|integer',
            'num_bureaux_regionaux' => 'required|integer',
            'num_total_membre' => 'required|integer',
            'types_financement' => 'required|array|min:1',
            'id_formulaire' => 'required|integer|exists:tp_type_formulaire,id_type',

            // Superviseur
            'nom_prenom_sup' => 'required|string|max:100',
            'profession_sup' => 'required|string|max:100',
            'role_sup' => 'required|string|max:100',
            'tel_sup' => 'required|string|max:12',
            'email_sup' => 'required|email|max:255',

            // Projet
            'nom_projet' => 'required|string|max:255',
            'justification_realisation' => 'required|string|max:255',
            'objectives' => 'required|string',
            'public' => 'required|string|max:255',
            'nombre_spectateur' => 'required|integer|min:1',
            'tranch_age_min' => 'required|integer|min:0',
            'tranch_age_max' => 'required|integer|gte:tranch_age_min',
            'parties_realisation' => 'required|string',
            'resultat_quantitatif' => 'required|string',
            'resultat_qualitatif' => 'required|string',
            'perspectives' => 'required|string',

            // Phases
            'phases.*.description' => 'required|string|max:200',
            'phases.*.completion_date' => 'required|date',
            'phases.*.completion_location' => 'required|string|max:100',

            // Budget
            'budget.*.total_cost' => 'required|numeric|min:0',
            'budget.*.self_percent' => 'required|numeric|min:0|max:100',
            'budget.*.self_amount' => 'required|numeric|min:0',
            'budget.*.ministry_amount' => 'required|numeric|min:0',
            'budget.*.ministry_percent' => 'required|numeric|min:0|max:100',
            'budget.*.supporters.*.name' => 'required|string|max:255',
            'budget.*.supporters.*.amount' => 'required|numeric|min:0',
            'budget.*.supporters.*.percent' => 'required|numeric|min:0|max:100',

            // Dépenses
            'depenses.*.*.libelle_depense' => 'required|string|max:200',
            'depenses.*.*.description' => 'required|string|max:255',
            'depenses.*.*.cout' => 'required|numeric|min:0',
            
            // Documents
            'id_statut2' => 'nullable|integer|in:1,2,3,4,5',
            'commentaire_motif2' => 'nullable|string|max:500',
            'decision_finale' => 'nullable|file|mimes:pdf,jpg,jpeg,png,doc,docx|max:5120',
            'doc_fin.*.id_doc' => 'required|integer|exists:tp_type_document,id_type_doc',
            'doc_fin.*.file' => 'nullable|file|mimes:pdf,jpg,jpeg,png,doc,docx|max:2048',
            'doc_fin.*.remarques' => 'nullable|string|max:100',
        ];

        $request->validate($rules);

        $docList = DocumentsFinancementList::findOrFail($id);
        $projet = $docList->projet;
        $association = $projet->association;
        $superviseur = $projet->superviseur;
        
        // --- Association update (guarded, create if possible) ---
        $associationData = $request->only([
            'nom_association','adresse','code_postal','tel','fax',
            'date_creation','num_visa','date_visa','num_identification_fiscale',
            'num_compte_courant','banque','adresse_mail','adresse_site_internet',
            'nom_president_association','tel_personnel','num_membre_bureau_national_a',
            'num_bureaux_regionaux','num_total_membre','id_formulaire'
        ]);

        if ($association) {
            try {
                $association->update($associationData);
            } catch (\Exception $ex) {
                Log::error("Failed to update association for projet {$projet->id_projet}: " . $ex->getMessage(), [
                    'docList' => $docList->id_doc_list
                ]);
            }
        } else {
            try {
                if (method_exists($projet, 'association')) {
                    $association = $projet->association()->create($associationData + ['id_formulaire' => $associationData['id_formulaire'] ?? null]);
                } else {
                    Log::warning("Association missing and projet->association relation not found; skipping association create.", ['projet' => $projet->id_projet, 'docList' => $docList->id_doc_list]);
                }
            } catch (\Exception $ex) {
                Log::error("Failed to create association for projet {$projet->id_projet}: " . $ex->getMessage(), ['docList' => $docList->id_doc_list]);
            }
        }

        // --- Superviseur update (guarded) ---
        $superviseurData = $request->only(['nom_prenom_sup','profession_sup','role_sup','tel_sup','email_sup']);
        if ($superviseur) {
            try {
                $superviseur->update($superviseurData);
            } catch (\Exception $ex) {
                Log::error("Failed to update superviseur for projet {$projet->id_projet}: " . $ex->getMessage(), ['docList' => $docList->id_doc_list]);
            }
        } else {
            Log::warning("Superviseur missing for projet {$projet->id_projet}; skipping superviseur update.", ['docList' => $docList->id_doc_list]);
        }

        $projet->update([
            'id_formulaire' => $request->id_formulaire,
            'nom_projet' => $request->nom_projet,
            'justification_realisation' => $request->justification_realisation,
            'objectives' => $request->objectives,
            'public' => $request->public,
            'id_user' => auth()->id(),
            'nombre_spectateur' => $request->nombre_spectateur,
            'tranch_age_min' => $request->tranch_age_min,
            'tranch_age_max' => $request->tranch_age_max,
            'parties_realisation' => $request->parties_realisation,
            'resultat_quantitatif' => $request->resultat_quantitatif,
            'resultat_qualitatif' => $request->resultat_qualitatif,
            'perspectives' => $request->perspectives,
        ]);

        $projet->activites()->delete();
        if ($request->has('phases')) {
            foreach ($request->phases as $phase) {
                if (!empty($phase['description'])) {
                    ActivitePhase::create([
                        'id_projet' => $projet->id_projet,
                        'description' => $phase['description'],
                        'completion_date' => $phase['completion_date'] ?? null,
                        'completion_location' => $phase['completion_location'] ?? null,
                    ]);
                }
            }
        }

        // Reconcile EstimationBudget (UPDATE / CREATE / DELETE)
        $incomingBudgets = $request->input('budget', []);
        $existingEstimations = $projet->estimationBudget()->with(['contributeurs', 'postesDepense'])->get();

        foreach ($incomingBudgets as $bIndex => $budgetData) {
            $estimation = $existingEstimations->get($bIndex);

            $estimationData = [
                'projet_id' => $projet->id_projet,
                'cout_total_estime' => $budgetData['total_cost'] ?? 0,
                'financement_propre_pourcentage' => $budgetData['self_percent'] ?? 0,
                'financement_propre_montant' => $budgetData['self_amount'] ?? 0,
                'aide_ministere_montant' => $budgetData['ministry_amount'] ?? 0,
                'aide_ministere_pourcentage' => $budgetData['ministry_percent'] ?? 0,
            ];

            if ($estimation) {
                $estimation->update($estimationData);
            } else {
                $estimation = EstimationBudget::create($estimationData);
            }

            // Recreate supporters (Contributeurs)
            $estimation->contributeurs()->delete();
            if (!empty($budgetData['supporters']) && is_array($budgetData['supporters'])) {
                foreach ($budgetData['supporters'] as $supporter) {
                    if (!empty($supporter['name'])) {
                        ContributeurProjet::create([
                            'id_estimation_budget' => $estimation->id_estimation,
                            'nom' => $supporter['name'],
                            'montant' => $supporter['amount'] ?? 0,
                            'pourcentage' => $supporter['percent'] ?? 0,
                        ]);
                    }
                }
            }

            // Reconcile postesDepense
            $estimation->postesDepense()->delete();
            $depensesForBudget = $request->input('depenses.' . $bIndex, []);
            if (!empty($depensesForBudget) && is_array($depensesForBudget)) {
                foreach ($depensesForBudget as $depenseData) {
                    if (!empty($depenseData['libelle_depense'])) {
                        PosteDepense::create([
                            'id_estimation_budget' => $estimation->id_estimation,
                            'libelle_depense' => $depenseData['libelle_depense'],
                            'description' => $depenseData['description'] ?? '',
                            'cout' => $depenseData['cout'] ?? 0,
                        ]);
                    }
                }
            }
        }

        // Delete surplus existing estimations
        $incomingCount = is_array($incomingBudgets) ? count($incomingBudgets) : 0;
        if ($existingEstimations->count() > $incomingCount) {
            $surplus = $existingEstimations->slice($incomingCount);
            foreach ($surplus as $estToDelete) {
                $estToDelete->contributeurs()->delete();
                $estToDelete->postesDepense()->delete();
                $estToDelete->delete();
            }
        }

        // -----------------------------
        // Documents : statut, commentaire et fichier final
        // -----------------------------
        if ($request->filled('id_statut')) {
            $docList->update([
                'id_statut2' => $request->id_statut,
                'commentaire_motif2' => $request->id_statut == 3 ? $request->commentaire_motif2 : null,
            ]);
        }

        // MODIFICATION: GESTION DES FICHIERS DANS LE DOSSIER EXTERNE
        $uploadDir = '/home/preprov/www/mjs/shared_uploads/images';
        
        // Vérifier si le dossier existe, sinon le créer
        if (!is_dir($uploadDir)) {
            if (!mkdir($uploadDir, 0755, true)) {
                DB::rollBack();
                abort(500, 'Impossible de créer le dossier upload');
            }
        }
        
        // Vérifier les permissions d'écriture
        if (!is_writable($uploadDir)) {
            DB::rollBack();
            abort(500, 'Dossier upload non accessible en écriture');
        }

        // Gestion des fichiers doc_fin
        if ($request->has('doc_fin')) {
            foreach ($request->doc_fin as $docData) {
                // Chercher le document existant
                $doc = DocumentsFinancement::where('id_doc_list', $docList->id_doc_list)
                                          ->where('id_doc', $docData['id_doc'])
                                          ->first();
                
                // Si le document n'existe pas, en créer un nouveau
                if (!$doc) {
                    $doc = new DocumentsFinancement();
                    $doc->id_doc = $docData['id_doc'];
                    $doc->id_doc_list = $docList->id_doc_list;
                }
                
                // Si un nouveau fichier est fourni
                if (!empty($docData['file']) && $docData['file'] instanceof \Illuminate\Http\UploadedFile) {
                    $file = $docData['file'];
                    
                    // Supprimer l'ancien fichier s'il existe (si c'est un chemin)
                    if ($doc->file && is_string($doc->file) && file_exists($doc->file)) {
                        @unlink($doc->file);
                    }
                    
                    // Récupérer les informations du fichier
                    $originalName = $file->getClientOriginalName();
                    $extension = $file->getClientOriginalExtension();
                    
                    // Nom sécurisé
                    $original = pathinfo($originalName, PATHINFO_FILENAME);
                    $safe = Str::slug(mb_substr($original, 0, 40));
                    if ($safe === '') {
                        $safe = 'document';
                    }
                    
                    // Nom unique avec timestamp et random
                    $filename = $safe . '-' . time() . '-' . rand(1000, 9999) . '.' . $extension;
                    $targetPath = $uploadDir . DIRECTORY_SEPARATOR . $filename;
                    
                    // Déplacer le fichier
                    try {
                        $file->move($uploadDir, $filename);
                    } catch (\Exception $e) {
                        DB::rollBack();
                        abort(500, 'Erreur lors du déplacement du fichier: ' . $e->getMessage());
                    }
                    
                    // Vérifier que le fichier a bien été créé
                    if (!file_exists($targetPath)) {
                        DB::rollBack();
                        abort(500, 'Fichier non créé après déplacement');
                    }
                    
                    // Mettre à jour le champ file (chemin du fichier)
                    $doc->file = $targetPath;
                }
                
                // Mettre à jour les remarques si fournies
                if (isset($docData['remarques'])) {
                    $doc->remarques = $docData['remarques'];
                }
                
                $doc->save();
            }
        }

        // Gestion du fichier decision_finale
        if ($request->hasFile('decision_finale')) {
            $file = $request->file('decision_finale');
            
            // Supprimer l'ancien fichier decision_finale s'il existe (si c'est un chemin)
            if ($docList->decision_finale && is_string($docList->decision_finale) && file_exists($docList->decision_finale)) {
                @unlink($docList->decision_finale);
            }
            
            // Récupérer les informations du fichier
            $originalName = $file->getClientOriginalName();
            $extension = $file->getClientOriginalExtension();
            
            // Nom sécurisé
            $original = pathinfo($originalName, PATHINFO_FILENAME);
            $safe = Str::slug(mb_substr($original, 0, 40));
            if ($safe === '') {
                $safe = 'decision';
            }
            
            // Nom unique avec timestamp et random
            $filename = $safe . '-decision-' . time() . '-' . rand(1000, 9999) . '.' . $extension;
            $targetPath = $uploadDir . DIRECTORY_SEPARATOR . $filename;
            
            // Déplacer le fichier
            try {
                $file->move($uploadDir, $filename);
            } catch (\Exception $e) {
                DB::rollBack();
                abort(500, 'Erreur lors du déplacement du fichier decision_finale: ' . $e->getMessage());
            }
            
            // Vérifier que le fichier a bien été créé
            if (!file_exists($targetPath)) {
                DB::rollBack();
                abort(500, 'Fichier decision_finale non créé après déplacement');
            }
            
            // Mettre à jour decision_finale avec le chemin
            $docList->decision_finale = $targetPath;
        }

        // --- PREPARE: capture original statut2 and compute new statut2 from request ---
        $normalizeStat = function ($val) {
            if (is_null($val) || $val === '') return null;
            $i = (int) $val;
            return ($i > 0) ? $i : null;
        };

        $origStat2 = $docList->getOriginal('id_statut2');
        $oldStat2 = $normalizeStat($origStat2);

        $inputStat2 = $request->input('id_statut2', null);
        $newStat2 = $normalizeStat($inputStat2);

        $commentForHistory = null;
        if ($request->filled('commentaire_refus')) {
            $commentForHistory = trim($request->input('commentaire_refus'));
        } elseif ($request->filled('commentaire_status')) {
            $commentForHistory = trim($request->input('commentaire_status'));
        } elseif ($request->filled('commentaire_motif2')) {
            $commentForHistory = trim($request->input('commentaire_motif2'));
        }

        // Prepare the update data for the docList
        $updateData = [];

        if (!is_null($newStat2)) {
            $updateData['id_statut2'] = $newStat2;
            $updateData['commentaire_motif2'] = ($newStat2 === 3 && $request->filled('commentaire_motif2'))
                ? trim($request->input('commentaire_motif2'))
                : ($newStat2 === 3 ? ($commentForHistory ?? null) : null);
        }

        if (!empty($updateData)) {
            $docList->update($updateData);
        }

        // --- HISTORY: only create history record if statut2 actually changed ---
        if ($oldStat2 !== $newStat2) {
            $projetIdForHistory = null;
            if (!empty($docList->id_projet)) {
                $existsProject = DB::table('projet_divertissement_jeune_enfant')
                                ->where('id_projet', $docList->id_projet)
                                ->exists();
                $projetIdForHistory = $existsProject ? $docList->id_projet : null;
            }

            ImmobilierStatutHistory::create([
                'documents_financement_list_id2' => $docList->id_doc_list,
                'projet_id' => $projetIdForHistory,
                'from_statut_id' => $oldStat2,
                'to_statut_id' => $newStat2,
                'changed_by' => Auth::id() ?? null,
                'commentaire' => $commentForHistory,
            ]);
        }

        DB::commit();

        // ---- send DB notification for statut2 change (if any) ----
        try {
            $recipient = $docList->user ?? null;

            if (!is_null($oldStat2) || !is_null($newStat2)) {
                $oldStat2 = $oldStat2 ?? (int) ($docList->getOriginal('id_statut2') ?? 0);
                $newStat2 = $newStat2 ?? (int) ($docList->id_statut2 ?? 0);
            }

            if ($oldStat2 !== $newStat2 && $recipient) {
                $statusMap = [
                    1 => 'accepted',
                    3 => 'rejected',
                    4 => 'incomplete',
                    5 => 'deferred',
                ];
                $type = $statusMap[$newStat2] ?? 'status_changed';

                $translationParams = [];
                if ($newStat2 === 3) {
                    $translationParams['reason'] = $request->input('commentaire_motif2') ?? $commentForHistory ?? '';
                } elseif ($newStat2 === 4) {
                    $translationParams['missing'] = $request->input('commentaire_acceptation') ?? $commentForHistory ?? '';
                } elseif ($newStat2 === 5) {
                    $translationParams['note'] = $request->input('commentaire_avis') ?? $commentForHistory ?? '';
                } elseif ($newStat2 === 1) {
                    $translationParams['date'] = now()->format('Y-m-d');
                }

                $idPad = str_pad($docList->id_doc_list, 3, '0', STR_PAD_LEFT);
                $translationParams['reference'] = $idPad;

                // Mail
                if (!empty($recipient->email)) {
                    try {
                        $statusMap = [1 => 'accepted', 3 => 'rejected', 4 => 'incomplete', 5 => 'deferred'];
                        $newStatVal = (int) ($docList->id_statut2 ?? $docList->id_statut1 ?? 0);
                        $type = $statusMap[$newStatVal] ?? 'status_changed';
                        $params = [];
                        if ($newStatVal === 3) $params['reason'] = $request->input('commentaire_motif2') ?? ($commentForHistory ?? '');
                        if ($newStatVal === 4) $params['missing'] = $request->input('commentaire_acceptation') ?? ($commentForHistory ?? '');
                        if ($newStatVal === 5) $params['note'] = $request->input('commentaire_avis') ?? ($commentForHistory ?? '');
                        if ($newStatVal === 1) $params['date'] = now()->format('Y-m-d');
                        $params['reference'] = str_pad($docList->id_doc_list, 3, '0', STR_PAD_LEFT);
                        if (!empty($commentForHistory)) $params['comment'] = $commentForHistory;

                        if (!empty($recipient->email)) {
                            try {
                                \Mail::to($recipient->email)->send(new \App\Mail\Statut2DocumentChanged($docList, $type, $params, $recipient));
                            } catch (\Exception $ex) {
                                \Log::error("Mail error while sending statut mail for docList {$docList->id_doc_list}: " . $ex->getMessage());
                            }
                        }
                    } catch (\Exception $ex) {
                        \Log::error("Mail error while sending statut2 mail for docList {$docList->id_doc_list}: " . $ex->getMessage());
                    }
                }

                // Store DB notification
                $translationKey = "site.notifications.documents_formulaire.{$type}";
                $recipient->notify(new \App\Notifications\Statut2DocumentNotification(
                    $docList,
                    $type,
                    $translationParams,
                    $docList->id_doc_list,
                    $translationKey
                ));
            }
        } catch (\Exception $ex) {
            \Log::error("Notification error after update for docList {$docList->id_doc_list}: " . $ex->getMessage());
        }

        return redirect()->route('admin.doc_fin2.index', $docList->id_doc_list)
            ->with('success', 'تم تحديث الوثيقة بنجاح');

    } catch (\Exception $e) {
        DB::rollBack();
        return back()->withErrors(['error' => 'خطأ أثناء التحديث: ' . $e->getMessage()]);
    }
}

public function showDecision($id)
{
    $doc = DocumentsFinancementList::findOrFail($id);
    
    if (!$doc->decision_finale) {
        abort(404, "Aucun fichier.");
    }
    
    // Si c'est un chemin (nouveau système), lire le fichier depuis le disque
    if (is_string($doc->decision_finale) && file_exists($doc->decision_finale)) {
        // Vérifier si le fichier existe physiquement
        if (!file_exists($doc->decision_finale)) {
            abort(404, 'Fichier decision_finale introuvable sur le disque.');
        }
        
        // Déterminer le type MIME à partir du fichier
        $mimeType = mime_content_type($doc->decision_finale);
        
        // Lire le contenu du fichier
        $content = file_get_contents($doc->decision_finale);
        
        return response($content)
            ->header('Content-Type', $mimeType)
            ->header('Content-Disposition', 'inline; filename="' . basename($doc->decision_finale) . '"');
    } 
    // Si c'est un BLOB (ancien système - rétrocompatibilité)
    else {
        return response($doc->decision_finale)
            ->header('Content-Type', 'application/octet-stream')
            ->header('Content-Disposition', 'inline; filename="decision_finale"');
    }
}

public function showFile($id)
{
    $doc = DocumentsFinancement::findOrFail($id);
    
    if (!$doc->file) {
        abort(404, 'Aucun fichier.');
    }
    
    // Si c'est un chemin (nouveau système), lire le fichier depuis le disque
    if (is_string($doc->file) && file_exists($doc->file)) {
        // Vérifier si le fichier existe physiquement
        if (!file_exists($doc->file)) {
            abort(404, 'Fichier introuvable sur le disque.');
        }
        
        // Déterminer le type MIME à partir du fichier
        $mimeType = mime_content_type($doc->file);
        
        // Lire le contenu du fichier
        $content = file_get_contents($doc->file);
        
        return response($content)
            ->header('Content-Type', $mimeType)
            ->header('Content-Disposition', 'inline; filename="' . basename($doc->file) . '"');
    } 
    // Si c'est un BLOB (ancien système - rétrocompatibilité)
    else {
        // Detecter automatiquement le type MIME
        $finfo = finfo_open();
        $mimeType = finfo_buffer($finfo, $doc->file, FILEINFO_MIME_TYPE);
        finfo_close($finfo);
        
        return response($doc->file)
            ->header('Content-Type', $mimeType);
    }
}

	/*public function showDecision($id)	{	
        	$doc = DocumentsFinancementList::findOrFail($id);	
            	if (!$doc->decision_finale) {	
                    abort(404, "Aucun fichier.");	
                }		
             return response($doc->decision_finale)->header('Content-Type', $doc->decision_finale_mime)->header('Content-Disposition', 'inline; filename="decision_finale"');
  }*/


    public function show($id)
    {
        $docList = DocumentsFinancementList::with([
            'projet.superviseur',
            'projet.activites',
            'projet.estimationBudget',
            'projet.estimationBudget.contributeurs',
            'projet.estimationBudget.postesDepense','documents',
            'documents.typeDocument',
            'statut1', 'statut2'
        ])->findOrFail($id);
        $currentStatus = $docList->id_statut2;
        $formulaire = TypeFormulaire::find(6);
        $formulaire2 = TypeFormulaire::find(7);
        $formulaire3 = TypeFormulaire::find(8);
        $type_financement = TypeFinancement::all();

        $demande_id = $docList->projet->id_demande;
        $demande = DemandeFinancement::with('association')->find($demande_id);

        $selectedAssociationId = $demande && $demande->association 
            ? $demande->association->id_association 
            : null;

        $selectedAssociationName = $demande && $demande->association 
            ? $demande->association->nom_association 
            : null;

        $id_projet = $docList->projet->id_projet; 
        $documents = DocumentsFinancementList::where('id_projet', $id_projet)->get();
        $typedocum = TpTypeDocument::all();

        $selectedTypes = DemandeFinancement::where('association_id', $demande->association_id)
                                            ->pluck('type_financement_id')
                                            ->toArray();

        $documents = $docList->documents;
        $statuts = TpStatut::orderBy('id_statut')->get();
        return view('backoffice.financement.documents_financement.show2', compact(
            'docList','formulaire','formulaire2','formulaire3','type_financement','currentStatus',
            'demande_id','demande','selectedAssociationId','selectedAssociationName','documents',
            'id_projet','typedocum','selectedTypes','docList','statuts'
        ));
    }

/*
public function toggleStatut(Request $request, $id_doc_list)
{
    $request->validate([
        // Accept either explicit id_statut1/id_statut2 or legacy id_statut
        'id_statut' => 'nullable|integer|in:1,2,3,4,5',
        'id_statut1' => 'nullable|integer|in:1,2,3,4,5',
        'id_statut2' => 'nullable|integer|in:1,2,3,4,5',
        'commentaire_motif1' => 'nullable|string|max:1000',
        'commentaire_motif2' => 'nullable|string|max:1000',
        'commentaire_refus'  => 'nullable|string|max:1000',
        'commentaire_status' => 'nullable|string|max:1000',
        'commentaire_acceptation' => 'nullable|string|max:1000',
        'commentaire_avis' => 'nullable|string|max:1000',
    ]);

    $docList = DocumentsFinancementList::with(['user','projet'])->findOrFail($id_doc_list);
    $user = $docList->user ?? null;

    // Decide which column to update: prefer explicit id_statut2 then id_statut1 then legacy id_statut -> maps to statut1
    if ($request->filled('id_statut2')) {
        $field = 'id_statut2';
        $rawNew = $request->input('id_statut2');
    } elseif ($request->filled('id_statut1')) {
        $field = 'id_statut1';
        $rawNew = $request->input('id_statut1');
    } elseif ($request->filled('id_statut')) {
        // map legacy to statut1 (adjust if you want different)
        $field = 'id_statut1';
        $rawNew = $request->input('id_statut');
    } else {
        return redirect()->back()->with('info', 'لم يتم تحديد وضعية لتغييرها.');
    }

    if (!is_numeric($rawNew)) {
        return redirect()->back()->with('error', 'قيمة الوضعية المرسلة غير صالحة.');
    }

    $new = (int) $rawNew;

    // read original DB value for accurate history 'from' value
    $orig = $docList->getOriginal($field);
    $old = ($orig !== null && $orig !== '') ? ((int)$orig > 0 ? (int)$orig : null) : null;
    $to  = ($new > 0) ? $new : null;

    // Choose comment by priority (same logic as update)
    $comment = $request->input('commentaire_refus')
             ?? $request->input('commentaire_status')
             ?? ($field === 'id_statut2' ? $request->input('commentaire_motif2') : $request->input('commentaire_motif1'))
             ?? $request->input('commentaire_acceptation')
             ?? $request->input('commentaire_avis')
             ?? null;

    $comment = is_string($comment) ? trim($comment) : null;
    if ($comment === '') $comment = null;

    // quick no-op if nothing changes
    if ($old === $to) {
        return redirect()->back()->with('info', 'لا تغيير في الوضعية.');
    }

    DB::beginTransaction();
    try {
        // assign new statut (use null for non-positive)
        $docList->{$field} = $to;

        // business rule: if setting statut1 to accepted (1) => set statut2 = 2
        if ($field === 'id_statut1' && $to === 1) {
            $docList->id_statut2 = 2;
        }

        // store comment in the correct model column if applicable
        if ($field === 'id_statut2') {
            if ($to === 3) {
                // refusal for statut2
                if ($request->filled('commentaire_motif2')) {
                    $docList->commentaire_motif2 = trim($request->input('commentaire_motif2'));
                } elseif ($comment) {
                    $docList->commentaire_motif2 = $comment;
                }
            } else {
                // clear motif2 when not a refusal to avoid stale data
                if (Schema::hasColumn($docList->getTable(), 'commentaire_motif2')) {
                    $docList->commentaire_motif2 = null;
                }
            }
        } else {
            // field is statut1
            if ($to === 3) {
                if ($request->filled('commentaire_motif1')) {
                    $docList->commentaire_motif1 = trim($request->input('commentaire_motif1'));
                } elseif ($comment) {
                    $docList->commentaire_motif1 = $comment;
                }
            } else {
                if (Schema::hasColumn($docList->getTable(), 'commentaire_motif1')) {
                    $docList->commentaire_motif1 = null;
                }
            }
        }

        // keep etat_formulaire consistent if you use it
        if (($docList->id_statut1 ?? 2) != 2) {
            $docList->etat_formulaire = 'reponse';
        }

        $docList->save();

        // create history row
        ImmobilierStatutHistory::create([
            'documents_financement_list_id' => $docList->id_doc_list,
            'projet_id'                     => $docList->id_projet ?: null,
            'from_statut_id'                => $old,
            'to_statut_id'                  => $to,
            'changed_by'                    => auth()->id() ?? null,
            'commentaire'                   => $comment,
        ]);

        // if we changed statut1 and statut2 was set as side-effect, create history for statut2 too
        if ($field === 'id_statut1') {
            $origStat2 = (int) ($docList->getOriginal('id_statut2') ?? 0);
            $curStat2  = (int) ($docList->id_statut2 ?? 0);
            if ($origStat2 !== $curStat2) {
                ImmobilierStatutHistory::create([
                    'documents_financement_list_id' => $docList->id_doc_list,
                    'projet_id'                     => $docList->id_projet ?: null,
                    'from_statut_id'                => ($origStat2 > 0 ? $origStat2 : null),
                    'to_statut_id'                  => ($curStat2 > 0 ? $curStat2 : null),
                    'changed_by'                    => auth()->id() ?? null,
                    'commentaire'                   => $comment,
                ]);
            }
        }

        DB::commit();
    } catch (\Exception $e) {
        DB::rollBack();
        \Log::error("toggleStatut error for DocumentsFinancementList #{$id_doc_list}: " . $e->getMessage(), [
            'trace' => $e->getTraceAsString(),
            'request' => $request->all(),
        ]);
        return back()->withErrors(['error' => 'خطأ أثناء تغيير الحالة: ' . $e->getMessage()]);
    }

    // Build notification/mail payload (prefer the history comment we just saved)
    try {
        // best comment source (last history comment for this doc list)
        $historyComment = ImmobilierStatutHistory::where('documents_financement_list_id', $docList->id_doc_list)
            ->whereNotNull('commentaire')
            ->orderBy('created_at', 'desc')
            ->value('commentaire');

        $messageComment = $historyComment ?? ($field === 'id_statut2' ? $docList->commentaire_motif2 : $docList->commentaire_motif1) ?? $comment;

        // normalized type
        $statusMap = [1 => 'accepted', 3 => 'rejected', 4 => 'incomplete', 5 => 'deferred'];
        $resolved = (int) ($docList->{$field} ?? 0);
        $type = $statusMap[$resolved] ?? 'status_changed';

        // prepare params for notification/email
        $params = [];
        if ($resolved === 3) $params['reason'] = $messageComment ?? ($request->input('commentaire_refus') ?? '');
        if ($resolved === 4) $params['missing'] = $messageComment ?? ($request->input('commentaire_acceptation') ?? '');
        if ($resolved === 5) $params['note'] = $messageComment ?? ($request->input('commentaire_avis') ?? '');
        if ($resolved === 1) $params['date'] = now()->format('Y-m-d');

        $params['reference'] = str_pad($docList->id_doc_list, 3, '0', STR_PAD_LEFT);
        if (!empty($messageComment)) $params['comment'] = $messageComment;

        // Send mail (best-effort)
        if ($user && !empty($user->email)) {
            try {
                Mail::to($user->email)->send(new Statut2DocumentChanged($docList, $type, $params, $user));
            } catch (\Exception $ex) {
                \Log::error("Mail send failed for docList {$docList->id_doc_list}: " . $ex->getMessage());
            }
        } else {
            \Log::warning("toggleStatut: no recipient/email for DocumentsFinancementList ID {$docList->id_doc_list}");
        }

        // DB notification
        if ($user) {
            $translationKey = "site.notifications.documents_formulaire.{$type}";
            try {
                $user->notify(new Statut2DocumentNotification($docList, $type, $params, $docList->id_doc_list, $translationKey));
            } catch (\Exception $ex) {
                \Log::error("Notification error for docList {$docList->id_doc_list}: " . $ex->getMessage());
            }
        }
    } catch (\Exception $ex) {
        \Log::error("toggleStatut post-commit error for docList {$docList->id_doc_list}: " . $ex->getMessage());
    }

    return redirect()->route('admin.doc_fin2.index', $docList->id_doc_list)
                     ->with('success', 'تم تحديث الحالة بنجاح');
}

*/

  

public function toggleStatut(Request $request, $id_doc_list)
{
    $request->validate([
        // accept either id_statut1 / id_statut2 or legacy id_statut
        'id_statut' => 'nullable|integer|in:1,2,3,4,5',
        'id_statut1' => 'nullable|integer|in:1,2,3,4,5',
        'id_statut2' => 'nullable|integer|in:1,2,3,4,5',
        'commentaire_motif2' => 'nullable|string|max:1000',
        'commentaire_refus'  => 'nullable|string|max:1000',
        'commentaire_status' => 'nullable|string|max:1000',
        'commentaire_acceptation' => 'nullable|string|max:1000',
        'commentaire_avis' => 'nullable|string|max:1000',
    ]);

    $docList = DocumentsFinancementList::with(['user', 'projet'])->findOrFail($id_doc_list);
    $user = $docList->user;

    // Determine which DB column is available and which input was sent:
    $table = $docList->getTable();
    $hasStat2 = Schema::hasColumn($table, 'id_statut2');
    $hasStat1 = Schema::hasColumn($table, 'id_statut1');

    // Prefer explicit inputs in this order: id_statut2, id_statut1, id_statut
    $rawNew = null;
    $field = null;

    if ($request->filled('id_statut2') && $hasStat2) {
        $rawNew = $request->input('id_statut2');
        $field = 'id_statut2';
    } elseif ($request->filled('id_statut1') && $hasStat1) {
        $rawNew = $request->input('id_statut1');
        $field = 'id_statut1';
    } elseif ($request->filled('id_statut')) {
        // map legacy id_statut to an existing column
        if ($hasStat2) {
            $rawNew = $request->input('id_statut');
            $field = 'id_statut2';
        } elseif ($hasStat1) {
            $rawNew = $request->input('id_statut');
            $field = 'id_statut1';
        } else {
            // no column to update
            return redirect()->back()->withErrors(['error' => 'لا يمكن تغيير الحالة: حقل الحالة غير موجود في قاعدة البيانات.']);
        }
    } else {
        return redirect()->back()->with('info', 'لم يتم تحديد وضعية لتغييرها.');
    }

    // normalize new value
    if (!is_numeric($rawNew)) {
        return redirect()->back()->withErrors(['error' => 'القيمة المرسلة للوضعية غير صحيحة.']);
    }
    $new = (int) $rawNew;
    $to = ($new > 0) ? $new : null;

    // original (DB) value for the chosen field
    $orig = $docList->getOriginal($field);
    $old = ($orig !== null && $orig !== '') ? ((int)$orig > 0 ? (int)$orig : null) : null;

    // pick comment following same priority as update()
    $comment = $request->input('commentaire_refus')
             ?? $request->input('commentaire_status')
             ?? $request->input('commentaire_motif2')
             ?? $request->input('commentaire_acceptation')
             ?? $request->input('commentaire_avis')
             ?? null;
    $comment = is_string($comment) ? trim($comment) : null;
    if ($comment === '') $comment = null;

    DB::beginTransaction();
    try {
        // assign the new status on the proper field only (no attempt to write id_statut)
        $docList->{$field} = $to;

        // business rule: when setting id_statut1 => 1 then id_statut2 = 2 (if column exists)
        if ($field === 'id_statut1' && $to === 1 && $hasStat2) {
            $docList->id_statut2 = 2;
        }

        // store refusal/comment fields on model - mirror update() logic
        if ($to === 3) {
            // try to store the most specific field available
            if ($field === 'id_statut2' && Schema::hasColumn($table, 'commentaire_motif2')) {
                $docList->commentaire_motif2 = $request->input('commentaire_motif2') ?? $comment ?? null;
            } elseif ($field === 'id_statut1' && Schema::hasColumn($table, 'commentaire_motif1')) {
                $docList->commentaire_motif1 = $request->input('commentaire_motif1') ?? $comment ?? null;
            } else {
                // fallback generic column if exists
                if (Schema::hasColumn($table, 'commentaire_motif')) {
                    $docList->commentaire_motif = $request->input('commentaire_motif') ?? $comment ?? null;
                }
            }
        } else {
            // clear refusal-only columns related to the changed field to avoid stale text
            if ($field === 'id_statut2' && Schema::hasColumn($table, 'commentaire_motif2')) {
                $docList->commentaire_motif2 = ($to === 3) ? ($docList->commentaire_motif2 ?? $comment) : null;
            }
            if ($field === 'id_statut1' && Schema::hasColumn($table, 'commentaire_motif1')) {
                $docList->commentaire_motif1 = ($to === 3) ? ($docList->commentaire_motif1 ?? $comment) : null;
            }
        }

        // if there's a generic etat_formulaire column, update consistency as update() does
        if (Schema::hasColumn($table, 'etat_formulaire') && ($docList->{$field} ?? 2) != 2) {
            $docList->etat_formulaire = 'reponse';
        }

        $docList->save();

        // create history row for this change
        ImmobilierStatutHistory::create([
            'documents_financement_list_id' => $docList->id_doc_list,
            'projet_id'                     => $docList->id_projet ?: null,
            'from_statut_id'                => $old,
            'to_statut_id'                  => $to,
            'changed_by'                    => auth()->id() ?? null,
            'commentaire'                   => $comment,
        ]);

        // if we set id_statut2 as a side-effect because id_statut1 was accepted, create history for id_statut2 as well
        if ($field === 'id_statut1' && $hasStat2) {
            $currentStat2 = (int) ($docList->id_statut2 ?? 0);
            $originalStat2 = (int) ($docList->getOriginal('id_statut2') ?? 0);
            if ($originalStat2 !== $currentStat2) {
                ImmobilierStatutHistory::create([
                    'documents_financement_list_id' => $docList->id_doc_list,
                    'projet_id'                     => $docList->id_projet ?: null,
                    'from_statut_id'                => $originalStat2 > 0 ? $originalStat2 : null,
                    'to_statut_id'                  => $currentStat2 > 0 ? $currentStat2 : null,
                    'changed_by'                    => auth()->id() ?? null,
                    'commentaire'                   => $comment,
                ]);
            }
        }

        DB::commit();
    } catch (\Exception $e) {
        DB::rollBack();
        Log::error("toggleStatut error for DocumentsFinancementList #{$id_doc_list}: " . $e->getMessage(), [
            'trace' => $e->getTraceAsString(),
            'request' => $request->all(),
        ]);
        return back()->withErrors(['error' => 'خطأ أثناء تغيير الحالة: ' . $e->getMessage()]);
    }

    // Post-commit: prepare normalized type + params and send mail/notification
    try {
        // prefer the latest history comment (best source of truth)
        $latestHistoryComment = ImmobilierStatutHistory::where('documents_financement_list_id', $docList->id_doc_list)
            ->whereNotNull('commentaire')
            ->orderBy('created_at', 'desc')
            ->value('commentaire');

        $messageComment = $latestHistoryComment
                       ?? ($docList->commentaire_motif2 ?? $docList->commentaire_motif1 ?? $docList->commentaire_motif ?? null)
                       ?? $comment;

        // resolved status after update
        $resolved = (int) ($docList->{$field} ?? 0);

        $statusMap = [1 => 'accepted', 3 => 'rejected', 4 => 'incomplete', 5 => 'deferred'];
        $type = $statusMap[$resolved] ?? 'status_changed';

        // build message text (HTML-ready) - same wording as your update()
        $messageStatus = '';
        switch ($resolved) {
            case 1:
                $messageStatus = " تم <b>قبول</b> ملفكم بنجاح. نهنئكم على هذا القبول.";
                break;
            case 3:
                $messageStatus = " تم <b>رفض</b> ملفكم.";
                if (!empty($messageComment)) {
                    $messageStatus .= "<br>السبب: <b>" . e($messageComment) . "</b>";
                }
                break;
            case 4:
                $messageStatus = " طلبكم <b>غير مكتمل</b>.<br>يرجى استكمال النواقص التالية: <b>" . e($messageComment ?? '') . "</b>";
                break;
            case 5:
                $messageStatus = " تم <b>إرجاء</b> ملفكم.";
                if (!empty($messageComment)) {
                    $messageStatus .= "<br>الملاحظة: <b>" . e($messageComment) . "</b>";
                }
                break;
            default:
                $messageStatus = "تم تحديث حالة ملفكم.";
        }

        // prepare notification params
        $params = [];
        if ($resolved === 3) $params['reason'] = $messageComment ?? '';
        if ($resolved === 4) $params['missing'] = $messageComment ?? '';
        if ($resolved === 5) $params['note'] = $messageComment ?? '';
        if ($resolved === 1) $params['date'] = now()->format('Y-m-d');

        // reference / dossier number
        $params['reference'] = str_pad($docList->id_doc_list, 3, '0', STR_PAD_LEFT);
        if (!empty($messageComment)) $params['comment'] = $messageComment;

        // send mail if possible
        if ($user && !empty($user->email)) {
            try {
                Mail::to($user->email)->send(new \App\Mail\Statut2DocumentChanged($docList, $type, $params, $user));
            } catch (\Exception $ex) {
                Log::error("Mail send failed for docList {$docList->id_doc_list}: " . $ex->getMessage());
            }
        } else {
            Log::warning("toggleStatut: no recipient/email for DocumentsFinancementList ID {$docList->id_doc_list}");
        }

        // send DB notification
        if ($user) {
            try {
                $translationKey = "site.notifications.documents_formulaire.{$type}";
                $user->notify(new \App\Notifications\Statut2DocumentNotification(
                    $docList,
                    $type,
                    $params,
                    $docList->id_doc_list,
                    $translationKey
                ));
            } catch (\Exception $ex) {
                Log::error("Notification error for docList {$docList->id_doc_list}: " . $ex->getMessage());
            }
        }
    } catch (\Exception $ex) {
        Log::error("toggleStatut post-commit error for docList {$docList->id_doc_list}: " . $ex->getMessage());
    }

    return redirect()->route('admin.doc_fin2.index', $docList->id_doc_list)
        ->with('success', 'تم تحديث حالة الوثيقة بنجاح.');
}

  public function destroy($id_doc_list)
    {
        $docList = DocumentsFinancementList::with('documents')->findOrFail($id_doc_list);

        foreach ($docList->documents as $doc) {
            if ($doc->file && \Storage::disk('public')->exists($doc->file)) {
                \Storage::disk('public')->delete($doc->file);
            }
            $doc->delete();
        }

        $docList->delete();

        return redirect()
            ->route('admin.doc_fin2.index')
            ->with('success', 'تمت إزالة وثيقة التمويل بنجاح!');
    }
	
	public function pdf($id)
{
    // Augmenter limites si nécessaire
    ini_set('pcre.backtrack_limit', '10000000');
    ini_set('pcre.recursion_limit', '10000000');
    ini_set('memory_limit', '512M');

    // Récupérer le document avec ses relations
    $docList = DocumentsFinancementList::with([
        'projet.superviseur',
        'projet.activites',
        'projet.estimationBudget',
        'projet.estimationBudget.contributeurs',
        'projet.estimationBudget.postesDepense',
        'documents',
        'documents.typeDocument',
        'statut1',
        'statut2'
    ])->findOrFail($id);

    $formulaire   = TypeFormulaire::find(6);
    $formulaire2  = TypeFormulaire::find(7);
    $formulaire3  = TypeFormulaire::find(8);
    $type_financement = TypeFinancement::all();

    $demande_id = $docList->projet->id_demande;
    $demande    = DemandeFinancement::with('association')->find($demande_id);

    $selectedAssociationId   = $demande && $demande->association ? $demande->association->id_association : null;
    $selectedAssociationName = $demande && $demande->association ? $demande->association->nom_association : null;

    $id_projet  = $docList->projet->id_projet;
    $documents  = $docList->documents;
    $typedocum  = TpTypeDocument::all();
    $selectedTypes = DemandeFinancement::where('association_id', $demande->association_id)
                                       ->pluck('type_financement_id')
                                       ->toArray();

  $lang = 'ar';
$direction = 'rtl';

    // Générer le HTML à partir de la vue Blade
    $html = view('backoffice.financement.documents_financement.pdf2', compact(
        'docList','formulaire','formulaire2','formulaire3','type_financement',
        'demande_id','demande','selectedAssociationId','selectedAssociationName',
        'documents','id_projet','typedocum','selectedTypes',
        'lang','direction'
    ))->render();

    // Générer le PDF avec mPDF
    $mpdf = new \Mpdf\Mpdf([
        'mode'             => 'utf-8',
        'format'           => 'A4',
        'default_font'     => 'dejavusans',
        'directionality'   => $direction,
        'autoScriptToLang' => true,
        'autoLangToFont'   => true,
        'tempDir'          => storage_path('app/mpdf'),
    ]);

    $mpdf->setFooter('{PAGENO} / {nbpg}');
    $mpdf->WriteHTML($html);

    // Retourner la réponse PDF au navigateur
    return response(
        $mpdf->Output('document_financement_'.$id.'.pdf', 'S'),
        200,
        [
            'Content-Type'        => 'application/pdf',
            'Content-Disposition' => 'inline; filename="document_financement_'.$id.'.pdf"',
        ]
    );
}

}
