<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\DescGouvr;
use App\Models\Gouvernorat;
use Illuminate\Support\Facades\Auth;

class DescGouvrController extends Controller
{
  public function index(Request $request) 
{
    $search = trim((string) $request->input('q', ''));

    // Bloquer les recherches vides ou contenant uniquement des %, _, ou espaces
    if ($search === '' || preg_match('/^[\s%_]+$/', $search)) {
        $search = null;
    }

    $query = DescGouvr::with(['gouvernorat', 'utilisateur'])
                      ->orderBy('id_desc', 'desc');

    if ($search) {
        $words = preg_split('/\s+/', $search);

        $query->where(function ($q) use ($words) {
            foreach ($words as $word) {
                // Échapper % et _ pour LIKE
                $word = str_replace(['\\', '%', '_'], ['\\\\', '\\%', '\\_'], $word);

                $q->whereHas('gouvernorat', function($q2) use ($word) {
                    $q2->where('nom_ar', 'like', "%{$word}%")
                       ->orWhere('nom_fr', 'like', "%{$word}%")
                       ->orWhere('nom_en', 'like', "%{$word}%");
                });
            }
        });
    }

    $desc_gouvr = $query->paginate(10)->withQueryString();

    // closure to compute usable image source
    $computeImageSrc = function ($stored) {
        if (empty($stored)) return null;

        // if it's already a valid URL
        if (filter_var($stored, FILTER_VALIDATE_URL)) {
            return $stored;
        }

        // if DB contains binary blob (older records)
        if (!is_string($stored)) {
            try {
                $b64 = base64_encode($stored);
                return "data:image/jpeg;base64,{$b64}";
            } catch (\Throwable $e) {
                // fallthrough to path logic
            }
        }

        // candidate filesystem locations (ordered)
        $candidates = [];
        $candidates[] = $stored; // as-stored (may be absolute path)
        $root = '/www/mjs/shared_uploads/images';
        $candidates[] = rtrim($root, '/') . '/' . basename($stored);
        $candidates[] = '/home/preprov/www/mjs/shared_uploads/images/' . basename($stored);
        $candidates[] = public_path('shared_uploads/images/' . basename($stored));
        $candidates[] = storage_path('app/' . ltrim($stored, '/'));

        // normalize & dedupe
        $seen = [];
        $unique = [];
        foreach ($candidates as $p) {
            if (!$p) continue;
            $p = preg_replace('#^file://#', '', (string)$p);
            if (!isset($seen[$p])) {
                $seen[$p] = true;
                $unique[] = $p;
            }
        }

        // try to read a file and return data URI
        foreach ($unique as $path) {
            if (@file_exists($path) && @is_file($path) && @is_readable($path)) {
                $contents = @file_get_contents($path);
                if ($contents !== false) {
                    $mime = @mime_content_type($path) ?: 'image/jpeg';
                    return "data:{$mime};base64," . base64_encode($contents);
                }
            }
        }

        // If the stored string contains /shared_uploads/... return a URL
        if (preg_match('#(/shared_uploads/.*)$#', $stored, $m)) {
            return url(ltrim($m[1], '/'));
        }

        // fallback: assume filename under shared_uploads/images and expose as URL
        return url('shared_uploads/images/' . basename($stored));
    };

    // attach image_gouvSrc to each item for use in the view
    foreach ($desc_gouvr as $item) {
        $item->image_gouvSrc = $computeImageSrc($item->image_gouv);
    }

    return view('backoffice.desc_gouvr.index', compact('desc_gouvr', 'search'));
}


    public function create()
    {
        $gouvernorats = Gouvernorat::all();

        return view('backoffice.desc_gouvr.create', compact('gouvernorats'));
    }
	
	
	public function store(Request $request)
{
    $imageRules = 'required|image|mimes:jpg,png,jpeg,webp|max:4096';

    $request->validate([

        'id_gouv'   => 'required|exists:tp_gouvernorat,id_gouver',
        'image_gouv' => $imageRules,
        'image1' => $imageRules,
        'image2' => $imageRules,
        'image3' => $imageRules,
        'desc1'      => 'required|string',
        'desc1_fr'   => 'required|string',
        'desc1_en'   => 'required|string',

        'desc2'      => 'required|string',
        'desc2_fr'   => 'required|string',
        'desc2_en'   => 'required|string',

        'desc3'      => 'required|string',
        'desc3_fr'   => 'required|string',
        'desc3_en'   => 'required|string',

    ], [

        'id_gouv.required'     => 'الرجاء اختيار الولاية',
        'image_gouv.required'  => 'الرجاء إضافة صورة الولاية',

        'desc1.required'       => 'الرجاء إدخال الوصف الأول بالعربية',
        'desc1_fr.required'    => 'الرجاء إدخال الوصف الأول بالفرنسية',
        'desc1_en.required'    => 'الرجاء إدخال الوصف الأول بالإنجليزية',
        'image1.required'      => 'الرجاء إضافة الصورة الأولى',

        'desc2.required'       => 'الرجاء إدخال الوصف الثاني بالعربية',
        'desc2_fr.required'    => 'الرجاء إدخال الوصف الثاني بالفرنسية',
        'desc2_en.required'    => 'الرجاء إدخال الوصف الثاني بالالإنجليزية',
        'image2.required'      => 'الرجاء إضافة الصورة الثانية',

        'desc3.required'       => 'الرجاء إدخال الوصف الثالث بالعربية',
        'desc3_fr.required'    => 'الرجاء إدخال الوصف الثالث بالفرنسية',
        'desc3_en.required'    => 'الرجاء إدخال الوصف الثالث بالالإنجليزية',
        'image3.required'      => 'الرجاء إضافة الصورة الثالثة',

        'id_gouv.exists'       => 'الولاية المختارة غير موجودة',

        'image_gouv.image' => 'يجب أن يكون الملف صورة',
        'image_gouv.mimes' => 'يجب أن تكون الصورة بصيغة jpg أو png أو jpeg أو webp',
        'image_gouv.max'   => 'حجم الصورة يجب ألا يتجاوز 4 ميغابايت',

        'image1.image' => 'يجب أن يكون الملف صورة',
        'image1.mimes' => 'يجب أن تكون الصورة بصيغة jpg أو png أو jpeg أو webp',
        'image1.max'   => 'حجم الصورة يجب ألا يتجاوز 4 ميغابايت',

        'image2.image' => 'يجب أن يكون الملف صورة',
        'image2.mimes' => 'يجب أن تكون الصورة بصيغة jpg أو png أو jpeg أو webp',
        'image2.max'   => 'حجم الصورة يجب ألا يتجاوز 4 ميغابايت',

        'image3.image' => 'يجب أن يكون الملف صورة',
        'image3.mimes' => 'يجب أن تكون الصورة بصيغة jpg أو png أو jpeg أو webp',
        'image3.max'   => 'حجم الصورة يجب ألا يتجاوز 4 ميغابايت',

    ]);

    $desc = new DescGouvr();
    $desc->id_gouv = $request->id_gouv;
    $desc->id_user = Auth::id();

    $desc->desc1 = $request->desc1;
    $desc->desc1_fr = $request->desc1_fr;
    $desc->desc1_en = $request->desc1_en;

    $desc->desc2 = $request->desc2;
    $desc->desc2_fr = $request->desc2_fr;
    $desc->desc2_en = $request->desc2_en;

    $desc->desc3 = $request->desc3;
    $desc->desc3_fr = $request->desc3_fr;
    $desc->desc3_en = $request->desc3_en;

    // Candidates for the images directory (ordered). We try to find a usable one.
    $candidates = [
        '/www/mjs/shared_uploads/images',
        '/home/preprov/www/mjs/shared_uploads/images',
        base_path('shared_uploads/images'),
        base_path('../shared_uploads/images'),
        public_path('../../shared_uploads/images'),
    ];

    // Resolve a usable directory (do this once)
    $usable = null;
    $tried = [];
    foreach ($candidates as $cand) {
        $rp = @realpath($cand);
        $tried[$cand] = [
            'realpath' => $rp ?: null,
            'exists' => @file_exists($cand) ? 'yes' : 'no',
            'is_dir' => @is_dir($cand) ? 'yes' : 'no',
            'is_writable' => @is_writable($cand) ? 'yes' : 'no'
        ];
        if ($rp && is_dir($rp) && is_writable($rp)) {
            $usable = $rp;
            break;
        }
    }

    if (!$usable) {
        // Log and return helpful error if no writable folder found
        \Log::warning('No usable upload folder found for DescGouvr image upload.', [
            'cwd' => @getcwd(),
            'open_basedir' => ini_get('open_basedir') ?: '(not set)',
            'tried' => $tried,
            'get_current_user' => @get_current_user(),
        ]);

        $msg = "Upload folder not accessible by PHP. Checked these candidates: ";
        foreach ($tried as $c => $info) {
            $msg .= "[{$c} -> realpath=" . ($info['realpath'] ?? 'null') . ", exists={$info['exists']}, is_dir={$info['is_dir']}, writable={$info['is_writable']}] ";
        }
        return back()->withInput()->withErrors(['image_gouv' => $msg]);
    }

    // Helper to move and return stored absolute path
    $moveAndStore = function ($file, $usableDir) {
        // sanitize original filename
        $originalName = $file->getClientOriginalName();
        $originalName = basename($originalName);
        $originalName = preg_replace('/[\0\/\\\\<>:"|?*\x00-\x1F]+/u', '-', $originalName);
        if ($originalName === '' || $originalName === '.' || $originalName === '..') {
            throw new \RuntimeException('اسم الملف الأصلي غير صالح.');
        }

        $targetPath = rtrim($usableDir, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . $originalName;

        // avoid overwrite by appending timestamp if exists
        if (file_exists($targetPath)) {
            $dotPos = strrpos($originalName, '.');
            if ($dotPos !== false) {
                $base = substr($originalName, 0, $dotPos);
                $ext  = substr($originalName, $dotPos);
            } else {
                $base = $originalName;
                $ext = '';
            }
            $originalName = $base . '-' . time() . $ext;
            $targetPath = rtrim($usableDir, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . $originalName;
        }

        // move uploaded file
        $file->move($usableDir, $originalName);

        // Return absolute path stored in DB
        return $targetPath;
    };

    // Process images and store paths in the model
    foreach (['image_gouv', 'image1', 'image2', 'image3'] as $imgField) {
        if ($request->hasFile($imgField)) {
            try {
                $file = $request->file($imgField);
                $storedPath = $moveAndStore($file, $usable);
                // store absolute path in DB column
                $desc->$imgField = $storedPath;
            } catch (\Throwable $e) {
                \Log::error('DescGouvr image upload failed', ['field' => $imgField, 'error' => $e->getMessage()]);
                return back()->withInput()->withErrors([$imgField => 'فشل حفظ الصورة على الخادم. (راجع سجلات الخادم).']);
            }
        } else {
            // Shouldn't happen because validation requires them, but keep consistent behavior
            $desc->$imgField = null;
        }
    }

    $desc->save();

    return redirect()
        ->route('desc_gouvr.index')
        ->with('success', 'تمت إضافة وصف الولاية بنجاح');
}

	
    
 /*  public function store(Request $request)
    {
		    $imageRules = 'required|image|mimes:jpg,png,jpeg,webp|max:4096';

        $request->validate([

			'id_gouv'   => 'required|exists:tp_gouvernorat,id_gouver',
			'image_gouv' => $imageRules,
			'image1' => $imageRules,
			'image2' => $imageRules,
			'image3' => $imageRules,
			'desc1'      => 'required|string',
			'desc1_fr'   => 'required|string',
			'desc1_en'   => 'required|string',

			'desc2'      => 'required|string',
			'desc2_fr'   => 'required|string',
			'desc2_en'   => 'required|string',

			'desc3'      => 'required|string',
			'desc3_fr'   => 'required|string',
			'desc3_en'   => 'required|string',

		], [

			'id_gouv.required'     => 'الرجاء اختيار الولاية',
			'image_gouv.required'  => 'الرجاء إضافة صورة الولاية',

			'desc1.required'       => 'الرجاء إدخال الوصف الأول بالعربية',
			'desc1_fr.required'    => 'الرجاء إدخال الوصف الأول بالفرنسية',
			'desc1_en.required'    => 'الرجاء إدخال الوصف الأول بالإنجليزية',
			'image1.required'      => 'الرجاء إضافة الصورة الأولى',

			'desc2.required'       => 'الرجاء إدخال الوصف الثاني بالعربية',
			'desc2_fr.required'    => 'الرجاء إدخال الوصف الثاني بالفرنسية',
			'desc2_en.required'    => 'الرجاء إدخال الوصف الثاني بالإنجليزية',
			'image2.required'      => 'الرجاء إضافة الصورة الثانية',

			'desc3.required'       => 'الرجاء إدخال الوصف الثالث بالعربية',
			'desc3_fr.required'    => 'الرجاء إدخال الوصف الثالث بالفرنسية',
			'desc3_en.required'    => 'الرجاء إدخال الوصف الثالث بالإنجليزية',
			'image3.required'      => 'الرجاء إضافة الصورة الثالثة',

			'id_gouv.exists'       => 'الولاية المختارة غير موجودة',

			'image_gouv.image' => 'يجب أن يكون الملف صورة',
			'image_gouv.mimes' => 'يجب أن تكون الصورة بصيغة jpg أو png أو jpeg أو webp',
			'image_gouv.max'   => 'حجم الصورة يجب ألا يتجاوز 4 ميغابايت',

			'image1.image' => 'يجب أن يكون الملف صورة',
			'image1.mimes' => 'يجب أن تكون الصورة بصيغة jpg أو png أو jpeg أو webp',
			'image1.max'   => 'حجم الصورة يجب ألا يتجاوز 4 ميغابايت',

			'image2.image' => 'يجب أن يكون الملف صورة',
			'image2.mimes' => 'يجب أن تكون الصورة بصيغة jpg أو png أو jpeg أو webp',
			'image2.max'   => 'حجم الصورة يجب ألا يتجاوز 4 ميغابايت',

			'image3.image' => 'يجب أن يكون الملف صورة',
			'image3.mimes' => 'يجب أن تكون الصورة بصيغة jpg أو png أو jpeg أو webp',
			'image3.max'   => 'حجم الصورة يجب ألا يتجاوز 4 ميغابايت',

		]);
        $desc = new DescGouvr();
        $desc->id_gouv = $request->id_gouv;
        $desc->id_user = Auth::id(); 

        $desc->desc1 = $request->desc1;
        $desc->desc1_fr = $request->desc1_fr;
        $desc->desc1_en = $request->desc1_en;
        
        $desc->desc2 = $request->desc2;
        $desc->desc2_fr = $request->desc2_fr;
        $desc->desc2_en = $request->desc2_en;

        $desc->desc3 = $request->desc3;
        $desc->desc3_fr = $request->desc3_fr;
        $desc->desc3_en = $request->desc3_en;

        

		foreach(['image_gouv','image1','image2','image3'] as $img){
				if($request->hasFile($img)){
					$desc->$img = file_get_contents($request->file($img));
				}
			}
	
        $desc->save();

        return redirect()
            ->route('desc_gouvr.index')
            ->with('success', 'تمت إضافة وصف الولاية بنجاح');
    }

*/    

public function edit(DescGouvr $desc_gouvr)
{
    $gouvernorats = Gouvernorat::all();

    // closure to compute a usable src (data: URI or URL) from stored value
    $computeImageSrc = function ($stored) {
        if (empty($stored)) return null;

        // already a full URL
        if (filter_var($stored, FILTER_VALIDATE_URL)) {
            return $stored;
        }

        // binary blob in DB (older records)
        if (!is_string($stored)) {
            try {
                $b64 = base64_encode($stored);
                return "data:image/jpeg;base64,{$b64}";
            } catch (\Throwable $e) {
                // fall through to path logic
            }
        }

        // candidate filesystem locations (ordered)
        $candidates = [];
        $candidates[] = $stored; // as-stored (maybe absolute path)
        $root = '/www/mjs/shared_uploads/images';
        $candidates[] = rtrim($root, '/') . '/' . basename($stored);
        $candidates[] = '/home/preprov/www/mjs/shared_uploads/images/' . basename($stored);
        $candidates[] = public_path('shared_uploads/images/' . basename($stored));
        $candidates[] = storage_path('app/' . ltrim($stored, '/'));

        // normalize & dedupe
        $seen = [];
        $unique = [];
        foreach ($candidates as $p) {
            if (!$p) continue;
            $p = preg_replace('#^file://#', '', (string)$p);
            if (!isset($seen[$p])) {
                $seen[$p] = true;
                $unique[] = $p;
            }
        }

        // try to read a file and return data URI
        foreach ($unique as $path) {
            if (@file_exists($path) && @is_file($path) && @is_readable($path)) {
                $contents = @file_get_contents($path);
                if ($contents !== false) {
                    $mime = @mime_content_type($path) ?: 'image/jpeg';
                    return "data:{$mime};base64," . base64_encode($contents);
                }
            }
        }

        // if stored contains /shared_uploads/... expose as URL relative to app URL
        if (preg_match('#(/shared_uploads/.*)$#', $stored, $m)) {
            return url(ltrim($m[1], '/'));
        }

        // fallback: assume filename under shared_uploads/images
        return url('shared_uploads/images/' . basename($stored));
    };

    // attach computed sources to the model (transient attributes for the view)
    $desc_gouvr->image_gouvSrc = $computeImageSrc($desc_gouvr->image_gouv);
    $desc_gouvr->image1Src     = $computeImageSrc($desc_gouvr->image1);
    $desc_gouvr->image2Src     = $computeImageSrc($desc_gouvr->image2);
    $desc_gouvr->image3Src     = $computeImageSrc($desc_gouvr->image3);

    return view('backoffice.desc_gouvr.edit', compact('desc_gouvr', 'gouvernorats'));
}

	
	
	public function update(Request $request, DescGouvr $desc_gouvr)
{
    $imageRules = 'image|mimes:jpg,png,jpeg,webp|max:4096';
    $request->validate([

        'id_gouv'   => 'required|exists:tp_gouvernorat,id_gouver',

        // images non obligatoires en update
        'image_gouv' => "sometimes|$imageRules",
        'image1'     => "sometimes|$imageRules",
        'image2'     => "sometimes|$imageRules",
        'image3'     => "sometimes|$imageRules",

        'desc1'      => 'required|string',
        'desc1_fr'   => 'required|string',
        'desc1_en'   => 'required|string',

        'desc2'      => 'required|string',
        'desc2_fr'   => 'required|string',
        'desc2_en'   => 'required|string',

        'desc3'      => 'required|string',
        'desc3_fr'   => 'required|string',
        'desc3_en'   => 'required|string',

    ], [

        'id_gouv.required' => 'الرجاء اختيار الولاية',
        'id_gouv.exists'   => 'الولاية المختارة غير موجودة',

        'desc1.required'    => 'الرجاء إدخال الوصف الأول بالعربية',
        'desc1_fr.required' => 'الرجاء إدخال الوصف الأول بالفرنسية',
        'desc1_en.required' => 'الرجاء إدخال الوصف الأول بالإنجليزية',

        'desc2.required'    => 'الرجاء إدخال الوصف الثاني بالعربية',
        'desc2_fr.required' => 'الرجاء إدخال الوصف الثاني بالفرنسية',
        'desc2_en.required' => 'الرجاء إدخال الوصف الثاني بالإنجليزية',

        'desc3.required'    => 'الرجاء إدخال الوصف الثالث بالعربية',
        'desc3_fr.required' => 'الرجاء إدخال الوصف الثالث بالفرنسية',
        'desc3_en.required' => 'الرجاء إدخال الوصف الثالث بالالإنجليزية',

        'image_gouv.image' => 'يجب أن يكون الملف صورة',
        'image_gouv.mimes' => 'يجب أن تكون الصورة بصيغة jpg أو png أو jpeg أو webp',
        'image_gouv.max'   => 'حجم الصورة يجب ألا يتجاوز 4 ميغابايت',

        'image1.image' => 'يجب أن يكون الملف صورة',
        'image1.mimes' => 'يجب أن تكون الصورة بصيغة jpg أو png أو jpeg أو webp',
        'image1.max'   => 'حجم الصورة يجب ألا يتجاوز 4 ميغابايت',

        'image2.image' => 'يجب أن يكون الملف صورة',
        'image2.mimes' => 'يجب أن تكون الصورة بصيغة jpg أو png أو jpeg أو webp',
        'image2.max'   => 'حجم الصورة يجب ألا يتجاوز 4 ميغابايت',

        'image3.image' => 'يجب أن يكون الملف صورة',
        'image3.mimes' => 'يجب أن تكون الصورة بصيغة jpg أو png أو jpeg أو webp',
        'image3.max'   => 'حجم الصورة يجب ألا يتجاوز 4 ميغابايت',
    ]);

    // Update text fields
    $desc_gouvr->id_gouv   = $request->id_gouv;

    $desc_gouvr->desc1     = $request->desc1;
    $desc_gouvr->desc1_fr  = $request->desc1_fr;
    $desc_gouvr->desc1_en  = $request->desc1_en;

    $desc_gouvr->desc2     = $request->desc2;
    $desc_gouvr->desc2_fr  = $request->desc2_fr;
    $desc_gouvr->desc2_en  = $request->desc2_en;

    $desc_gouvr->desc3     = $request->desc3;
    $desc_gouvr->desc3_fr  = $request->desc3_fr;
    $desc_gouvr->desc3_en  = $request->desc3_en;

    // If no images uploaded, just save text changes
    $anyFile = $request->hasFile('image_gouv') || $request->hasFile('image1') || $request->hasFile('image2') || $request->hasFile('image3');
    $usable = null;
    $tried = [];

    // If at least one image is being uploaded, resolve usable upload dir
    if ($anyFile) {
        $candidates = [
            '/www/mjs/shared_uploads/images',
            '/home/preprov/www/mjs/shared_uploads/images',
            base_path('shared_uploads/images'),
            base_path('../shared_uploads/images'),
            public_path('../../shared_uploads/images'),
        ];

        foreach ($candidates as $cand) {
            $rp = @realpath($cand);
            $tried[$cand] = [
                'realpath' => $rp ?: null,
                'exists' => @file_exists($cand) ? 'yes' : 'no',
                'is_dir' => @is_dir($cand) ? 'yes' : 'no',
                'is_writable' => @is_writable($cand) ? 'yes' : 'no'
            ];
            if ($rp && is_dir($rp) && is_writable($rp)) {
                $usable = $rp;
                break;
            }
        }

        if (!$usable) {
            \Log::warning('No usable upload folder found for DescGouvr image update.', [
                'cwd' => @getcwd(),
                'open_basedir' => ini_get('open_basedir') ?: '(not set)',
                'tried' => $tried,
                'get_current_user' => @get_current_user(),
            ]);

            $msg = "Upload folder not accessible by PHP. Checked these candidates: ";
            foreach ($tried as $c => $info) {
                $msg .= "[{$c} -> realpath=" . ($info['realpath'] ?? 'null') . ", exists={$info['exists']}, is_dir={$info['is_dir']}, writable={$info['is_writable']}] ";
            }
            return back()->withInput()->withErrors(['image_gouv' => $msg]);
        }
    }

    // Helper to move uploaded file and return stored absolute path
    $moveAndStore = function ($file, $usableDir) {
        $originalName = $file->getClientOriginalName();
        $originalName = basename($originalName);
        $originalName = preg_replace('/[\0\/\\\\<>:"|?*\x00-\x1F]+/u', '-', $originalName);
        if ($originalName === '' || $originalName === '.' || $originalName === '..') {
            throw new \RuntimeException('اسم الملف الأصلي غير صالح.');
        }

        $targetPath = rtrim($usableDir, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . $originalName;

        // avoid overwrite by appending timestamp if exists
        if (file_exists($targetPath)) {
            $dotPos = strrpos($originalName, '.');
            if ($dotPos !== false) {
                $base = substr($originalName, 0, $dotPos);
                $ext  = substr($originalName, $dotPos);
            } else {
                $base = $originalName;
                $ext = '';
            }
            $originalName = $base . '-' . time() . $ext;
            $targetPath = rtrim($usableDir, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . $originalName;
        }

        $file->move($usableDir, $originalName);

        return $targetPath;
    };

    // Process each image field: if a new file uploaded, delete old file (if it's a path) and save new path
    foreach (['image_gouv', 'image1', 'image2', 'image3'] as $imgField) {
        if ($request->hasFile($imgField)) {
            try {
                // attempt to delete previous file if it looks like a filesystem path
                $old = $desc_gouvr->$imgField;
                if (is_string($old) && $old !== '') {
                    // normalize and try unlink; silence errors
                    $oldPath = preg_replace('#^file://#', '', $old);
                    if (@file_exists($oldPath) && @is_file($oldPath)) {
                        @unlink($oldPath);
                    }
                }

                // move new file into usable dir and store absolute path
                $file = $request->file($imgField);
                $storedPath = $moveAndStore($file, $usable);
                $desc_gouvr->$imgField = $storedPath;
            } catch (\Throwable $e) {
                \Log::error('DescGouvr image update failed', ['field' => $imgField, 'error' => $e->getMessage()]);
                return back()->withInput()->withErrors([$imgField => 'فشل حفظ الصورة على الخادم. (راجع سجلات الخادم).']);
            }
        }
        // if no new file, keep existing value untouched
    }

    $desc_gouvr->save();

    return redirect()
        ->route('desc_gouvr.index')
        ->with('success', 'تم تحديث وصف الولاية بنجاح');
}

	

 /*   public function update(Request $request, DescGouvr $desc_gouvr)
{
    $imageRules = 'image|mimes:jpg,png,jpeg,webp|max:4096';
    $request->validate([

        'id_gouv'   => 'required|exists:tp_gouvernorat,id_gouver',

        // images non obligatoires en update
        'image_gouv' => "sometimes|$imageRules",

        'desc1'      => 'required|string',
        'desc1_fr'   => 'required|string',
        'desc1_en'   => 'required|string',
        'image1' => "sometimes|$imageRules",
		
        'desc2'      => 'required|string',
        'desc2_fr'   => 'required|string',
        'desc2_en'   => 'required|string',
        'image2'     => "sometimes|$imageRules",

        'desc3'      => 'required|string',
        'desc3_fr'   => 'required|string',
        'desc3_en'   => 'required|string',
        'image3' => "sometimes|$imageRules",

    ], [

        'id_gouv.required' => 'الرجاء اختيار الولاية',
        'id_gouv.exists'   => 'الولاية المختارة غير موجودة',

        'desc1.required'    => 'الرجاء إدخال الوصف الأول بالعربية',
        'desc1_fr.required' => 'الرجاء إدخال الوصف الأول بالفرنسية',
        'desc1_en.required' => 'الرجاء إدخال الوصف الأول بالإنجليزية',

        'desc2.required'    => 'الرجاء إدخال الوصف الثاني بالعربية',
        'desc2_fr.required' => 'الرجاء إدخال الوصف الثاني بالفرنسية',
        'desc2_en.required' => 'الرجاء إدخال الوصف الثاني بالإنجليزية',

        'desc3.required'    => 'الرجاء إدخال الوصف الثالث بالعربية',
        'desc3_fr.required' => 'الرجاء إدخال الوصف الثالث بالفرنسية',
        'desc3_en.required' => 'الرجاء إدخال الوصف الثالث بالإنجليزية',

        // Messages d’erreur pour les images
        'image_gouv.image' => 'يجب أن يكون الملف صورة',
        'image_gouv.mimes' => 'يجب أن تكون الصورة بصيغة jpg أو png أو jpeg أو webp',
        'image_gouv.max'   => 'حجم الصورة يجب ألا يتجاوز 4 ميغابايت',

        'image1.image' => 'يجب أن يكون الملف صورة',
        'image1.mimes' => 'يجب أن تكون الصورة بصيغة jpg أو png أو jpeg أو webp',
        'image1.max'   => 'حجم الصورة يجب ألا يتجاوز 4 ميغابايت',

        'image2.image' => 'يجب أن يكون الملف صورة',
        'image2.mimes' => 'يجب أن تكون الصورة بصيغة jpg أو png أو jpeg أو webp',
        'image2.max'   => 'حجم الصورة يجب ألا يتجاوز 4 ميغابايت',

        'image3.image' => 'يجب أن يكون الملف صورة',
        'image3.mimes' => 'يجب أن تكون الصورة بصيغة jpg أو png أو jpeg أو webp',
        'image3.max'   => 'حجم الصورة يجب ألا يتجاوز 4 ميغابايت',
    ]);

    // Champs texte
    $desc_gouvr->id_gouv   = $request->id_gouv;

    $desc_gouvr->desc1     = $request->desc1;
    $desc_gouvr->desc1_fr  = $request->desc1_fr;
    $desc_gouvr->desc1_en  = $request->desc1_en;

    $desc_gouvr->desc2     = $request->desc2;
    $desc_gouvr->desc2_fr  = $request->desc2_fr;
    $desc_gouvr->desc2_en  = $request->desc2_en;

    $desc_gouvr->desc3     = $request->desc3;
    $desc_gouvr->desc3_fr  = $request->desc3_fr;
    $desc_gouvr->desc3_en  = $request->desc3_en;

    // Gestion des images
    foreach(['image_gouv','image1','image2','image3'] as $img){
        if($request->hasFile($img)){
            $desc_gouvr->$img = file_get_contents($request->file($img));
        }
    }

    $desc_gouvr->save();

    return redirect()
        ->route('desc_gouvr.index')
        ->with('success', 'تم تحديث وصف الولاية بنجاح');
}

*/
    public function destroy(DescGouvr $desc_gouvr)
    {
        foreach(['image_gouv','image1','image2','image3'] as $img){
            if($desc_gouvr->$img){
                \Storage::disk('public')->delete($desc_gouvr->$img);
            }
        }

        $desc_gouvr->delete();

        return redirect()->route('desc_gouvr.index')
                        ->with('success','تم حذف الوصف بنجاح');
    }
}
