<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Contact;

class ContactController extends Controller
{
    /**
     * Store contact request from public form
     */
    public function store(Request $request)
    {
        // validate (use localized messages via __('site...') if you have translation keys)
        $validated = $request->validate([
            'name'    => 'required|string|max:191',
            'email'   => 'required|email|max:191',
            'phone'   => 'nullable|string|max:50',
            'subject' => 'nullable|string|max:100',
            'message' => 'required|string',
        ], [
            'name.required'    => __('site.validation.name_required'),
            'email.required'   => __('site.validation.email_required'),
            'email.email'      => __('site.validation.email_invalid'),
            'message.required' => __('site.validation.message_required'),
        ]);

        // Save contact
        Contact::create([
            'name'    => $validated['name'],
            'email'   => $validated['email'],
            'phone'   => $validated['phone'] ?? null,
            'subject' => $validated['subject'] ?? null,
            'message' => $validated['message'],
        ]);

        // redirect back with localized success message key (ensure you have this key in lang files)
        return back()->with('contact_success', __('site.contact_sent'));
    }

public function index(Request $request)
{
    // Récupérer la recherche et nettoyer les espaces
    $q = trim((string)$request->input('q', ''));

    // Bloquer les recherches vides ou contenant uniquement % ou espaces
    if ($q === '' || preg_match('/^[\s%]+$/', $q)) {
        $q = null;
    }

    $query = Contact::query();

    if ($q) {
        $query->where(function ($qr) use ($q) {
            $qr->where('name', 'like', "%{$q}%")
               ->orWhere('email', 'like', "%{$q}%")
               ->orWhere('phone', 'like', "%{$q}%")
               ->orWhere('subject', 'like', "%{$q}%")
               ->orWhere('message', 'like', "%{$q}%");
        });
    }

    $contacts = $query->orderBy('created_at', 'desc')
                      ->paginate(20)
                      ->withQueryString();

    return view('backoffice.newsletter_contact.contact_index', compact('contacts', 'q'));
}


    // show single contact (mark as read)
    public function show(Contact $contact)
    {
        // mark read if not already
        if (! $contact->is_read) {
            $contact->is_read = true;
            $contact->save();
        }

        return view('backoffice.newsletter_contact.contact_show', compact('contact'));
    }

    // delete contact
    public function destroy(Contact $contact)
    {
        $contact->delete();
        return redirect()->route('contacts.index')->with('ok', 'تم حذف الرسالة بنجاح.');
    }


}
